package com.geoway.landteam.landcloud.service.customtask.task;

import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.gw.base.log.GiLoger;
import com.gw.base.log.GwLoger;
import org.apache.commons.collections.CollectionUtils;
import org.locationtech.jts.geom.*;
import org.locationtech.jts.io.WKTReader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 图斑分割服务层对象
 */
@Service
public class TaskGeometrySplitService {

    /**
     * 日志对象
     */
    private final GiLoger logger = GwLoger.getLoger(TaskGeometrySplitService.class);

    /**
     * 数据库模板对象
     */
    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Value("${project.union.spatialAnalysis:false}")
    private Boolean spatialAnalysis;

    @Value("${project.union.spatialAnalysisUrl:}")
    private  String spatialAnalysisUrl;

    /**
     * 图斑切割
     *
     * @param polygon   被切割的图斑
     * @param splitLine 切割图斑的线
     * @return 切割后的地块WKT集合
     */
    public List<String> split(Geometry polygon, Geometry splitLine) throws Exception{
        if(polygon!=null && splitLine != null && !polygon.intersects(splitLine)){
            throw new RuntimeException("分割失败");
        }
        WKTReader wktReader = new WKTReader();
        Coordinate[] coordinates = splitLine.getCoordinates();
        Point start = (Point) wktReader.read("POINT("+ coordinates[0].x +" "+ coordinates[0].y +")");
        Point end = (Point) wktReader.read("POINT("+ coordinates[coordinates.length-1].x +" "+ coordinates[coordinates.length-1].y +")");
        if(splitLine.getGeometryType().equals("LineString") &&(start.within(polygon) || end.within(polygon))){
            throw new RuntimeException("分割失败");
        }

        String polygonWkt = polygon.toString();
        String splitLineWkt = splitLine.toString();
        if (this.spatialAnalysis) { // 通过分析服务进行切割
            return this.splitByServer(polygonWkt, splitLineWkt);
        }
        // 通过自身服务进行切割
        String sql = "";
        if (splitLine instanceof LineString) { // 根据线进行切割
            sql += "SELECT ST_ASTEXT(";
            sql += "(ST_DUMP(";
            sql += "ST_SPLIT(";
            sql += "ST_GEOMFROMTEXT('" + polygonWkt + "',0),";
            sql += "ST_GEOMFROMTEXT('" + splitLineWkt + "',0)";
            sql += ")";
            sql += ")).geom) AS \"result\"";
        } else { // 根据面进行切割
            sql += "SELECT * from (SELECT (";
            sql += "(ST_DUMP(";
            sql += "ST_DIFFERENCE(";
            sql += "ST_GEOMFROMTEXT('" + polygonWkt + "',0),";
            sql += "ST_GEOMFROMTEXT('" + splitLineWkt + "',0)";
            sql += ")";
            sql += ")).geom) AS \"result\"";
            sql += " UNION ALL ";
            sql += "SELECT (";
            sql += "ST_INTERSECTION(";
            sql += "ST_GEOMFROMTEXT('" + polygonWkt + "',0),";
            sql += "ST_GEOMFROMTEXT('" + splitLineWkt + "',0)";
            sql += ")) AS \"result\" ) g where st_area(g.result)*power(10,15)>1 ";
        }
        logger.info(sql);
        List<Map<String, Object>> list = this.jdbcTemplate.queryForList(sql);
        List<String> wkts = list.stream().map(m -> m.get("result").toString()).collect(Collectors.toList());
        //循环判断如果多部件则改为multipolygon
        List<String> result  = new ArrayList<>();
        Geometry geometry = null;
        GeometryFactory geometryFactory = new GeometryFactory();
        for (String wkt : wkts) {
            Polygon polygon1 = null;
            geometry = new WKTReader().read(wkt);
            polygon1 = (Polygon) geometry;
            if(polygon1.getNumInteriorRing()>0){
                result.add(wkt.replace("POLYGON","MULTIPOLYGON(")+")");
            }else{
                result.add(wkt);
            }
        }
        if(CollectionUtils.isNotEmpty(wkts) && wkts.size() <= 1){
            throw new RuntimeException("分割失败");
        }
        return result;
    }


    /**
     * 通过分析服务分割图斑
     */
    private List<String> splitByServer(String polygonWkt, String splitLineWkt) {
        String url = this.spatialAnalysisUrl + "/spatial/split";
        String response = HttpUtil.createPost(url).form("polygon", polygonWkt).form("splitLine", splitLineWkt).execute().body();
        JSONObject result = JSONObject.parseObject(response);
        JSONArray dataArray = result.getJSONArray("data");
        List<String> list = new ArrayList<>();
        for (int i = 0; i < dataArray.size(); i++) {
            list.add(dataArray.getString(i));
        }
        return list;
    }
}
