package com.geoway.landteam.landcloud.service.datacq;

import com.geoway.landteam.landcloud.model.datacq.analyze.CommonInfoCQ;
import com.geoway.landteam.landcloud.model.datacq.analyze.EnumAnalyzeResult;
import com.geoway.landteam.landcloud.model.datacq.analyze.ProjectInfo;
import com.geoway.landteam.landcloud.model.datacq.analyze.ResultInfo;
import com.geoway.landteam.landcloud.servface.datacq.BysxzService;
import com.geoway.landteam.landcloud.service.thirddata.utils.PdfAnalyzeUtil;
import com.itextpdf.text.*;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 村镇选址
 * 不需要国家的云查询结果，但是没有国家的app会报错，所以随意保留了国家的一个云分析项(20210923可能不会报错了，未测试)
 * Created by licong on 2021/4/25.
 */
@Service("CloudAnalyzeCQBysxzService")
@Primary
public class CloudAnalyzeCQBysxzService extends CloudAnalyzeCQService {
    @Autowired
    private BysxzService bysxzService;

    public CloudAnalyzeCQBysxzService() throws IOException, DocumentException {
    }
    public boolean ExportReport(List<ProjectInfo> infoList, String requestId) throws DocumentException, IOException {

        Document document = null;
        try {
            ProjectInfo info = infoList.get(0);
            document =this.createPdf(info);

            document.addTitle("村镇选址云分析报告");
            this.addTitle(info.projectName + "村镇选址云分析报告", titleFont,document);
            String content = "按照您勾绘的项目范围，依据国土调查云后台数据，生成的村镇选址云分析报告如下。受数据齐全性、时效性和精度限制，本分析报告仅用于辅助参考，不具有法律效应。";
            this.addContent(content,document);

            ResultInfo result = new ResultInfo();
            List<Map<String, Object>> czxzList = this.bysxzService.getBysxzData(requestId);
            this.BaseInfo(infoList, czxzList,document);

            this.secondPart(infoList, result,document);

            this.thirdPart(infoList, result,document);

            this.fourthPart(infoList, result,document);

            this.fifthPart(infoList, result,document);
//            result.passDzfx = EnumAnalyzeResult.AllPass;

            this.SummarizeAnalysis(info, result,document);

            this.AddSign(info, czxzList,document);

            this.addDescribe(document);

            document.close();
            return true;
        } catch (Exception ex) {
            ex.printStackTrace();
            if (document != null && document.isOpen()) {
                document.close();
            }
            return false;
        }
    }

    /**
     * 一、基本情况
     * @param infoList
     * @throws DocumentException
     * @throws IOException
     */
    protected void BaseInfo(List<ProjectInfo> infoList, List<Map<String, Object>> czxzList,Document document) throws DocumentException, IOException {
        ProjectInfo info = this.getOtherNode(infoList);
        this.addFirstTitle("一、基本情况",document);
        String content ="%s项目位于%s%s%s%s（东经：%s°，北纬：%s°），项目地块面积%s平方米。";
        if (StringUtils.isEmpty(info.location)) {
            content = String.format(content,info.projectName,info.provinceName,(info.logicCity?"": info.cityName),info.countyName,info.villageName,info.lontitude,info.latitude, String.valueOf(PdfAnalyzeUtil.Round(info.area,2)));
        } else {
            content ="%s项目位于%s%s（东经：%s°，北纬：%s°），项目地块面积%s平方米。";
            content = String.format(content,info.projectName,info.provinceName,info.location,info.lontitude,info.latitude, String.valueOf(PdfAnalyzeUtil.Round(info.area,2)));
        }

        if (null != czxzList && !czxzList.isEmpty()) {
            Map<String, Object> objMap = czxzList.get(0);
            if (objMap != null) {
                content = "项目名称为：" + objMap.get("f_xmmz").toString() + "，项目用地性质为" + objMap.get("f_sqlx").toString() + "，" + content;
            }
        }

        content = this.getAreaContent(info, content);
        this.addContent(content,document);
        this.addTable(info,document);
        this.AddBaseImg(infoList,document);
    }

    /**
     * 基本信息各类占用面积
     * @param info
     * @param content
     * @return
     */
    protected String getAreaContent(ProjectInfo info, String content) {
        String dlinfo="";
        if(info.tdghCqInfos!=null && info.tdghCqInfos.size()>0){
            dlinfo="土地利用现状：";
            dlinfo += this.getTdlyxz(info);
            if(info.tdqsInfos!=null && info.tdqsInfos.size()>0) {
                dlinfo+="；";
            } else {
                dlinfo+="。";
            }
        }
        if(info.tdqsInfos!=null && info.tdqsInfos.size()>0) {
            dlinfo += "土地权属为";
            Map<String,Double> tdqsMap = this.getTdqs(info);
            for (Map.Entry<String, Double> entry: tdqsMap.entrySet()) {
                dlinfo += entry.getKey() + PdfAnalyzeUtil.Round(entry.getValue(), 2) + "平方米、";
            }
            if(dlinfo.endsWith("、")) {
                dlinfo = dlinfo.substring(0, dlinfo.length() - 1) + "；";
            }
        }
        if (info.cqsdInfos != null && info.cqsdInfos.size() > 0) {
            Double area = 0.0d;
            for (CommonInfoCQ obj : info.cqsdInfos) {
                area += obj.area;
            }
            dlinfo += "湿地" + PdfAnalyzeUtil.Round(area, 2) + "平方米。";
        } else {
            dlinfo += "不占用湿地。";
        }
        content+=dlinfo;
        return content;
    }

    /**
     * 图例利用现状占用面积
     * @param info
     * @return
     */
    protected String getTdlyxz(ProjectInfo info) {
        String dlinfo = "";
        double otherArea = 0;
        int count = 0;
        for (CommonInfoCQ item : info.tdghCqInfos) {
            count++;
            if(count>2) {
                otherArea+=item.area;
            } else {
                dlinfo += item.type + PdfAnalyzeUtil.Round(item.area, 2) + "平方米";
                if (count < info.tdghCqInfos.size()) {
                    dlinfo += "、";
                }
            }
        }
        if(otherArea>0) {
            dlinfo += ("其他" + PdfAnalyzeUtil.Round(otherArea, 2) + "平方米");
        }
        return dlinfo;
    }

    /**
     * 写入基本情况
     * @param infoList
     * @throws DocumentException
     */
    protected void AddBaseImg(List<ProjectInfo> infoList,Document document) throws DocumentException {
        ProjectInfo otherInfo = this.getOtherNode(infoList);
        PdfPCell c1 = this.createImgTitle("项目区位置");
        PdfPCell cimg1 = null;
        PdfPCell meta1 = null;
        if (null != otherInfo.ownershipImg) {//项目区位置图
            cimg1 = this.creaeteImgCell(otherInfo.ownershipImg);
            meta1 = this.createImgRoot(otherInfo.ownershipImgMeta);
        }
        int col=2;
        if(otherInfo.narrow) {
            col = 1;
            PdfPTable t1 = new PdfPTable(col);//创建一个1列表格对象
            t1.setSpacingBefore(pgSpace);//设置上边距
            t1.setSpacingAfter(pgSpace);//设置下边距
            t1.setWidthPercentage(80);
            if (otherInfo.ownershipImg != null) {
                t1.addCell(c1);
                t1.addCell(cimg1);
                t1.addCell(meta1);
            }

            document.add(t1);
        } else {
            PdfPTable t = new PdfPTable(col);//创建一个2列表格对象
            t.setSpacingBefore(pgSpace);//设置上边距
            t.setSpacingAfter(pgSpace);//设置下边距
            t.setWidthPercentage(100);
            t.addCell(c1);
            t.addCell(cimg1);
            t.addCell(meta1);
            if (otherInfo.photo1 != null || otherInfo.photo2 != null) {
                this.addPhoto(otherInfo, t);
            }
            document.add(t);
        }
    }

    protected void addTable(ProjectInfo info,Document document) throws DocumentException {
        this.addTitle("土地利用现状表", sectionFont,document);
        this.addTableRightTxt("单位：平方米",document);
        int[] tableWidth = { 30, 30, 30};
        PdfPTable table = this.getPdfPTable(3, tableWidth);

        table.addCell(this.createTableCell(info.tdghCqInfos.size() + 1, "现状地类名称"));
        table.addCell(this.createImgTitle("地类"));
        table.addCell(this.createImgTitle("面积"));
        for (int i = 0; i < info.tdghCqInfos.size(); i++) {
            CommonInfoCQ obj = info.tdghCqInfos.get(i);
            table.addCell(this.createImgTitle(obj.type));
            table.addCell(this.createImgTitle(String.valueOf(PdfAnalyzeUtil.Round(obj.area,2)) + "平方米"));
        }

        Map<String,Double> tdqsMap = this.getTdqs(info);
        table.addCell(this.createTableCell(tdqsMap.size() + 1,"权属性质"));
        table.addCell(this.createImgTitle("权属性质"));
        table.addCell(this.createImgTitle("面积"));
        for (Map.Entry<String, Double> entry: tdqsMap.entrySet()) {
            table.addCell(this.createImgTitle(entry.getKey()));
            table.addCell(this.createImgTitle(String.valueOf(PdfAnalyzeUtil.Round(entry.getValue(),2)) + "平方米"));
        }

        if (null != info.cqsdInfos && !info.cqsdInfos.isEmpty()) {
            table.addCell(this.createTableCell(info.cqsdInfos.size() + 1, "湿地范围内地类"));
            table.addCell(this.createImgTitle("地类"));
            table.addCell(this.createImgTitle("面积"));
            for (int i = 0; i < info.cqsdInfos.size(); i++) {
                CommonInfoCQ obj = info.cqsdInfos.get(i);
                table.addCell(this.createImgTitle(this.getSdByType(obj.type)));
                table.addCell(this.createImgTitle(String.valueOf(PdfAnalyzeUtil.Round(obj.area,2)) + "平方米"));
            }
        }
        document.add(table);
    }

    protected Map<String, Double> getTdqs(ProjectInfo info){
        HashMap<String,Double> result = new HashMap<>();
        for(CommonInfoCQ item : info.tdqsInfos) {
            String type = this.getQsxzByType(item.type);
            if (!result.containsKey(type)) {
                result.put(type, 0.0d);
            }
            result.put(type, result.get(type) + item.area);
        }
        return result;
    }

    private PdfPCell createTableCell(Integer col, String content){
        PdfPCell cell = new PdfPCell(new Phrase(content, normalFont));
//        cell.setColspan(col);
        cell.setRowspan(col);
//        this.addImgTxtFormat(cell);
        cell.setHorizontalAlignment(Element.ALIGN_CENTER);
        cell.setVerticalAlignment(Element.ALIGN_MIDDLE);
        cell.setPaddingTop(5);
        cell.setPaddingBottom(5);
        return cell;
    }

    protected void addTableRightTxt(String sign,Document document) throws DocumentException {
        Paragraph paragraph = new Paragraph(sign, normalFont);
        paragraph.setAlignment(Element.ALIGN_RIGHT);
//        paragraph.setFirstLineIndent(300);
//        paragraph.setExtraParagraphSpace(30);
//        paragraph.setSpacingAfter(24);
//        paragraph.setLeading(22);
        document.add(paragraph);
    }

    protected PdfPTable getPdfPTable(int column, int[] tableWidth) throws DocumentException {
        PdfPTable table = new PdfPTable(column);
        table.setWidths(tableWidth);
//        t.setKeepTogether(true);
        table.setSpacingBefore(pgSpace);//设置上边距
        table.setSpacingAfter(pgSpace);//设置下边距
        table.getDefaultCell().setBorder(0);
        table.setWidthPercentage(90);
        return table;
    }

    private String getQsxzByType(String type) {
        switch (type) {
            case "10":
                return "国有土地所有权";
            case "20":
                return "国有土地使用权";
            case "30":
                return "集体土地所有权";
        }
        return type;
    }
    private String getSdByType(String type) {
        switch (type) {
            case "201":
                return "永久性河流";
            case "203":
                return "泛洪平原湿地";
            case "301":
                return "永久性淡水湖";
            case "402":
                return "草本沼泽";
            case "407":
                return "地热湿地";
            case "501":
                return "蓄水区";
            case "502":
                return "淡水养殖池塘";
        }
        return type;
    }

    /**
     * 生态保护红线分析
     * @param infoList
     * @param result
     * @throws DocumentException
     */
    protected void secondPart(List<ProjectInfo> infoList,ResultInfo result,Document document) throws DocumentException {
        ProjectInfo info = null;
        ProjectInfo other = this.getOtherNode(infoList);
        this.addFirstTitle("二、生态保护底线分析",document);
        this.addSthxMessage(result, info, other, "(一)生态保护红线情况分析",document);
        this.addZrbhdMessage(result, other, "(二)自然保护地情况分析",document);
    }

    /**
     * 永久基本农田分析
     * @param infoList
     * @param result
     * @throws DocumentException
     */
    protected void thirdPart(List<ProjectInfo> infoList,ResultInfo result,Document document) throws DocumentException {
        ProjectInfo info = null;
        ProjectInfo other = this.getOtherNode(infoList);
        this.addFirstTitle("三、永久基本农田分析",document);
        this.addJbntMessageContent(result, info, other,document);
    }

    /**
     * 村镇国土空间规划
     * @param infoList
     * @param result
     * @throws DocumentException
     * 1，有村规，符合村规就结束
     * 2，有村规，不符合村规，分析土规，符合土规（显示土规），不符合土规就不显示
     * 3，无村规，不显示村规，分析土规，显示土规
     */
    protected void fourthPart(List<ProjectInfo> infoList,ResultInfo result,Document document) throws DocumentException {
        ProjectInfo other = this.getOtherNode(infoList);
        this.addFirstTitle("四、国土空间规划分析",document);
        this.addCzkfbjMessage(result, other, "(一)城镇开发边界情况分析",document);
        this.addSencondTitle("(二)镇村国土空间规划",document);
        this.addCghContent(result, other,document);
        if (!(EnumAnalyzeResult.AllPass == result.passCgh && other.cghInfos != null)) {
            this.addTdghCQMessage(result, other,document);
        } else {
            result.passTdgh = result.passCgh;
        }
    }

    /**
     * 地质灾害安全分析
     * @param infoList
     * @param result
     * @throws DocumentException
     */
    protected void fifthPart(List<ProjectInfo> infoList,ResultInfo result,Document document) throws DocumentException {
        ProjectInfo other = this.getOtherNode(infoList);
        this.addFirstTitle("五、地质灾害安全分析",document);
        this.addDzzhCQMessageContent(other, result,document);
    }

    protected void SummarizeAnalysis(ProjectInfo info,ResultInfo result,Document document) throws DocumentException {
        this.addFirstTitle("六、总体分析",document);

        String content = "根据已有数据分析结果，拟建设项目区";
        boolean legal = result.passSthx == EnumAnalyzeResult.AllPass && result.passZrbhd == EnumAnalyzeResult.AllPass
                && result.passJbnt == EnumAnalyzeResult.AllPass && result.passCgh == EnumAnalyzeResult.AllPass && result.passTdgh == EnumAnalyzeResult.AllPass;
        boolean suitable = result.passDzfx == EnumAnalyzeResult.AllPass ;
        boolean unlegal = result.passSthx == EnumAnalyzeResult.NoPass || result.passZrbhd == EnumAnalyzeResult.NoPass
                || result.passJbnt == EnumAnalyzeResult.NoPass || result.passCgh == EnumAnalyzeResult.NoPass || result.passTdgh == EnumAnalyzeResult.NoPass;
        boolean unsuitable = result.passDzfx == EnumAnalyzeResult.NoPass ;

        boolean allPass = legal && suitable;
        if (allPass) {
            content += "合法、适宜。";
            content += "不涉及生态红线";
            content += "不涉及占用自然保护地、";
            content += "不涉及占用永久基本农田、";
            content += "符合村规划、";
            content += "符合土地利用总体规划、";
            content += "不存在地质灾害高易发区。";;
        } else if (unlegal) {
            content += "不合法、不适宜。";
            switch (result.passSthx) {
                case NoPass:
                    content += "涉及生态红线、";
                    break;
                case PartPass:
                    content += "部分区域涉及生态红线、";
                    break;
            }
            switch (result.passZrbhd) {
                case NoPass:
                    content += "涉及占用自然保护地、";
                    break;
                case PartPass:
                    content += "部分区域涉及占用自然保护地、";
                    break;
            }
            switch (result.passJbnt) {
                case NoPass:
                    content += "涉及占用永久基本农田、";
                    break;
                case PartPass:
                    content += "部分区域涉及占用永久基本农田、";
                    break;
            }
            if (result.passTdgh == EnumAnalyzeResult.NoPass && result.passCgh == EnumAnalyzeResult.NoPass) {
                content += "不符合村庄规划或土地利用总体规划、";
            }
//            switch (result.passCgh) {
//                case NoPass:
//                    content += "不符合村庄规划、";
//                    break;
//                case PartPass:
//                    content += "部分符合村庄规划、";
//                    break;
//            }
//            switch (result.passTdgh) {
//                case NoPass:
//                    content += "不符合土地利用总体规划、";
//                    break;
//                case PartPass:
//                    content += "部分区域不符合土地利用总体规划、";
//                    break;
//            }
            switch (result.passDzfx) {
                case NoPass:
                    content += "存在地质灾害高易发区、";
                    break;
                case PartPass:
                    content += "部分区域存在地质灾害高易发区、";
                    break;
            }
            content = content.substring(0, content.length() - 1) + "。";
        } else if (unsuitable) {
            content += "不适宜。";
            switch (result.passDzfx) {
                case NoPass:
                    content += "存在地质灾害高易发区、";
                    break;
                case PartPass:
                    content += "部分区域存在地质灾害高易发区、";
                    break;
            }
            content = content.substring(0, content.length() - 1) + "。";
        } else {
            boolean partlegal = false;
            boolean partsuitable = false;
            String tmp = "";
            if (result.passSthx == EnumAnalyzeResult.PartPass) {
                tmp += "涉及生态红线、";
                partlegal = true;
            }
            if (result.passZrbhd == EnumAnalyzeResult.PartPass) {
                tmp += "涉及占用自然保护地、";
                partlegal = true;
            }
            if (result.passJbnt == EnumAnalyzeResult.PartPass) {
                tmp += "涉及占用永久基本农田、";
                partlegal = true;
            }
            if (result.passTdgh == EnumAnalyzeResult.PartPass) {
                tmp += "不符合土地利用总体规划、";
                partlegal = true;
            }
            if (result.passCgh == EnumAnalyzeResult.PartPass) {
                tmp += "不符合村庄规划、";
                partlegal = true;
            }
            if (result.passDzfx == EnumAnalyzeResult.PartPass) {
                tmp += "存在地质灾害高易发区、";
                partsuitable = true;
            }
            if (!StringUtils.isEmpty(tmp))
                tmp = tmp.substring(0, tmp.length() - 1) + "。";
            if (partlegal && partsuitable) {
                content += "存在不符合法律法规、不适宜的情况。比如部分地区";
            } else if (partlegal) {
                content += "存在不符合法律法规的情况。比如部分地区";
            } else if (partsuitable) {
                content += "存在不适宜的情况。比如部分地区";
            }
            content += tmp;
        }

        this.addContent(content,document);
    }

    protected void addDescribe(Document document) throws DocumentException {
        document.newPage();
        this.addFirstTitle("附：涉及政策解读",document);
        this.addSthxDescribe("1、关于生态保护红线",document);
        this.addZrbhqDescribe("2、关于自然保护地",document);
        this.addJbntDescribe("3、关于永久基本农田",document);
        this.addCzkfbjDescribe("4、关于城镇开发边界",document);
        this.addTdghDescribe("5、关于村镇国土空间规划",document);
        this.addDzzhDescribe("6、关于地址灾害安全",document);
    }

    protected void AddSign(ProjectInfo info, List<Map<String, Object>> czxzList,Document document) throws DocumentException {
        if (!czxzList.isEmpty()) {
            Map<String, Object> objMap = czxzList.get(0);
            if (objMap != null) {
                String sign="申请人： " + objMap.get("f_lxr").toString();
                this.addSignContent(info.narrow, sign,document);
            }
        }
        String sign="踏勘人员： "+info.userName;
        this.addSignContent(info.narrow, sign,document);

        String timeStr = "时  间： "+ new SimpleDateFormat("yyyy-MM-dd").format(new Date()).toString();
        this.addSignContent(info.narrow, timeStr,document);
    }

    /**
     * 自然保护地情况分析
     * @param result
     * @throws DocumentException
     */
    protected void addZrbhdMessage(ResultInfo result, ProjectInfo other, String title,Document document) throws DocumentException {
        this.addSencondTitle(title,document);
        String zrbhdDesc = "基于重庆自然保护地数据分析：";
        Double area = 0d;
        if(null != other.zrbhdInfos && other.zrbhdInfos.size() > 0) {
            result.passZrbhd= EnumAnalyzeResult.NoPass;
            Double hxqarea = 0d;
            zrbhdDesc += "拟建设项目区涉及自然保护地，涉及自然保护地面积为";
            for (int i = 0; i < other.zrbhdInfos.size(); i++) {
                zrbhdDesc += other.zrbhdInfos.get(i).type + PdfAnalyzeUtil.Round(other.zrbhdInfos.get(i).area,2) + "平方米";
                area += other.zrbhdInfos.get(i).area;
                if (i != other.zrbhdInfos.size() - 1) {
                    zrbhdDesc += "、";
                }
                if (other.zrbhdInfos.get(i).type.contains("核心")) {
                    hxqarea += other.zrbhdInfos.get(i).area;
                }
            }
            if(area < other.area) {
                result.passZrbhd=EnumAnalyzeResult.PartPass;
            }

            if (hxqarea != 0d) {
//                zrbhdDesc += "，其中核心区" + PdfAnalyzeUtil.Round(hxqarea,2) + "平方米";
            }
        } else {
            result.passZrbhd=EnumAnalyzeResult.AllPass;
            zrbhdDesc += "拟建设项目区不涉及自然保护地。";
        }
        this.addContent(zrbhdDesc,document);

        if(other.zrbhdImg != null) {
            String desc = "自然保护地："+PdfAnalyzeUtil.Round(area,2)+"平方米";
            AddImg("自然保护用地",other.zrbhdImg,desc,document);
        }
    }

    /**
     * 重庆土地利用总体规划数据分析
     * @param result
     * @param info
     * @throws DocumentException
     */
    protected void addTdghCQMessage(ResultInfo result, ProjectInfo info,Document document) throws DocumentException {
        String tdghResult ="基于重庆土地利用总体规划数据分析：";
        Boolean cantBuild = true;
        String tdghDesc ="";
        if(info.tdghCqInfos != null) {
            tdghResult += "拟建项目区土地利用总体规划中规划地类为";
            for(int i =0; i < info.tdghCqInfos.size(); i++) {
                CommonInfoCQ item = info.tdghCqInfos.get(i);
                String type = item.type;
                Double typeArea = item.area;
                tdghDesc += type + ":" + PdfAnalyzeUtil.Round(typeArea,2) + "平方米";
                if (i != info.tdghCqInfos.size() - 1)
                    tdghDesc += "，";
                if (!("农村居民点用地").equals(type) && !"城镇用地".equals(type) && typeArea > 0){
                    cantBuild = false;
                }
            }
            tdghResult += tdghDesc + "。";
        }

        if(!cantBuild) {
            result.passTdgh = EnumAnalyzeResult.NoPass;
            if ((EnumAnalyzeResult.NoPass == result.passCgh && info.cghInfos != null))
                return;//有村规且不符合土规，只显示村规划
            tdghResult +="拟建设项目区不符合土地利用总体规划。";
        } else {
            result.passTdgh = EnumAnalyzeResult.AllPass;
            tdghResult +="拟建设项目区符合土地利用总体规划。";
        }
        this.addContent(tdghResult,document);

        if(info.tdghCqImg!=null) {
            AddImg("土地利用总体规划",info.tdghCqImg,tdghDesc,document);
        }
    }

    protected void addDzzhCQMessageContent(ProjectInfo other, ResultInfo result,Document document) throws DocumentException {
        String dzzhResult ="基于重庆地质灾害高易发区数据分析：";
        double dzfxArea = 0;
        if((null != other.dzzhArea && other.dzzhArea > 0)) {
            result.passDzfx = EnumAnalyzeResult.NoPass;
            if(dzfxArea < other.area) {
                result.passDzfx = EnumAnalyzeResult.PartPass;
            }
            dzzhResult +="拟建设项目区存在地质灾害高易发区。";
        } else {
            result.passDzfx = EnumAnalyzeResult.AllPass;
            dzzhResult +="拟建设项目区不存在地质灾害高易发区。";
        }
        this.addContent(dzzhResult,document);

        if(other.dzzhImg!=null) {
            String desc = "地质灾害高易发区：" + PdfAnalyzeUtil.Round(other.dzzhArea,2)+"平方米";
            AddImg("地质灾害高易发区",other.dzzhImg,desc,document);
        }
    }
}
