package com.geoway.landteam.landcloud.service.datacq;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.geoway.landteam.cloudquery.model.pub.entity.CloudQuery;
import com.geoway.landteam.cloudquery.repository.pub.CloudQueryRepository;
import com.geoway.landteam.cloudquery.servface.pub.CloudQueryService;
import com.geoway.landteam.landcloud.common.util.ResultSetInfo;
import com.geoway.landteam.landcloud.common.util.geometry.GeometryUtil;
import com.geoway.landteam.landcloud.common.util.orm.SqlliteConnTool;
import com.geoway.landteam.landcloud.core.model.base.entity.AppMedia;
import com.geoway.landteam.landcloud.core.model.base.entity.Region;
import com.geoway.landteam.landcloud.core.model.base.entity.RegionCorrect;
import com.geoway.landteam.landcloud.core.model.pub.entity.EnumeratorDomain;
import com.geoway.landteam.landcloud.core.repository.base.AppMediaRepository;
import com.geoway.landteam.landcloud.core.servface.region.RegionCorrectService;
import com.geoway.landteam.landcloud.core.servface.region.RegionService;
import com.geoway.landteam.landcloud.core.service.base.DefaultOssOperatorService;
import com.geoway.landteam.landcloud.core.service.base.TemporarySignedUrlService;
import com.geoway.landteam.landcloud.model.datacq.analyze.*;
import com.geoway.landteam.landcloud.model.datacq.entity.CloudAnalyzeConfig;
import com.gw.base.log.GiLoger;
import com.gw.base.log.GwLoger;
import com.itextpdf.text.*;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.itextpdf.text.pdf.PdfWriter;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.util.Constants;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.io.WKTReader;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.*;
import java.net.URL;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.*;

@Service
@Transactional(rollbackFor = Exception.class)
public class CloudAnalyzeService {
    private GiLoger logger = GwLoger.getLoger(CloudAnalyzeService.class);
    @Value("${project.downloadDir}")
    String downloadDir;

    @Autowired
    CloudQueryRepository cloudQueryDao;

    @Autowired
    RegionCorrectService regionCorrectService;

    @Autowired
    RegionService regionService;

    @Autowired
    AppMediaRepository appMediaDao;

    @Autowired
    TemporarySignedUrlService temporarySignedUrlService;

    @Autowired
    DefaultOssOperatorService ossOperatorService;


    @Autowired
    CloudQueryService cloudQueryService;

    //Document document;

    Font titleFont;

    Font chapterFont;

    Font sectionFont;

    Font normalFont;

    Font boldFont;

    Font roleFont;

    Font redFont;

    float pgSpace = 18;
    float lineSpace = 16;
    float firstLineIndent = 24;


    public CloudAnalyzeService() throws IOException, DocumentException {
        ClassPathResource STSONG = new ClassPathResource("/static/STSONG.TTF");
        ClassPathResource simhei = new ClassPathResource("/static/simhei.ttf");
        ClassPathResource simkai = new ClassPathResource("/static/simkai.ttf");
        String pathClassPath = Constants.class.getClassLoader().getResource("").getPath();
        /* String rootPath = pathClassPath.substring(0, pathClassPath.indexOf("WEB-INF"));*/
        if (STSONG.exists()) {
            InputStream STSONGinputStream = STSONG.getInputStream();
            InputStream simheiinputStream = simhei.getInputStream();
            InputStream simkaiinputStream = simkai.getInputStream();
            //生成目标文件
            File STSONGFile = File.createTempFile("template_STSONG_copy", ".TTF");
            File simheiFile = File.createTempFile("template_simhei_copy", ".ttf");
            File simkaiFile = File.createTempFile("template_simkai_copy", ".ttf");
            try {
                FileUtils.copyInputStreamToFile(STSONGinputStream, STSONGFile);
                FileUtils.copyInputStreamToFile(simheiinputStream, simheiFile);
                FileUtils.copyInputStreamToFile(simkaiinputStream, simkaiFile);

                titleFont = new Font(BaseFont.createFont(simheiFile.getAbsolutePath(), BaseFont.IDENTITY_H, BaseFont.NOT_EMBEDDED,  BaseFont.NOT_CACHED, null, null), 16);
                titleFont.setStyle(Font.BOLD);

                chapterFont = new Font(BaseFont.createFont(simheiFile.getAbsolutePath(), BaseFont.IDENTITY_H, BaseFont.NOT_EMBEDDED, BaseFont.NOT_CACHED, null, null), 14);
                chapterFont.setStyle(Font.BOLD);

                sectionFont = new Font(BaseFont.createFont( simkaiFile.getAbsolutePath(), BaseFont.IDENTITY_H, BaseFont.NOT_EMBEDDED, BaseFont.NOT_CACHED, null, null), 12);
                sectionFont.setStyle(Font.BOLD);

                normalFont = new Font(BaseFont.createFont(  STSONGFile.getAbsolutePath(), BaseFont.IDENTITY_H, BaseFont.NOT_EMBEDDED, BaseFont.NOT_CACHED, null, null), 12);
                boldFont = new Font(BaseFont.createFont(STSONGFile.getAbsolutePath(), BaseFont.IDENTITY_H, BaseFont.NOT_EMBEDDED, BaseFont.NOT_CACHED, null, null), 12);
                boldFont.setStyle(Font.BOLD);
                roleFont = new Font(BaseFont.createFont( simkaiFile.getAbsolutePath(), BaseFont.IDENTITY_H, BaseFont.NOT_EMBEDDED, BaseFont.NOT_CACHED, null, null), 10);

                redFont = new Font(BaseFont.createFont( simkaiFile.getAbsolutePath(), BaseFont.IDENTITY_H, BaseFont.NOT_EMBEDDED, BaseFont.NOT_CACHED, null, null), 12);
                redFont.setColor(255, 0, 0);
            } finally {
                IOUtils.closeQuietly(STSONGinputStream,simheiinputStream,simkaiinputStream);
            }
        }
    }

    public Document createPdf(String outFile,boolean narrow) throws DocumentException, FileNotFoundException {
        File file = new File(outFile);
        file.getParentFile().mkdirs();

        // 生成PDF文档
        Rectangle size = PageSize.A4;
        if (narrow) {
            size = PageSize.A5;
        }
        Rectangle rectPageSize = new Rectangle(size);
        //创建一个文档对象，设置初始化大小和页边距
        Document document = new Document(rectPageSize, 40, 40, 40, 40);// 上、下、左、右间距

        PdfWriter pdfWriter = PdfWriter.getInstance(document, new FileOutputStream(outFile));
        //设置行间距
        pdfWriter.setInitialLeading(30);
        document.open();
        return document;
    }

    public ProjectInfo Parse(String requestId, String tbid, String userName, int analyzeType,String tag, boolean px) throws Exception {

        ProjectInfo info = new ProjectInfo();
        if (tbid != null && !tbid.isEmpty()) {
            List<AppMedia> medias = appMediaDao.queryPhotosByGid(tbid);
            if (medias != null) {
                int count = medias.size();
                if (count >= 1) {
                    AppMedia media = medias.get(0);
                    String url = media.getDownloadUrl();
                    url = temporarySignedUrlService.getTemporarySignedUrl("", url, null);
                    if(url.indexOf("%3D")>-1){
                        url = url.replace("%3D","=");
                    }
                    String fileName = media.getId() + ".jpg";
                    File lsFile = new File(downloadDir, fileName);
                    if (!lsFile.exists()) {
                        FileUtils.copyURLToFile(new URL(url), lsFile);
                    }
                    FileInputStream fis1 = new FileInputStream(lsFile);
                    byte[] photo1 = new byte[(int) lsFile.length()];
                    fis1.read(photo1);
                    fis1.close();

                    info.photo1 = Image.getInstance(photo1); //Image.getInstance(url);url 模式会自动下载，界面上会显示下载信息
                    lsFile.delete();
                    info.photoMeta1 = userName + " " + media.getLon().toString() + "°E " + media.getLat().toString() + "°N";
                    String azi = media.getAzimuth();
                    double azid = Double.parseDouble(azi);
                    String time = media.getTime();
                    Date dTime = new Date(Long.parseLong(time));
                    time = new SimpleDateFormat("yyyy-MM-dd").format(dTime).toString();
                    info.photoMeta1 += (" " + GetDirction(azid) + " " + time);
                }
                if (count >= 2) {
                    AppMedia media = medias.get(1);
                    String url = media.getDownloadUrl();
                    if(url.indexOf("%3D")>-1){
                        url = url.replace("%3D","=");
                    }
                    url = temporarySignedUrlService.getTemporarySignedUrl("", url, null);
                    String fileName = media.getId() + ".jpg";
                    File lsFile = new File(downloadDir, fileName);
                    if (!lsFile.exists()) {
                        FileUtils.copyURLToFile(new URL(url), lsFile);
                    }
                    FileInputStream fis1 = new FileInputStream(lsFile);
                    byte[] photo2 = new byte[(int) lsFile.length()];
                    fis1.read(photo2);
                    fis1.close();

                    info.photo2 = Image.getInstance(photo2);
                    lsFile.delete();
                    info.photoMeta2 = userName + " " + media.getLon().toString() + "°E " + media.getLat().toString() + "°N";
                    String azi = media.getAzimuth();
                    double azid = Double.parseDouble(azi);
                    String time = media.getTime();
                    Date dTime = new Date(Long.parseLong(time));
                    time = new SimpleDateFormat("yyyy-MM-dd").format(dTime).toString();
                    info.photoMeta2 += (" " + GetDirction(azid) + " " + time);
                }
            }
        }
        List<CloudQuery> cloudQueryList = cloudQueryDao.findByRequestIdTag(requestId, analyzeType,tag);
        if (cloudQueryList == null || cloudQueryList.size() < 1) {
            throw new RuntimeException("无查询结果");
        }
        for (CloudQuery query : cloudQueryList) {
            if (info.lontitude == null || info.lontitude.isEmpty()) {
                String param = query.getParam();
                JSONObject detail = JSONObject.parseObject(param);
                if (detail.containsKey("lon")) {
                    info.lontitude = detail.get("lon").toString();
                }
                if (detail.containsKey("lat")) {
                    info.latitude = detail.get("lat").toString();
                }
                if (detail.containsKey("mj")) {
                    String mj = detail.get("mj").toString();
                    info.area = Double.parseDouble(mj);
                }
                String lonStr = info.lontitude;
                String latStr = info.latitude;
                double lon = Double.parseDouble(lonStr);
                double lat = Double.parseDouble(latStr);
                RegionCorrect regionCorrect = regionCorrectService.getOfCoord(lon, lat);
                if (regionCorrect != null) {
                    String countyCode = regionCorrect.getId();
                    if (countyCode.length() == 6) {
                        //根据县代码获取省市县名称
                        String cityCode = countyCode.substring(0, 4) + "00";
                        String provinceCode = countyCode.substring(0, 2) + "0000";
                        Region county = regionService.findRegionByCode(countyCode);
                        Region city = regionService.findRegionByCode(cityCode);
                        Region province = regionService.findRegionByCode(provinceCode);
                        info.provinceCode = provinceCode;
                        info.cityCode = cityCode;
                        info.countyCode = countyCode;
                        if (province != null) {
                            info.provinceName = province.getName();
                        }
                        if (city != null) {
                            info.cityName = city.getName();
                        }
                        if (county != null) {
                            info.countyName = county.getName();
                        }
                    }
                }
            }

            String url = cloudQueryService.getDBUrl(query.getId(),null);

            String[] paths = url.split("/");
            String fileName = paths[paths.length - 1]; // 保存的文件名称

            if(fileName.indexOf("?")>-1) {
                fileName = fileName.substring(0, fileName.indexOf("?"));
            }
            File lsFile = new File(downloadDir, fileName);
            if (!lsFile.exists()) {
                FileUtils.copyURLToFile(new URL(url), lsFile);
            }
            SqlliteConnTool dbHelper = new SqlliteConnTool(lsFile.getAbsolutePath());
            ArrayList<String> tables = dbHelper.getTables();
            if (tables == null || tables.size() == 0) {
                continue;
            }
            if (analyzeType != 0) {
                for (String table : tables) {
                    Parse(dbHelper, table, info);
                }
                break;
            } else {
                if (tables.contains("Image_Analyze"))//时序查询结果，里面包含了地类分析数据等
                {
                    for (String table : tables) {
                        String tn = table.toLowerCase();
                        System.out.println("table:" + tn);
                        switch (tn) {
                            case "image_analyze":
                                info.ownershipImgMeta = GetImageMeta(dbHelper,
                                        "Image_Analyze");
                                break;
                        }

                    }
                } else {
                    for (String table : tables) {
                        Parse(dbHelper, table, info);
                    }
                }
            }
            dbHelper.Dispose();
        }
        return info;
    }

    public LawlessInfo ParseLawless(String requestId, String tbid, String userName, int analyzeType,String tag) throws Exception {

        LawlessInfo info = new LawlessInfo();
        info.userName = userName;
        List<CloudQuery> cloudQueryList = cloudQueryDao.findByRequestIdTag(requestId, analyzeType,tag);
        if (cloudQueryList == null || cloudQueryList.size() < 1) {
            throw new RuntimeException("无查询结果");
        }
        for (CloudQuery query : cloudQueryList) {
            if (info.lontitude == null || info.lontitude.isEmpty()) {
                String param = query.getParam();
                JSONObject detail = JSONObject.parseObject(param);
                if (detail.containsKey("lon")) {
                    info.lontitude = detail.get("lon").toString();
                }
                if (detail.containsKey("lat")) {
                    info.latitude = detail.get("lat").toString();
                }
                if (detail.containsKey("mj")) {
                    String mj = detail.get("mj").toString();
                    info.area = Double.parseDouble(mj);
                }
                String lonStr = info.lontitude;
                String latStr = info.latitude;
                double lon = Double.parseDouble(lonStr);
                double lat = Double.parseDouble(latStr);
                RegionCorrect regionCorrect = regionCorrectService.getOfCoord(lon, lat);
                if (regionCorrect != null) {
                    String countyCode = regionCorrect.getId();
                    if (countyCode.length() == 6) {
                        //根据县代码获取省市县名称
                        String cityCode = countyCode.substring(0, 4) + "00";
                        String provinceCode = countyCode.substring(0, 2) + "0000";
                        Region county = regionService.findRegionByCode(countyCode);
                        Region city = regionService.findRegionByCode(cityCode);
                        Region province = regionService.findRegionByCode(provinceCode);
                        info.provinceCode = provinceCode;
                        info.cityCode = cityCode;
                        info.countyCode = countyCode;
                        if (province != null) {
                            info.provinceName = province.getName();
                        }
                        if (city != null) {
                            info.cityName = city.getName();
                        }
                        if (county != null) {
                            info.countyName = county.getName();
                        }
                    }
                }
            }
            String url = cloudQueryService.getDBUrl(query.getId(),null);
            String[] paths = url.split("/");
            String fileName = paths[paths.length - 1]; // 保存的文件名称

            if(fileName.indexOf("?")>-1) {
                fileName = fileName.substring(0, fileName.indexOf("?"));
            }
            File lsFile = new File(downloadDir, fileName);
            if (!lsFile.exists()) {
                FileUtils.copyURLToFile(new URL(url), lsFile);
            }
            SqlliteConnTool dbHelper = new SqlliteConnTool(lsFile.getAbsolutePath());
            ArrayList<String> tables = dbHelper.getTables();
            if (tables == null || tables.size() == 0) {
                continue;
            }
            if (analyzeType != 0) {
                for (String table : tables) {
                    ParseLawless(dbHelper, table, info);
                }
                break;
            } else {
                if (tables.contains("Image_Analyze"))//时序查询结果，里面包含了地类分析数据等
                {
                    for (String table : tables) {
                        String tn = table.toLowerCase();
                        System.out.println("table:" + tn);
                        switch (tn) {
                            case "image_analyze":
                                info.ownershipImgMeta = GetImageMeta(dbHelper,
                                        "Image_Analyze");
                                ParseLawlessImgCompare(dbHelper, "Image_Analyze", info);
                                break;
                        }

                    }
                } else {
                    for (String table : tables) {
                        ParseLawless(dbHelper, table, info);
                    }
                }
            }
            dbHelper.Dispose();
        }
        return info;
    }

    public YnbhInfo ParseYnbh(String requestId, String tbid, String userName, int analyzeType,String tag) throws Exception {

        YnbhInfo info = new YnbhInfo();
        info.userName = userName;
        List<CloudQuery> cloudQueryList = cloudQueryDao.findByRequestIdTag(requestId, analyzeType,tag);
        if (cloudQueryList == null || cloudQueryList.size() < 1) {
            throw new RuntimeException("无查询结果");
        }
        for (CloudQuery query : cloudQueryList) {
            if (info.lontitude == null || info.lontitude.isEmpty()) {
                String param = query.getParam();
                JSONObject detail = JSONObject.parseObject(param);
                if (detail.containsKey("lon")) {
                    info.lontitude = detail.get("lon").toString();
                }
                if (detail.containsKey("lat")) {
                    info.latitude = detail.get("lat").toString();
                }
                if (detail.containsKey("mj")) {
                    String mj = detail.get("mj").toString();
                    info.area = Double.parseDouble(mj);
                }
                if(detail.containsKey("range2000"))
                {
                    String wkt = detail.getString("range2000").toString();
                    WKTReader wktReader = new WKTReader();
                    Geometry geo = wktReader.read(wkt);
                    CoordinateReferenceSystem prjRef = GeometryUtil.getCgcsPrjCoordinateReferenceSystem(geo);
                    geo = GeometryUtil.transCgcs2000ToPrj(geo,prjRef);
                    info.dkgzd =  geo.getArea()/geo.getLength();
                }
                String lonStr = info.lontitude;
                String latStr = info.latitude;
                double lon = Double.parseDouble(lonStr);
                double lat = Double.parseDouble(latStr);
                RegionCorrect regionCorrect = regionCorrectService.getOfCoord(lon, lat);
                if (regionCorrect != null) {
                    String countyCode = regionCorrect.getId();
                    if (countyCode.length() == 6) {
                        //根据县代码获取省市县名称
                        String cityCode = countyCode.substring(0, 4) + "00";
                        String provinceCode = countyCode.substring(0, 2) + "0000";
                        Region county = regionService.findRegionByCode(countyCode);
                        Region city = regionService.findRegionByCode(cityCode);
                        Region province = regionService.findRegionByCode(provinceCode);
                        info.provinceCode = provinceCode;
                        info.cityCode = cityCode;
                        info.countyCode = countyCode;
                        if (province != null) {
                            info.provinceName = province.getName();
                        }
                        if (city != null) {
                            info.cityName = city.getName();
                        }
                        if (county != null) {
                            info.countyName = county.getName();
                        }
                    }
                }
            }

            String url = cloudQueryService.getDBUrl2(query.getId(),"https://xcx.geoway.com.cn:18381");
            String[] paths = url.split("/");
            String fileName = paths[paths.length - 1]; // 保存的文件名称

            if(fileName.indexOf("?")>-1) {
                fileName = fileName.substring(0, fileName.indexOf("?"));
            }
            File lsFile = new File(downloadDir, fileName);
            if (!lsFile.exists()) {
                FileUtils.copyURLToFile(new URL(url), lsFile);
            }
            SqlliteConnTool dbHelper = new SqlliteConnTool(lsFile.getAbsolutePath());
            ArrayList<String> tables = dbHelper.getTables();
            if (tables == null || tables.size() == 0) {
                continue;
            }
            if (analyzeType != 0) {
                for (String table : tables) {
                    ParseYnbh(dbHelper, table, info);
                }
                break;
            }

            dbHelper.Dispose();
        }
        return info;
    }


    public boolean ExportReport(ProjectInfo info, Boolean px) throws DocumentException, IOException {
        Document document =null;
        try {

            document = createPdf(info.outFile,info.narrow);

            if(px)
            {
                document.addTitle("土地开发选址云分析报告");
                Paragraph title = new Paragraph(info.projectName + "土地开发选址云分析报告", titleFont);
                title.setAlignment(Element.ALIGN_CENTER);
                document.add(title);
                Paragraph first = new Paragraph("按照您勾绘的项目范围，依据国土调查云后台数据，生成的土地开发选址云分析报告如下。受数据齐全性、时效性和精度限制，本分析报告仅用于辅助参考。 ", normalFont);
                first.setFirstLineIndent(firstLineIndent);
                first.setExtraParagraphSpace(30);
                first.setSpacingBefore(pgSpace);
                first.setLeading(lineSpace);
                document.add(first);
            }
            else
            {
                document.addTitle("建设用地选址云分析报告");
                Paragraph title = new Paragraph(info.projectName + "建设用地选址云分析报告", titleFont);
                title.setAlignment(Element.ALIGN_CENTER);
                document.add(title);
                Paragraph first = new Paragraph("按照您勾绘的项目范围，依据国土调查云后台数据，生成的建设用地选址云分析报告如下。受数据齐全性、时效性和精度限制，本分析报告仅用于辅助参考。 ", normalFont);
                first.setFirstLineIndent(firstLineIndent);
                first.setExtraParagraphSpace(30);
                first.setSpacingBefore(pgSpace);
                first.setLeading(lineSpace);
                document.add(first);
            }



            ResultInfo result = new ResultInfo();
            BaseInfo(document,info, result);

            LegalAnalysis(document,info, result);

            BzInfo(document);

            SuitableAnalysis(document,info, result,px);

            SummarizeAnalysis(document,info, result,px);

            AddSign(document,info);

            document.close();

            return true;
        } catch (Exception ex) {
            System.out.println(ex.getMessage());

            return false;
        }
        finally {
            if (document != null && document.isOpen()) {
                document.close();
            }
        }
    }

    public boolean ExportReportLawLess(LawlessInfo info) {
        Document document =null;
        try {

            document = createPdf(info.outFile,info.narrow);

            document.addTitle("违法用地云分析报告");
            Paragraph title = new Paragraph("遥感监测疑似违法用地云分析报告", titleFont);
            title.setAlignment(Element.ALIGN_CENTER);
            document.add(title);

            Paragraph first = new Paragraph("依据国土调查云后台数据，生成的遥感监测疑似违法用地云分析报告如下。受数据齐全性、时效性和精度限制，本分析报告仅用于辅助参考。", normalFont);
            first.setFirstLineIndent(firstLineIndent);
            first.setExtraParagraphSpace(30);
            first.setSpacingBefore(pgSpace);
            first.setLeading(lineSpace);
            document.add(first);

            ResultInfo result = new ResultInfo();
            BaseInfoLawless(document,info, result);

            JcqkAnalysis(document,info, result);

            LegalAnalysisLawless(document,info, result);

            BzInfo(document);

            SummarizeAnalysisLawless(document,info, result);

            AddSign(document,info);

            document.close();

            return true;
        } catch (Exception ex) {
            System.out.println(ex.getMessage());

            return false;
        }
        finally {
            if (document != null && document.isOpen()) {
                document.close();
            }
        }

    }

    public boolean ExportReportYn(YnbhInfo info) throws DocumentException, IOException {
        Document document =null;
        try {

            document = createPdf(info.outFile,info.narrow);



            document.addTitle("永久基本农田补划选址云分析报告");
            Paragraph title = new Paragraph("永久基本农田补划选址云分析报告", titleFont);
            title.setAlignment(Element.ALIGN_CENTER);
            document.add(title);
            Paragraph first = new Paragraph("按照您勾绘的地块范围，依据后台数据，生成的永久基本农田补划选址云分析报告如下。受数据齐全性、时效性和精度限制，本分析报告仅用于辅助参考。", normalFont);
            first.setFirstLineIndent(firstLineIndent);
            first.setExtraParagraphSpace(30);
            first.setSpacingBefore(pgSpace);
            first.setLeading(lineSpace);
            document.add(first);

            Paragraph chapter1Pg = new Paragraph("一、基本情况", chapterFont);
            chapter1Pg.setExtraParagraphSpace(30);
            chapter1Pg.setSpacingBefore(pgSpace);
            chapter1Pg.setLeading(lineSpace);
            document.add(chapter1Pg);

            String content ="地块位于（东经：%s°，北纬：%s°），项目地块面积%s亩。";
            content = String.format(content,info.lontitude,info.latitude, String.valueOf(Round(info.area,2)));
            Paragraph baseInfo = new Paragraph(content, normalFont);
            baseInfo.setFirstLineIndent(firstLineIndent);
            baseInfo.setExtraParagraphSpace(30);
            baseInfo.setSpacingBefore(pgSpace);
            baseInfo.setLeading(lineSpace);
            document.add(baseInfo);

            Paragraph chapter2Pg = new Paragraph("二、合法性分析", chapterFont);
            chapter2Pg.setExtraParagraphSpace(30);
            chapter2Pg.setSpacingBefore(pgSpace);
            chapter2Pg.setLeading(lineSpace);
            document.add(chapter2Pg);

            Paragraph jbntPg = new Paragraph("(一)是否涉及现有永久基本农田情况分析", sectionFont);
            jbntPg.setExtraParagraphSpace(30);
            jbntPg.setSpacingBefore(pgSpace);
            jbntPg.setLeading(lineSpace);
            document.add(jbntPg);
            String jbntResult ="拟选址地块" + (info.jbntArea>0?"涉及":"不涉及")+"永久基本农田。";
            Paragraph jbntInfo = new Paragraph(jbntResult, normalFont);
            jbntInfo.setFirstLineIndent(firstLineIndent);
            jbntInfo.setExtraParagraphSpace(30);
            jbntInfo.setSpacingBefore(pgSpace);
            jbntInfo.setLeading(lineSpace);
            document.add(jbntInfo);


            Paragraph sthxPg = new Paragraph("（二）是否涉及生态保护红线情况分析", sectionFont);
            sthxPg.setExtraParagraphSpace(30);
            sthxPg.setSpacingBefore(pgSpace);
            sthxPg.setLeading(lineSpace);
            document.add(sthxPg);
            String sthxResult ="拟选址地块" + (info.sthxArea>0?"涉及":"不涉及")+"生态保护红线。";
            Paragraph sthxInfo = new Paragraph(sthxResult, normalFont);
            sthxInfo.setFirstLineIndent(firstLineIndent);
            sthxInfo.setExtraParagraphSpace(30);
            sthxInfo.setSpacingBefore(pgSpace);
            sthxInfo.setLeading(lineSpace);
            document.add(sthxInfo);

            Paragraph bwdgdPg = new Paragraph("（三）是否涉及不稳定耕地或非耕地范围情况分析", sectionFont);
            bwdgdPg.setExtraParagraphSpace(30);
            bwdgdPg.setSpacingBefore(pgSpace);
            bwdgdPg.setLeading(lineSpace);
            document.add(bwdgdPg);
            String bwdgdResult ="拟选址地块" + (info.bwdgdArea>0?"涉及":"不涉及")+"不稳定耕地或非耕地范围。";
            Paragraph bwdgdInfo = new Paragraph(bwdgdResult, normalFont);
            bwdgdInfo.setFirstLineIndent(firstLineIndent);
            bwdgdInfo.setExtraParagraphSpace(30);
            bwdgdInfo.setSpacingBefore(pgSpace);
            bwdgdInfo.setLeading(lineSpace);
            document.add(bwdgdInfo);

            Paragraph xctbPg = new Paragraph("（四）是否是狭长图斑情况分析", sectionFont);
            xctbPg.setExtraParagraphSpace(30);
            xctbPg.setSpacingBefore(pgSpace);
            xctbPg.setLeading(lineSpace);
            document.add(xctbPg);
            String xctbResult ="拟选址地块" + (info.dkgzd<0.2?"是":"不是")+"狭长图斑。";
            Paragraph xctbInfo = new Paragraph(xctbResult, normalFont);
            xctbInfo.setFirstLineIndent(firstLineIndent);
            xctbInfo.setExtraParagraphSpace(30);
            xctbInfo.setSpacingBefore(pgSpace);
            xctbInfo.setLeading(lineSpace);
            document.add(xctbInfo);


            Paragraph czcPg = new Paragraph("（五）是否涉及城镇村等用地范围情况分析", sectionFont);
            czcPg.setExtraParagraphSpace(30);
            czcPg.setSpacingBefore(pgSpace);
            czcPg.setLeading(lineSpace);
            document.add(czcPg);
            String czcResult ="拟选址地块" + (info.czcdydArea>0?"涉及":"不涉及")+"城镇村等用地范围。";
            Paragraph czcInfo = new Paragraph(czcResult, normalFont);
            czcInfo.setFirstLineIndent(firstLineIndent);
            czcInfo.setExtraParagraphSpace(30);
            czcInfo.setSpacingBefore(pgSpace);
            czcInfo.setLeading(lineSpace);
            document.add(czcInfo);

            Paragraph nyzPg = new Paragraph("（六）是否涉及农转用范围情况分析", sectionFont);
            nyzPg.setExtraParagraphSpace(30);
            nyzPg.setSpacingBefore(pgSpace);
            nyzPg.setLeading(lineSpace);
            document.add(nyzPg);
            String nzyResult ="拟选址地块" + (info.pewyMj>0?"涉及":"不涉及")+"农转用范围。";
            Paragraph nzyInfo = new Paragraph(nzyResult, normalFont);
            nzyInfo.setFirstLineIndent(firstLineIndent);
            nzyInfo.setExtraParagraphSpace(30);
            nzyInfo.setSpacingBefore(pgSpace);
            nzyInfo.setLeading(lineSpace);
            document.add(nzyInfo);


            Paragraph chapter3Pg = new Paragraph("三、适宜性评价", chapterFont);
            chapter3Pg.setExtraParagraphSpace(30);
            chapter3Pg.setSpacingBefore(pgSpace);
            chapter3Pg.setLeading(lineSpace);
            document.add(chapter3Pg);

            Paragraph tdzlPg = new Paragraph("（一）土地质量条件", sectionFont);
            tdzlPg.setExtraParagraphSpace(30);
            tdzlPg.setSpacingBefore(pgSpace);
            tdzlPg.setLeading(lineSpace);
            document.add(tdzlPg);
            String tdzlResult =String.format("拟选址地块的坡度评分为%s，耕地等别评分为%s，田坎系数评分为%s。",info.pddj.score,info.gddb.score,info.kcxs.score);
            Paragraph tdzlInfo = new Paragraph(tdzlResult, normalFont);
            tdzlInfo.setFirstLineIndent(firstLineIndent);
            tdzlInfo.setExtraParagraphSpace(30);
            tdzlInfo.setSpacingBefore(pgSpace);
            tdzlInfo.setLeading(lineSpace);
            document.add(tdzlInfo);

            Paragraph gdqwPg = new Paragraph("（二）耕地区位条件", sectionFont);
            gdqwPg.setExtraParagraphSpace(30);
            gdqwPg.setSpacingBefore(pgSpace);
            gdqwPg.setLeading(lineSpace);
            document.add(gdqwPg);
            String gdqwResult =String.format("拟选址地块的道路通达度评分为%s，与居民点距离区位条件评分为%s。",info.dlScore,info.jmdScore);
            Paragraph gdqwInfo = new Paragraph(gdqwResult, normalFont);
            gdqwInfo.setFirstLineIndent(firstLineIndent);
            gdqwInfo.setExtraParagraphSpace(30);
            gdqwInfo.setSpacingBefore(pgSpace);
            gdqwInfo.setLeading(lineSpace);
            document.add(gdqwInfo);

            Paragraph dkkjxtPg = new Paragraph("（三）地块空间形态", sectionFont);
            dkkjxtPg.setExtraParagraphSpace(30);
            dkkjxtPg.setSpacingBefore(pgSpace);
            dkkjxtPg.setLeading(lineSpace);
            document.add(dkkjxtPg);
            String dkkjxtResult =String.format("拟选址地块与邻近的永久基本农田的集中连片度评分为%s。",info.jzlpScore);
            Paragraph dkkjxtInfo = new Paragraph(dkkjxtResult, normalFont);
            dkkjxtInfo.setFirstLineIndent(firstLineIndent);
            dkkjxtInfo.setExtraParagraphSpace(30);
            dkkjxtInfo.setSpacingBefore(pgSpace);
            dkkjxtInfo.setLeading(lineSpace);
            document.add(dkkjxtInfo);

            Paragraph chapter4Pg = new Paragraph("三、总体分析", chapterFont);
            chapter4Pg.setExtraParagraphSpace(30);
            chapter4Pg.setSpacingBefore(pgSpace);
            chapter4Pg.setLeading(lineSpace);
            document.add(chapter4Pg);

            double allScore = 0.25*info.pddj.score+0.21*info.gddb.score+0.12*info.kcxs.score+0.16*info.dlScore+0.1*info.jmdScore+0.16*info.jzlpScore;
            boolean wf=false;
            if(info.jbntArea>0)
            {
                wf=true;
            }
            if(info.sthxArea>0)
            {
                wf=true;
            }
            if(info.bwdgdArea>0)
            {
                wf=true;
            }
            if(info.dkgzd<0.2)
            {
                wf=true;
            }
            if(info.czcdydArea>0)
            {
                wf=true;
            }
            if(info.pewyMj>0)
            {
                wf=true;
            }

            String zzpfResult =String.format("根据已有数据分析结果，拟选址地块最终评分为%s。",(wf?0:allScore));
            Paragraph zzpfInfo = new Paragraph(zzpfResult, normalFont);
            zzpfInfo.setFirstLineIndent(firstLineIndent);
            zzpfInfo.setExtraParagraphSpace(30);
            zzpfInfo.setSpacingBefore(pgSpace);
            zzpfInfo.setLeading(lineSpace);
            document.add(zzpfInfo);

            AddSign(document,info);

            document.close();

            return true;
        } catch (Exception ex) {
            System.out.println(ex.getMessage());

            return false;
        }
        finally {
            if (document != null && document.isOpen()) {
                document.close();
            }
        }
    }

    private void Parse(SqlliteConnTool db, String tableName, ProjectInfo info) {
        try {
            String tn = tableName.toLowerCase();
            System.out.println("table:" + tn);
            switch (tn) {
                case "landtype_analyze":
                    ParseLandType(db, tableName, info);
                    break;
                case "landtype_analyze_image":
                    ParseLandTypeImage(db, tableName, info);
                    break;
                case "ownership_analyze":
                    ParseOwnerShip(db, tableName, info);
                    break;
                case "ownership_analyze_image":
                    ParseOwnerShipImage(db, tableName, info);
                    break;
                case "primefarm_analyze":
                    ParsePrimeFarm(db, tableName, info);
                    break;
                case "primefarm_analyze_image":
                    ParsePrimeFarmImage(db, tableName, info);
                    break;
                case "plan_analyze":
                    ParseTdgh(db, tableName, info);
                    break;
                case "plan_analyze_image":
                    ParseTdghImage(db, tableName, info);
                    break;
                case "spba_analyze":
                    ParseSpba(db, tableName, info);
                    break;
                case "spba_analyze_image":
                    ParseSpbaImage(db, tableName, info);
                    break;
                case "naturereserve_analyze":
                    ParseZrbhq(db, tableName, info);
                    break;
                case "naturereserve_analyze_image":
                    ParseZrbhqImage(db, tableName, info);
                    break;
                case "dzfx_analyze":
                    ParseDzfx(db, tableName, info);
                    break;
                case "dzfx_analyze_image":
                    ParseDzfxImage(db, tableName, info);
                    break;
                case "ecologicalland_analyze":
                    ParseEcologicalLand(db, tableName, info);
                    break;
                case "ecologicalland_analyze_image":
                    ParseEcologicalLandImage(db, tableName, info);
                    break;
                case "landprice_analyze":
                    ParseLandPrice(db, tableName, info);
                    break;
                case "landprice_analyze_image":
                    ParseLandPriceImage(db, tableName, info);
                    break;
                case "aggregate_analyze":
                    ParseLandUseGrade(db, tableName, info);
                    break;
                case "aggregate_analyze_image":
                    ParseLandUseGradeImage(db, tableName, info);
                    break;
                case "image_analyze":
                    info.ownershipImgMeta = GetImageMeta(db,
                            "Image_Analyze");
                    break;
                case "pxstbhhx_analyze":
                    ParsePxStbhhx(db, tableName, info);
                    break;
                case "pxstbhhx_analyze_image":
                    ParsePxStbhhxImage(db, tableName, info);
                    break;
                case "pxgyl_analyze":
                    ParsePxGyl(db, tableName, info);
                    break;
                case "pxgyl_analyze_image":
                    ParsePxGylImage(db, tableName, info);
                    break;
                case "pxpd_analyze":
                    ParsePxPd(db, tableName, info);
                    break;
                case "pxpd_analyze_image":
                    ParsePxPdImage(db, tableName, info);
                    break;
            }
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseLawless(SqlliteConnTool db, String tableName, LawlessInfo info) {
        try {
            String tn = tableName.toLowerCase();
            System.out.println("table:" + tn);
            switch (tn) {
                case "landtype_analyze":
                    ParseLandTypeLawless(db, tableName, info);
                    break;
                case "landtype_analyze_image":
                    ParseLandTypeImageLawless(db, tableName, info);
                    break;
                case "ownership_analyze":
                    ParseOwnerShipLawless(db, tableName, info);
                    break;
                case "ownership_analyze_image":
                    ParseOwnerShipImageLawless(db, tableName, info);
                    break;
                case "primefarm_analyze":
                    ParsePrimeFarmLawless(db, tableName, info);
                    break;
                case "primefarm_analyze_image":
                    ParsePrimeFarmImageLawless(db, tableName, info);
                    break;
                case "plan_analyze":
                    ParseTdghLawless(db, tableName, info);
                    break;
                case "plan_analyze_image":
                    ParseTdghImageLawless(db, tableName, info);
                    break;
                case "spba_analyze":
                    ParseSpbaLawless(db, tableName, info);
                    break;
                case "spba_analyze_image":
                    ParseSpbaImageLawless(db, tableName, info);
                    break;
                case "naturereserve_analyze":
                    ParseZrbhqLawless(db, tableName, info);
                    break;
                case "naturereserve_analyze_image":
                    ParseZrbhqImageLawless(db, tableName, info);
                    break;
                case "image_analyze":
                    info.ownershipImgMeta = GetImageMeta(db,
                            "Image_Analyze");
                    ParseLawlessImgCompare(db, tableName, info);
                    break;
                case "jctb_analyze":
                    ParseJctb(db, tableName, info);
                    break;
            }
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseYnbh(SqlliteConnTool db, String tableName, YnbhInfo info) {
        try {
            String tn = tableName.toLowerCase();
            System.out.println("table:" + tn);
            switch (tn) {
                //基本农田
                case "primefarm_analyze":
                    ParsePrimeFarmYn(db, tableName, info);
                    break;
                //生态红线
                case "sthx_analyze":
                    ParseSthxYn(db, tableName, info);
                    break;
                //不稳定耕地
                case "bwdgd_analyze":
                    ParseBwdgdYn(db, tableName, info);
                    break;
                //城镇村类型
                case "sdczcdyd_analyze":
                    ParseCzcdydYn(db, tableName, info);
                    break;
                //河湖岸线
                case "hhax_analyze":
                    ParseHhaxYn(db, tableName, info);
                    break;
                //批而未用
                case "pewy_analyze":
                    ParsePewyYn(db, tableName, info);
                    break;
                //设施农用地
                case "ssnyd_analyze":
                    ParseSsnydYn(db, tableName, info);
                    break;
                //坡度等级
                case "pddjavg_analyze":
                    ParsePddjYn(db, tableName, info);
                    break;
                //耕地等别
                case "gddbavg_analyze":
                    ParseGddbYn(db, tableName, info);
                    break;
                //扣除系数
                case "kcxsavg_analyze":
                    ParseKcxsYn(db, tableName, info);
                    break;
                //道路通达度
                case "dlbuffer_analyze":
                    ParseDlYn(db, tableName, info);
                    break;
                //临近居民点
                case "jmdbuffer_analyze":
                    ParseJmdYn(db, tableName, info);
                    break;
                //集中连片
                case "jbntbuffer_analyze":
                    ParseJzlpYn(db, tableName, info);
                    break;
            }
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParsePrimeFarmYn(SqlliteConnTool db, String tableName,
                                  YnbhInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                if(set.findColumn("type")>=0)
                {
                    String type = set.getString("type");
                    if ("基本农田".equals(type)) {
                        double mj = set.getDouble("mj");
                        area += mj;
                    }
                }
                else
                {
                    double mj = set.getDouble("mj");
                    area += mj;
                }

            }
            info.jbntArea = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }
    private void ParseSthxYn(SqlliteConnTool db, String tableName,
                             YnbhInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                double mj = set.getDouble("mj");
                area += mj;

            }
            info.sthxArea = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseBwdgdYn(SqlliteConnTool db, String tableName,
                              YnbhInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            List<String>dlbms = new ArrayList<>();
            dlbms.add("010");
            dlbms.add("011");
            dlbms.add("012");
            dlbms.add("013");
            dlbms.add("0100");
            dlbms.add("0101");
            dlbms.add("0102");
            dlbms.add("0103");
            List<String>tbxhdms = new ArrayList<>();
            tbxhdms.add("LQGD");
            tbxhdms.add("MQGD");
            tbxhdms.add("SMGD");
            tbxhdms.add("SHGD");
            tbxhdms.add("HDGD");

            ResultSet set = result.getResultSet();
            while (set.next()) {
                String dlbm = set.getString("dlbm");
                String tbxhdm = set.getString("tbxhdm");
                String pddj = set.getString("gdpddj");
                boolean iswdgd=true;
                if(StringUtils.isBlank(dlbm)||!dlbms.contains(dlbm))
                {
                    iswdgd=false;
                }
                else if(StringUtils.isBlank(tbxhdm)||tbxhdm.contains(tbxhdm.toUpperCase()))
                {
                    iswdgd=false;
                }
                else if(StringUtils.isNotBlank(pddj)&&pddj.equals(5))
                {
                    iswdgd=false;
                }
                if(!iswdgd) {
                    double mj = set.getDouble("mj");
                    area += mj;
                }

            }
            info.bwdgdArea = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseCzcdydYn(SqlliteConnTool db, String tableName,
                               YnbhInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            List<String>czclxs = new ArrayList<>();
            czclxs.add("201");
            czclxs.add("202");
            czclxs.add("203");

            ResultSet set = result.getResultSet();
            while (set.next()) {
                String czclx = set.getString("czclx");
                if(StringUtils.isNotBlank(czclx)&&czclxs.contains(czclx))
                {
                    double mj = set.getDouble("mj");
                    area += mj;
                }
            }
            info.czcdydArea = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }


    private void ParseHhaxYn(SqlliteConnTool db, String tableName,
                             YnbhInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                double mj = set.getDouble("mj");
                area += mj;

            }
            info.hhaxMj = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }


    private void ParsePewyYn(SqlliteConnTool db, String tableName,
                             YnbhInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                double mj = set.getDouble("mj");
                area += mj;

            }
            info.pewyMj = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }


    private void ParseSsnydYn(SqlliteConnTool db, String tableName,
                              YnbhInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                double mj = set.getDouble("mj");
                area += mj;

            }
            info.ssnydMj = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParsePddjYn(SqlliteConnTool db, String tableName,
                             YnbhInfo info) {

        try {
            //D1=（全省最大坡度-该地块坡度）÷最大坡度×100
            //（平地+梯田）占地块的面积比重*100+坡耕地占地块面积比重*D1=总得分
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double mj=0;
            double pgdMj=0;
            double pdttMj=0;
            double maxPdjb=0;
            double pgdPdjb=0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                mj = set.getDouble("mj");
                pgdMj = set.getDouble("pgd_area");
                pdttMj = set.getDouble("pdtt_area");
                maxPdjb = set.getDouble("max_pdjb");
                pgdPdjb = set.getDouble("pgd_pdjb");
                break;
            }
            double pddj = pdttMj/mj*100+pgdMj/mj*((maxPdjb - pgdPdjb)/maxPdjb*100);
            java.text.DecimalFormat df = new java.text.DecimalFormat("#.00");

            info.pddj = new YnbhInfo.Pddj();
            info.pddj.area = mj;
            info.pddj.maxPdjb = maxPdjb;
            info.pddj.pgdPdjb=pgdPdjb;
            info.pddj.pdttArea = pdttMj;
            info.pddj.pgdArea = pgdMj;
            info.pddj.score =Double.parseDouble(df.format(pddj));
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseGddbYn(SqlliteConnTool db, String tableName,
                             YnbhInfo info) {

        try {
            //D1=（全省最大坡度-该地块坡度）÷最大坡度×100
            //（平地+梯田）占地块的面积比重*100+坡耕地占地块面积比重*D1=总得分
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double mj=0;
            double maxGddb=0;
            double avgGddb=0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                mj = set.getDouble("mj");
                maxGddb = set.getDouble("max_gddb");
                avgGddb = set.getDouble("avg_gddb");
                break;
            }
            double gddb =avgGddb;// (maxGddb - avgGddb)/maxGddb*100;
            //java.text.DecimalFormat df = new java.text.DecimalFormat("#.00");
            //Double score = Double.parseDouble(df.format(gddb));
            Double score = 0d;
            if(gddb>=1&&gddb<=4)
            {
                score=100d;
            }
            else if(gddb>4&&gddb<=8)
            {
                score=80d;
            }
            else if(gddb>8&&gddb<=12)
            {
                score=60d;
            }
            else
            {
                score=30d;
            }
            info.gddb = new YnbhInfo.Gddb();
            info.gddb.area = mj;
            info.gddb.maxGddb = maxGddb;
            info.gddb.avgGddb = avgGddb;
            info.gddb.score =score;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseKcxsYn(SqlliteConnTool db, String tableName,
                             YnbhInfo info) {

        try {
            //D1=（全省最大坡度-该地块坡度）÷最大坡度×100
            //（平地+梯田）占地块的面积比重*100+坡耕地占地块面积比重*D1=总得分
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double mj=0;
            double maxKcxs=0;
            double avgKcxs=0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                mj = set.getDouble("mj");
                maxKcxs = set.getDouble("max_kcxs");
                avgKcxs = set.getDouble("avg_kcxs");
                break;
            }
            double kcxs = (maxKcxs - avgKcxs)/maxKcxs*100;
            java.text.DecimalFormat df = new java.text.DecimalFormat("#.00");

            info.kcxs = new YnbhInfo.Kcxs();
            info.kcxs.area = mj;
            info.kcxs.maxKcxs = maxKcxs;
            info.kcxs.avgKcxs = avgKcxs;
            info.kcxs.score =Double.parseDouble(df.format(kcxs));
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseDlYn(SqlliteConnTool db, String tableName,
                           YnbhInfo info) {

        try {
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double buffer=0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                buffer = set.getDouble("buffer");

                break;
            }
            info.dlScore=0;
            if(buffer==100) {
                info.dlScore = 100;
            }
            else if(buffer==500)
            {
                info.dlScore=80;
            }
            else if(buffer==1000)
            {
                info.dlScore=40;
            }

            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseJmdYn(SqlliteConnTool db, String tableName,
                            YnbhInfo info) {

        try {
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double buffer=0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                buffer = set.getDouble("buffer");

                break;
            }
            info.jmdScore=0;
            if(buffer==100) {
                info.jmdScore = 100;
            }
            else if(buffer==500)
            {
                info.jmdScore=80;
            }
            else if(buffer==1000)
            {
                info.jmdScore=40;
            }

            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseJzlpYn(SqlliteConnTool db, String tableName,
                             YnbhInfo info) {

        try {
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double buffer=0;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                buffer = set.getDouble("buffer");

                break;
            }
            info.jzlpScore=0;
            if(buffer==100) {
                info.jzlpScore = 100;
            }
            else if(buffer==500)
            {
                info.jzlpScore=80;
            }
            else if(buffer==1000)
            {
                info.jzlpScore=40;
            }

            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }
    private void ParseLandType(SqlliteConnTool db, String tableName,
                               ProjectInfo info) throws SQLException {
        try {
            if (info.landTypes == null) {
                info.landTypes = new ArrayList<LandType>();
            }
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                LandType land = new LandType();
                land.dlmc = set.getString("name");
                land.dlbm = set.getString("code");
                land.area = set.getDouble("mj");
                info.landTypes.add(land);
            }
            Collections.sort(info.landTypes);
            Collections.reverse(info.landTypes);
            int count = 0;
            int leng = info.landTypes.size();
            String dlInfo = "";
            double otherArea = 0;
            for (LandType item : info.landTypes) {
                count++;
                if (count > 2) {
                    otherArea += item.area;
                } else {
                    dlInfo += item.dlmc + Round(item.area, 2) + "亩";
                    if (count < leng) {
                        dlInfo += "/";
                    }
                }

            }
            if (otherArea > 0) {
                dlInfo += ("其他" + Round(otherArea, 2) + "亩");
            }
            info.landtypeImgMeta = dlInfo;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseLandTypeImage(SqlliteConnTool db, String tableName,
                                    ProjectInfo info) throws SQLException {
        Image img = GetImageByBlob(db, tableName, 0);
        info.landtypeImg = img;
    }

    private void ParseLandTypeLawless(SqlliteConnTool db, String tableName,
                                      LawlessInfo info) throws SQLException {
        try {
            if (info.landTypes == null) {
                info.landTypes = new ArrayList<LandType>();
            }
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                LandType land = new LandType();
                land.dlmc = set.getString("name");
                land.dlbm = set.getString("code");
                land.area = set.getDouble("mj");
                info.landTypes.add(land);
            }
            Collections.sort(info.landTypes);
            Collections.reverse(info.landTypes);
            int count = 0;
            int leng = info.landTypes.size();
            String dlInfo = "";
            double otherArea = 0;
            for (LandType item : info.landTypes) {
                count++;
                if (count > 2) {
                    otherArea += item.area;
                } else {
                    dlInfo += item.dlmc + Round(item.area, 2) + "亩";
                    if (count < leng) {
                        dlInfo += "/";
                    }
                }

            }
            if (otherArea > 0) {
                dlInfo += ("其他" + Round(otherArea, 2) + "亩");
            }
            info.landtypeImgMeta = dlInfo;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseLandTypeImageLawless(SqlliteConnTool db, String tableName,
                                           LawlessInfo info) throws SQLException {
        Image img = GetImageByBlob(db, tableName, 0);
        info.landtypeImg = img;
    }

    private void ParseOwnerShip(SqlliteConnTool db, String tableName,
                                ProjectInfo info) {
        try {
            if (info.ownerships == null) {
                info.ownerships = new ArrayList<Ownership>();
            }
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                Ownership owner = new Ownership();
                String code = set.getString("owner_dept_name");
                String name = set.getString("owner_dept_code");
                String jsonStr = set.getString("area_detail");
                double area = set.getDouble("mj");

                JSONObject json = JSONObject.parseObject(jsonStr);
                OwnerDetailInfo detail = JSON.toJavaObject(json,
                        OwnerDetailInfo.class);
                owner.qsmc = name;
                owner.qsdm = code;
                owner.area = area;
                owner.detail = detail;
                info.ownerships.add(owner);
            }
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseOwnerShipImage(SqlliteConnTool db, String tableName,
                                     ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 1);
        info.ownershipImg = img;
    }

    private void ParseOwnerShipLawless(SqlliteConnTool db, String tableName,
                                       LawlessInfo info) {
        try {
            if (info.ownerships == null) {
                info.ownerships = new ArrayList<Ownership>();
            }
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                Ownership owner = new Ownership();
                String code = set.getString("owner_dept_name");
                String name = set.getString("owner_dept_code");
                String jsonStr = set.getString("area_detail");
                double area = set.getDouble("mj");

                JSONObject json = JSONObject.parseObject(jsonStr);
                OwnerDetailInfo detail = JSON.toJavaObject(json,
                        OwnerDetailInfo.class);
                owner.qsmc = name;
                owner.qsdm = code;
                owner.area = area;
                owner.detail = detail;
                info.ownerships.add(owner);
            }
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseOwnerShipImageLawless(SqlliteConnTool db, String tableName,
                                            LawlessInfo info) {
        Image img = GetImageByBlob(db, tableName, 1);
        info.ownershipImg = img;
    }

    private void ParsePrimeFarm(SqlliteConnTool db, String tableName,
                                ProjectInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                String type = set.getString("type");
                if ("基本农田".equals(type)) {
                    double mj = set.getDouble("mj");
                    area += mj;
                }
            }
            info.jbntArea = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParsePrimeFarmImage(SqlliteConnTool db, String tableName,
                                     ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.jbntImg = img;

    }

    private void ParsePrimeFarmLawless(SqlliteConnTool db, String tableName,
                                       LawlessInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                String type = set.getString("type");
                if ("基本农田".equals(type)) {
                    double mj = set.getDouble("mj");
                    area += mj;
                }
            }
            info.jbntArea = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParsePrimeFarmImageLawless(SqlliteConnTool db, String tableName,
                                            LawlessInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.jbntImg = img;

    }


    private void ParseTdgh(SqlliteConnTool db, String tableName,
                           ProjectInfo info) {
        try {

            if (info.tdghInfos == null) {
                info.tdghInfos = new ArrayList<TdghInfo>();
            }
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                TdghInfo tdgh = new TdghInfo();
                tdgh.jsbm = set.getString("code");
                tdgh.jsmc = set.getString("type");
                tdgh.area = set.getDouble("mj");
                info.tdghInfos.add(tdgh);
            }

            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseTdghImage(SqlliteConnTool db, String tableName,
                                ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.tdghImg = img;
    }

    private void ParseTdghLawless(SqlliteConnTool db, String tableName,
                                  LawlessInfo info) {
        try {

            if (info.tdghInfos == null) {
                info.tdghInfos = new ArrayList<TdghInfo>();
            }
            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            while (set.next()) {

                TdghInfo tdgh = new TdghInfo();
                tdgh.jsbm = set.getString("code");
                tdgh.jsmc = set.getString("type");
                tdgh.area = set.getDouble("mj");
                info.tdghInfos.add(tdgh);
            }

            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseTdghImageLawless(SqlliteConnTool db, String tableName,
                                       LawlessInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.tdghImg = img;
    }

    private void ParseSpba(SqlliteConnTool db, String tableName,
                           ProjectInfo info) {
        try {

            String sql = "select license_number, sum(approval_mj) approval_mj from " + tableName + " group by license_number";
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            ArrayList<SpbaInfo> spbaArr = new ArrayList<SpbaInfo>();

            while (set.next()) {
                SpbaInfo spba = new SpbaInfo();
                spba.area = set.getDouble("approval_mj");
                String license = set.getString("license_number") + "";
                if (license == null || license.isEmpty()) {
                    license = "未知";
                }
                spba.license = license;
                spbaArr.add(spba);
            }
            info.spbaInfos = spbaArr;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseSpbaImage(SqlliteConnTool db, String tableName,
                                ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.spbaImg = img;

    }

    private void ParseSpbaLawless(SqlliteConnTool db, String tableName,
                                  LawlessInfo info) {
        try {

            String sql = "select license_number, sum(approval_mj) approval_mj from " + tableName + " group by license_number";
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            ArrayList<SpbaInfo> spbaArr = new ArrayList<SpbaInfo>();

            while (set.next()) {
                SpbaInfo spba = new SpbaInfo();
                spba.area = set.getDouble("approval_mj");
                String license = set.getString("license_number") + "";
                if (license == null || license.isEmpty()) {
                    license = "未知";
                }
                spba.license = license;
                spbaArr.add(spba);
            }
            info.spbaInfos = spbaArr;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseSpbaImageLawless(SqlliteConnTool db, String tableName,
                                       LawlessInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.spbaImg = img;

    }

    private void ParseZrbhq(SqlliteConnTool db, String tableName,
                            ProjectInfo info) {
        try {

            String sql = "select sum(mj) from " + tableName;
            Object obj = db.queryScale(sql);
            if (obj != null) {
                double area = Double.valueOf(obj.toString());
                info.zrbhqArea = area;
            }
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseZrbhqImage(SqlliteConnTool db, String tableName,
                                 ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.zrbhqImg = img;

    }

    private void ParseZrbhqLawless(SqlliteConnTool db, String tableName,
                                   LawlessInfo info) {
        try {

            String sql = "select sum(mj) from " + tableName;
            Object obj = db.queryScale(sql);
            if (obj != null) {
                double area = Double.valueOf(obj.toString());
                info.zrbhqArea = area;
            }
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseZrbhqImageLawless(SqlliteConnTool db, String tableName,
                                        LawlessInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.zrbhqImg = img;

    }

    private void ParseDzfx(SqlliteConnTool db, String tableName,
                           ProjectInfo info) {
        try {

            String sql = "select level,sum(area) area from " + tableName + " group by level";
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            ArrayList<DzfxInfo> dzfxArr = new ArrayList<DzfxInfo>();

            while (set.next()) {
                DzfxInfo dzfx = new DzfxInfo();
                dzfx.area = set.getDouble("area");
                String level = set.getString("level") + "";
                if (level == null || level.isEmpty()) {
                    level = "未知";
                }
                dzfx.level = level;
                dzfxArr.add(dzfx);
            }
            info.dzfxInfos = dzfxArr;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseDzfxImage(SqlliteConnTool db, String tableName,
                                ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.dzfxImg = img;

    }

    private void ParseEcologicalLand(SqlliteConnTool db, String tableName,
                                     ProjectInfo info) {
        try {

            String sql = "select sum(area) from" + tableName;
            Object obj = db.queryScale(sql);
            if (obj != null) {
                double area = Double.valueOf(obj.toString());
                info.sdArea = area;
            }
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseEcologicalLandImage(SqlliteConnTool db, String tableName,
                                          ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.sdImg = img;

    }

    private void ParseLandPrice(SqlliteConnTool db, String tableName,
                                ProjectInfo info) {
        try {

            String sql = "select sum(area) mj,sum(price) price from" + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            double mj = 0;
            double price = 0;
            while (set.next()) {
                mj = set.getDouble("mj");
                price = set.getDouble("price");

            }
            info.landPrice = mj == 0 ? 0 : price / mj;
            result.Close();

        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }

    }

    private void ParseLandPriceImage(SqlliteConnTool db, String tableName,
                                     ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.landPriceImg = img;

    }

    private void ParseLandUseGrade(SqlliteConnTool db, String tableName, ProjectInfo info) {
        try {
            String sql = "select content from " + tableName + " where analyze_name ='土地利用等别分析'";

            Object obj = db.queryScale(sql);
            if (obj == null) {
                return;
            }
            String jsonStr = obj.toString();
            List<LandUseGrade> detail = JSONArray.parseArray(jsonStr, LandUseGrade.class);
            if (detail != null) {
                for (LandUseGrade lg : detail) {
                    lg.field1 = lg.field1.replace(",", "-");
                }
            }
            info.gradeInfos = (ArrayList<LandUseGrade>) detail;

        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParseLandUseGradeImage(SqlliteConnTool db, String tableName,
                                        ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, "土地利用等别分析", 0);
        info.gradeImg = img;

    }
    private void ParseJctb(SqlliteConnTool db, String tableName,
                           LawlessInfo info) {

        try {

            String sql = "select tblx,sum(mj) as mj from " + tableName+" group by tblx" ;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            info.jctbMap = new HashMap<String, Double>();
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                String type = set.getString("tblx");
                double mj = set.getDouble("mj");
                info.jctbMap.put(type,mj);
            }
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParsePxStbhhx(SqlliteConnTool db, String tableName,
                               ProjectInfo info) {

        try {

            String sql = "select * from " + tableName;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                String type = set.getString("type");
                if ("生态保护红线".equals(type)) {
                    double mj = set.getDouble("mj");
                    area += mj;
                }
            }
            info.px_stbhhx_area = area;
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParsePxStbhhxImage(SqlliteConnTool db, String tableName,
                                    ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.px_stbhhx_Img = img;

    }

    private void ParsePxGyl(SqlliteConnTool db, String tableName,
                            ProjectInfo info) {

        try {

            String sql = "select sum(mj) as mj from " + tableName;
            Object obj= db.queryScale(sql);
            if(obj!=null)
            {
                double area = 0;
                area = Double.parseDouble(obj.toString());
                info.px_gyl_area  = area;
            }

        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParsePxGylImage(SqlliteConnTool db, String tableName,
                                 ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.px_gyl_img = img;

    }

    private void ParsePxPd(SqlliteConnTool db, String tableName,
                           ProjectInfo info) {

        try {

            String sql = "select pdjb,sum(mj) as mj from " + tableName+" group by pdjb" ;
            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            info.px_pd = new HashMap<String, Double>();
            double area = 0;
            ResultSet set = result.getResultSet();
            while (set.next()) {
                String type = set.getString("pdjb");
                double mj = set.getDouble("mj");
                info.px_pd.put(type,mj);
            }
            result.Close();
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
        }
    }

    private void ParsePxPdImage(SqlliteConnTool db, String tableName,
                                ProjectInfo info) {
        Image img = GetImageByBlob(db, tableName, 0);
        info.px_pd_img = img;

    }

    private Image GetImageByBlob(SqlliteConnTool db, String tableName,
                                 int imageType) {
        try {
            Image img = null;
            String sql = "select layer_pic from " + tableName
                    + " where image_type = " + imageType;
            ResultSetInfo info = db.doQuery(sql);
            if (info == null)
                return null;
            ResultSet set = info.getResultSet();
            while (set.next()) {

                byte[] imgBytes = set.getBytes("layer_pic");
                if (imgBytes != null) {
                    img = Image.getInstance(imgBytes);
                }
                break;
            }
            info.Close();
            return img;
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
            return null;
        }
    }

    private Image GetImageByBlob(SqlliteConnTool db, String tableName,String analyzeName,
                                 int imageType) {
        try {
            Image img = null;
            String sql = "select layer_pic from " + tableName
                    + " where analyze_name = '" + analyzeName +"' and image_subtype = " + imageType;
            ResultSetInfo info = db.doQuery(sql);
            if (info == null)
                return null;
            ResultSet set = info.getResultSet();
            while (set.next()) {

                byte[] imgBytes = set.getBytes("layer_pic");
                if (imgBytes != null) {
                    img = Image.getInstance(imgBytes);
                }
                break;
            }
            info.Close();
            return img;
        } catch (Exception ex) {
            System.out.println(ex.getMessage());
            return null;
        }
    }

    private String GetImageMeta(SqlliteConnTool db, String tableName) {
        String sql = "select meta_info from Image_Analyze where date is not null order by date desc limit 1";
        Object obj = db.queryScale(sql);
        if (obj == null)
            return null;
        String sjy = "";
        String sx = "";
        String jsonStr = obj.toString();
        JSONObject json = JSONObject.parseObject(jsonStr);
        if (json.containsKey("SJY")) {
            sjy = json.getString("SJY");
        }
        if (json.containsKey("SX")) {
            sx = json.getString("SX");
        }
        if (sjy == null || sjy.isEmpty()) {
            sjy = "未知";
        }
        if (sx == null || sx.isEmpty()) {
            sx = "未知";
        }
        String meta = "数据源：" + sjy + " 时相：" + sx;
        return meta;
    }

    private void ParseLawlessImgCompare(SqlliteConnTool db,String tableName,LawlessInfo info)
    {
        String sql  = "select * from Image_Analyze where date is not null order by date  desc limit 2";
        try {

            ResultSetInfo result = db.doQuery(sql);
            if (result == null)
                return;
            ResultSet set = result.getResultSet();
            int count = 0;
            while (set.next()) {
                count ++;
                Image img = null;
                String meta="";
                String year="";
                byte[] imgBytes = set.getBytes("layer_pic");
                if (imgBytes != null) {
                    img= Image.getInstance(imgBytes);
                }
                String date= set.getString("date");
                if(date!=null&&!date.isEmpty()&&date.length()>=4)
                {
                    year =date.substring(0,4);
                }
                String sjy = "";
                String sx = "";
                String jsonStr = set.getString("meta_info");
                JSONObject json = JSONObject.parseObject(jsonStr);
                if (json.containsKey("SJY")) {
                    sjy = json.getString("SJY");
                }
                if (json.containsKey("SX")) {
                    sx = json.getString("SX");
                }
                if (sjy == null || sjy.isEmpty()) {
                    sjy = "未知";
                }
                if (sx == null || sx.isEmpty()) {
                    sx = "未知";
                }
                meta = sjy + " " + sx;
                if(count==1)
                {
                    info.year2 = year;
                    info.image2 = img;
                    info.imageMeta2 = meta;
                }
                else
                {
                    info.year1 = year;
                    info.image1= img;
                    info.imageMeta1 = meta;
                }


            }
            result.Close();

        } catch (Exception ex) {
            System.out.println(ex.getMessage());

        }
    }

    private double Round(double value, int count) {
        String formate = "#.";
        for (int i = 0; i < count; i++) {
            formate += "0";
        }
        String temp = new java.text.DecimalFormat(formate).format(value);
        return Double.parseDouble(temp);

    }


    private void BaseInfo(Document document,ProjectInfo info,ResultInfo result) throws DocumentException, IOException {
        Paragraph chapter1Pg = new Paragraph("一、基本情况", chapterFont);
        chapter1Pg.setExtraParagraphSpace(30);
        chapter1Pg.setSpacingBefore(pgSpace);
        chapter1Pg.setLeading(lineSpace);
        document.add(chapter1Pg);

        String content ="%s项目位于%s%s%s%s（东经：%s°，北纬：%s°），项目地块面积%s亩。";
        content = String.format(content,info.projectName,info.provinceName,info.cityName,info.countyName,info.villageName,info.lontitude,info.latitude, String.valueOf(Round(info.area,2)));
        String dlinfo="";
        int lengLy=0;
        int lengOw=0;
        if(info.landTypes!=null&&info.landTypes.size()>0) {
            lengLy = info.landTypes.size();
        }
        if(info.ownerships!=null&&info.ownerships.size()>0) {
            lengOw = info.ownerships.size();
        }
        if(lengLy>0){
            dlinfo="土地利用现状：";
            double otherArea = 0;
            int count = 0;
            for (LandType item : info.landTypes) {
                count++;
                if(count>2)
                {
                    otherArea+=item.area;
                }
                else
                {
                    dlinfo += item.dlmc + Round(item.area, 2) + "亩";
                    if (count < lengLy) {
                        dlinfo += "、";
                    }
                }

            }
            if(otherArea>0)
            {
                dlinfo += ("其他" + Round(otherArea, 2) + "亩");
            }
            if(lengOw>0)
            {
                dlinfo+="；";
            }
            else
            {
                dlinfo+="。";
            }
            /*
            int count = 0;
            for(LandType item : info.landTypes)
            {
                count++;
                dlinfo+=item.dlmc+Round(item.area,2)+"亩";
                if(count<lengLy)
                {
                    dlinfo+="、";
                }
                else
                {
                    if(lengOw>0)
                    {
                        dlinfo+=";";
                    }
                    else
                    {
                        dlinfo+="。";
                    }
                }
            }
            */

        }
        if(lengOw>0)
        {
            dlinfo+="土地权属为";
            HashMap<String,Double> dic = new HashMap<String,Double>();
            dic.put("国有",0.0);
            dic.put("集体",0.0);
            dic.put("其他",0.0);
            for(Ownership item : info.ownerships)
            {

                if(item.detail!=null)
                {
                    dic.put("国有",dic.get("国有")+item.detail.lx10);
                    dic.put("国有",dic.get("国有")+item.detail.lx20);
                    dic.put("集体",dic.get("集体")+item.detail.lx30);
                    dic.put("集体",dic.get("集体")+item.detail.lx40);
                    dic.put("其他",dic.get("其他")+item.detail.lx99);

                }
            }
            if(Round(dic.get("国有"),2)>0) {
                dlinfo += "国有" + Round(dic.get("国有"), 2) + "亩、";
            }
            if(Round(dic.get("集体"),2)>0) {
                dlinfo += "集体" + Round(dic.get("集体"), 2) + "亩、";
            }
            if(Round(dic.get("其他"),2)>0) {
                dlinfo += "其他" + Round(dic.get("其他"), 2) + "亩、";
            }
            if(dlinfo.endsWith("、")) {
                dlinfo = dlinfo.substring(0, dlinfo.length() - 1) + "。";
            }
        }

        content+=dlinfo;
        Paragraph baseInfo = new Paragraph(content, normalFont);
        baseInfo.setFirstLineIndent(firstLineIndent);
        baseInfo.setExtraParagraphSpace(30);
        baseInfo.setSpacingBefore(pgSpace);
        baseInfo.setLeading(lineSpace);
        document.add(baseInfo);

        //document.newPage();
        AddBaseImg(document,info);
        /*
        int col=2;
        if(info.narrow)
        {
            col = 1;
        }
        PdfPTable t = new PdfPTable(col);//创建一个2列表格对象
        t.setSpacingBefore(pgSpace);//设置上边距
        t.setSpacingAfter(pgSpace);//设置下边距
        t.setWidthPercentage(100);
        if(info.narrow)
        {
            t.setWidthPercentage(80);
        }



            PdfPCell c1 = null;
            c1 =new PdfPCell(new Phrase("项目区位置", normalFont));
            c1.setVerticalAlignment(Element.ALIGN_CENTER);
            c1.setHorizontalAlignment(Element.ALIGN_CENTER);
            c1.setPaddingTop(5);
            c1.setPaddingBottom(5);

            PdfPCell c2 = null;
            c2=  new PdfPCell(new Phrase("土地利用现状图", normalFont));
            c2.setVerticalAlignment(Element.ALIGN_CENTER);
            c2.setHorizontalAlignment(Element.ALIGN_CENTER);
            c2.setPaddingTop(5);
            c2.setPaddingBottom(5);

            PdfPCell cimg1 = null;

            float width = 250;
            float heigth=250;
            if(info.ownershipImg!=null) {
               heigth = info.ownershipImg.getHeight() / (info.ownershipImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.ownershipImg.scaleToFit(width, heigth);
                cimg1 = new PdfPCell(info.ownershipImg);
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            }
            else
            {
                cimg1 = new PdfPCell();
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            }
            PdfPCell cimg2=null;
            if(info.landtypeImg!=null) {
                heigth = info.landtypeImg.getHeight() / (info.landtypeImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.landtypeImg.scaleToFit(width, heigth);
                cimg2 = new PdfPCell(info.landtypeImg);
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            }
            else
            {
                cimg2 = new PdfPCell();
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            }
            PdfPCell meta1 =null;
            meta1 = new PdfPCell(new Phrase(info.ownershipImgMeta, roleFont));
            meta1.setVerticalAlignment(Element.ALIGN_CENTER);
            meta1.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta1.setPaddingTop(5);
            meta1.setPaddingBottom(5);


            PdfPCell meta2 =null;
            meta2 = new PdfPCell(new Phrase(info.landtypeImgMeta, roleFont));
            meta2.setVerticalAlignment(Element.ALIGN_CENTER);
            meta2.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta2.setPaddingTop(5);
            meta2.setPaddingBottom(5);


            if(info.narrow)
            {
                if(info.ownershipImg!=null) {
                    t.addCell(c1);
                    t.addCell(cimg1);
                    t.addCell(meta1);

                }
                if(info.landtypeImg!=null) {
                    t.addCell(c2);
                    t.addCell(cimg2);
                    t.addCell(meta2);

                }
            }
            else
            {
                t.addCell(c1);
                t.addCell(c2);
                t.addCell(cimg1);
                t.addCell(cimg2);
                t.addCell(meta1);
                t.addCell(meta2);
            }
            if (info.photo1 != null || info.photo2 != null) {
                PdfPCell photo1 = new PdfPCell(new Phrase("实地照片1", normalFont));
                photo1.setVerticalAlignment(Element.ALIGN_CENTER);
                photo1.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo1.setPaddingTop(5);
                photo1.setPaddingBottom(5);


                PdfPCell photo2 = new PdfPCell(new Phrase("实地照片2", normalFont));
                photo2.setVerticalAlignment(Element.ALIGN_CENTER);
                photo2.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo2.setPaddingTop(5);
                photo2.setPaddingBottom(5);


                PdfPCell cp1 = null;
                if (info.photo1 != null) {
                    Image img1 = Image.getInstance(info.photo1);
                    heigth = img1.getHeight() / (img1.getWidth() / width);
                    img1.scaleToFit(width, heigth);
                    cp1 = new PdfPCell(img1);
                } else {
                    cp1 = new PdfPCell();
                }
                cp1.setVerticalAlignment(Element.ALIGN_CENTER);
                cp1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp1.setPaddingTop(5);
                cp1.setPaddingBottom(5);
                cp1.setPaddingLeft(5);
                cp1.setPaddingRight(5);


                PdfPCell cp2 = null;
                if (info.photo2 != null) {
                    Image img2 = Image.getInstance(info.photo2);
                    heigth = img2.getHeight() / (img2.getWidth() / width);
                    img2.scaleToFit(width, heigth);
                    cp2 = new PdfPCell(img2);
                } else {
                    cp2 = new PdfPCell();
                }
                cp2.setVerticalAlignment(Element.ALIGN_CENTER);
                cp2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp2.setPaddingTop(5);
                cp2.setPaddingBottom(5);
                cp2.setPaddingLeft(5);
                cp2.setPaddingRight(5);

                PdfPCell pmeta1 = new PdfPCell(new Phrase(info.photoMeta1, normalFont));
                pmeta1.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta1.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta1.setPaddingTop(5);
                pmeta1.setPaddingBottom(5);

                PdfPCell pmeta2 = new PdfPCell(new Phrase(info.photoMeta2, normalFont));
                pmeta2.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta2.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta2.setPaddingTop(5);
                pmeta2.setPaddingBottom(5);

                if(info.narrow)
                {
                    t.addCell(photo1);
                    t.addCell(cp1);
                    t.addCell(pmeta1);
                    t.addCell(photo2);
                    t.addCell(cp2);
                    t.addCell(pmeta2);
                }
                else
                {
                    t.addCell(photo1);
                    t.addCell(photo2);
                    t.addCell(cp1);
                    t.addCell(cp2);
                    t.addCell(pmeta1);
                    t.addCell(pmeta2);
                }
            }


        document.add(t);
         */

    }

    private void BaseInfoLawless(Document document, LawlessInfo info,ResultInfo result) throws DocumentException, IOException {
        Paragraph chapter1Pg = new Paragraph("一、基本情况", chapterFont);
        chapter1Pg.setExtraParagraphSpace(30);
        chapter1Pg.setSpacingBefore(pgSpace);
        chapter1Pg.setLeading(lineSpace);
        document.add(chapter1Pg);

        String content ="该遥感监测疑似违法用地位于%s%s%s（东经：%s°，北纬：%s°），监测面积%s亩。";
        content = String.format(content,info.provinceName,info.cityName,info.countyName,info.lontitude,info.latitude, String.valueOf(Round(info.area,2)));
        String dlinfo="";
        int lengLy=0;
        int lengOw=0;
        if(info.landTypes!=null&&info.landTypes.size()>0) {
            lengLy = info.landTypes.size();
        }
        if(info.ownerships!=null&&info.ownerships.size()>0) {
            lengOw = info.ownerships.size();
        }
        if(lengLy>0){
            dlinfo="土地利用现状：";
            double otherArea = 0;
            int count = 0;
            for (LandType item : info.landTypes) {
                count++;
                if(count>2)
                {
                    otherArea+=item.area;
                }
                else
                {
                    dlinfo += item.dlmc + Round(item.area, 2) + "亩";
                    if (count < lengLy) {
                        dlinfo += "、";
                    }
                }

            }
            if(otherArea>0)
            {
                dlinfo += ("其他" + Round(otherArea, 2) + "亩");
            }
            if(lengOw>0)
            {
                dlinfo+="；";
            }
            else
            {
                dlinfo+="。";
            }
            /*
            int count = 0;
            for(LandType item : info.landTypes)
            {
                count++;
                dlinfo+=item.dlmc+Round(item.area,2)+"亩";
                if(count<lengLy)
                {
                    dlinfo+="、";
                }
                else
                {
                    if(lengOw>0)
                    {
                        dlinfo+=";";
                    }
                    else
                    {
                        dlinfo+="。";
                    }
                }
            }
            */

        }
        if(lengOw>0)
        {
            dlinfo+="土地权属为";
            HashMap<String,Double> dic = new HashMap<String,Double>();
            dic.put("国有",0.0);
            dic.put("集体",0.0);
            dic.put("其他",0.0);
            for(Ownership item : info.ownerships)
            {

                if(item.detail!=null)
                {
                    dic.put("国有",dic.get("国有")+item.detail.lx10);
                    dic.put("国有",dic.get("国有")+item.detail.lx20);
                    dic.put("集体",dic.get("集体")+item.detail.lx30);
                    dic.put("集体",dic.get("集体")+item.detail.lx40);
                    dic.put("其他",dic.get("其他")+item.detail.lx99);

                }
            }
            if(Round(dic.get("国有"),2)>0) {
                dlinfo += "国有" + Round(dic.get("国有"), 2) + "亩、";
            }
            if(Round(dic.get("集体"),2)>0) {
                dlinfo += "集体" + Round(dic.get("集体"), 2) + "亩、";
            }
            if(Round(dic.get("其他"),2)>0) {
                dlinfo += "其他" + Round(dic.get("其他"), 2) + "亩、";
            }
            if(dlinfo.endsWith("、")) {
                dlinfo = dlinfo.substring(0, dlinfo.length() - 1) + "。";
            }
        }

        content+=dlinfo;
        Paragraph baseInfo = new Paragraph(content, normalFont);
        baseInfo.setFirstLineIndent(firstLineIndent);
        baseInfo.setExtraParagraphSpace(30);
        baseInfo.setSpacingBefore(pgSpace);
        baseInfo.setLeading(lineSpace);
        document.add(baseInfo);
        AddBaseImgLawless(document,info);

    }

    private void JcqkAnalysis(Document document,LawlessInfo info,ResultInfo result) throws DocumentException {
        Paragraph chapter2Pg = new Paragraph("二、监测情况分析", chapterFont);
        chapter2Pg.setExtraParagraphSpace(30);
        chapter2Pg.setSpacingBefore(pgSpace);
        chapter2Pg.setLeading(lineSpace);
        document.add(chapter2Pg);

        Paragraph comparePg = new Paragraph("(一)两期影像比对提取", sectionFont);
        comparePg.setExtraParagraphSpace(30);
        comparePg.setSpacingBefore(pgSpace);
        comparePg.setLeading(lineSpace);
        document.add(comparePg);

        if(info.narrow)
        {
            AddImg2(document,info.year1 + "度影像",info.image1,info.imageMeta1,true,false);
            AddImg2(document,info.year2 + "度影像",info.image2,info.imageMeta2,false,true);
            /*
            float width = 250;
            float heigth = 250;

            PdfPTable t1 = new PdfPTable(1);
            t1.setSpacingBefore(pgSpace);//设置上边距
            t1.setSpacingAfter(pgSpace);//设置下边距
            t1.setWidthPercentage(80);

            PdfPCell c1 = null;
            c1 = new PdfPCell(new Phrase(info.year1 + "度影像", normalFont));
            c1.setVerticalAlignment(Element.ALIGN_CENTER);
            c1.setHorizontalAlignment(Element.ALIGN_CENTER);
            c1.setPaddingTop(5);
            c1.setPaddingBottom(5);

            PdfPCell cimg1 = null;

            if (info.image1 != null) {
                heigth = info.image1.getHeight() / (info.image1.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.image1.scaleToFit(width, heigth);
                cimg1 = new PdfPCell(info.image1);
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            } else {
                cimg1 = new PdfPCell();
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            }

            PdfPCell meta1 = null;
            meta1 = new PdfPCell(new Phrase(info.imageMeta1, roleFont));
            meta1.setVerticalAlignment(Element.ALIGN_CENTER);
            meta1.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta1.setPaddingTop(5);
            meta1.setPaddingBottom(5);

            PdfPTable t2 = new PdfPTable(1);
            t2.setSpacingBefore(pgSpace);//设置上边距
            t2.setSpacingAfter(pgSpace);//设置下边距
            t2.setWidthPercentage(80);

            PdfPCell c2 = null;
            c2 = new PdfPCell(new Phrase(info.year2 + "度影像", normalFont));
            c2.setVerticalAlignment(Element.ALIGN_CENTER);
            c2.setHorizontalAlignment(Element.ALIGN_CENTER);
            c2.setPaddingTop(5);
            c2.setPaddingBottom(5);


            PdfPCell cimg2 = null;
            if (info.image2 != null) {
                heigth = info.image2.getHeight() / (info.image2.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.image2.scaleToFit(width, heigth);
                cimg2 = new PdfPCell(info.image2);
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            } else {
                cimg2 = new PdfPCell();
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            }

            PdfPCell meta2 = null;
            meta2 = new PdfPCell(new Phrase(info.imageMeta2, roleFont));
            meta2.setVerticalAlignment(Element.ALIGN_CENTER);
            meta2.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta2.setPaddingTop(5);
            meta2.setPaddingBottom(5);
            t1.addCell(c1);
            t1.addCell(c2);
            t1.addCell(cimg1);
            t2.addCell(cimg2);
            t2.addCell(meta1);
            t2.addCell(meta2);
            document.add(t1);
            document.add(t2);
            */
        }
        else {
            PdfPTable t = new PdfPTable(2);//创建一个2列表格对象
            t.setSpacingBefore(pgSpace);//设置上边距
            t.setSpacingAfter(pgSpace);//设置下边距
            t.setWidthPercentage(100);

            PdfPCell c1 = null;
            c1 = new PdfPCell(new Phrase(info.year1 + "度影像", normalFont));
            c1.setVerticalAlignment(Element.ALIGN_CENTER);
            c1.setHorizontalAlignment(Element.ALIGN_CENTER);
            c1.setPaddingTop(5);
            c1.setPaddingBottom(5);

            PdfPCell c2 = null;
            c2 = new PdfPCell(new Phrase(info.year2 + "度影像", normalFont));
            c2.setVerticalAlignment(Element.ALIGN_CENTER);
            c2.setHorizontalAlignment(Element.ALIGN_CENTER);
            c2.setPaddingTop(5);
            c2.setPaddingBottom(5);

            PdfPCell cimg1 = null;

            float width = 250;
            float heigth = 250;
            if (info.image1 != null) {
                heigth = info.image1.getHeight() / (info.image1.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.image1.scaleToFit(width, heigth);
                cimg1 = new PdfPCell(info.image1);
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            } else {
                cimg1 = new PdfPCell();
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            }
            PdfPCell cimg2 = null;
            if (info.image2 != null) {
                heigth = info.image2.getHeight() / (info.image2.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.image2.scaleToFit(width, heigth);
                cimg2 = new PdfPCell(info.image2);
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            } else {
                cimg2 = new PdfPCell();
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            }
            PdfPCell meta1 = null;
            meta1 = new PdfPCell(new Phrase(info.imageMeta1, roleFont));
            meta1.setVerticalAlignment(Element.ALIGN_CENTER);
            meta1.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta1.setPaddingTop(5);
            meta1.setPaddingBottom(5);


            PdfPCell meta2 = null;
            meta2 = new PdfPCell(new Phrase(info.imageMeta2, roleFont));
            meta2.setVerticalAlignment(Element.ALIGN_CENTER);
            meta2.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta2.setPaddingTop(5);
            meta2.setPaddingBottom(5);
            t.addCell(c1);
            t.addCell(c2);
            t.addCell(cimg1);
            t.addCell(cimg2);
            t.addCell(meta1);
            t.addCell(meta2);
            document.add(t);
        }
        Paragraph analyzePg = new Paragraph("(二)监测类型分析", sectionFont);
        analyzePg.setExtraParagraphSpace(30);
        analyzePg.setSpacingBefore(pgSpace);
        analyzePg.setLeading(lineSpace);
        document.add(analyzePg);

        String content="经分析，遥感监测疑似违法用地图斑监测类型为:";
        if(info.jctbMap!=null&&info.jctbMap.size()>0)
        {
            int count = 0;
            for (Map.Entry kv :info.jctbMap.entrySet()) {
                count++;
                Object key = kv.getKey();
                Object value = kv.getValue();
                String tblb = "未知";
                if (key != null) {
                    tblb = key.toString();
                    if (info.configs != null) {
                        for (CloudAnalyzeConfig con : info.configs) {
                            if (tblb.equals(con.getKey())) {
                                tblb += "(" + con.getValue() + ")";
                            }
                        }
                    }
                }
                if (count < info.jctbMap.size())
                {
                    tblb+="；";
                }

                content+=tblb;
            }

        }
        else
        {
            content+="未知";
        }
        content+="。";
        Paragraph analyzeDescPg = new Paragraph(content, normalFont);
        analyzeDescPg.setFirstLineIndent(firstLineIndent);
        analyzeDescPg.setExtraParagraphSpace(30);
        analyzeDescPg.setSpacingBefore(pgSpace);
        analyzeDescPg.setLeading(lineSpace);
        document.add(analyzeDescPg);
        //AddRoleDesc("需先实地确认该图斑为公路用地。建议提前准备好证明该监测图斑为农村道路的相关证明材料，待执法检查时解释。","技术要求");

    }

    private void LegalAnalysis(Document document,ProjectInfo info,ResultInfo result) throws DocumentException
    {
        Paragraph chapter2Pg = new Paragraph("二、合法性分析", chapterFont);
        chapter2Pg.setExtraParagraphSpace(30);
        chapter2Pg.setSpacingBefore(pgSpace);
        chapter2Pg.setLeading(lineSpace);
        document.add(chapter2Pg);

        Paragraph jbntPg = new Paragraph("(一)永久基本农田情况分析", sectionFont);
        jbntPg.setExtraParagraphSpace(30);
        jbntPg.setSpacingBefore(pgSpace);
        jbntPg.setLeading(lineSpace);
        document.add(jbntPg);

        String jbntResult ="";
        if(info.jbntArea>0)
        {
            result.passJbnt = EnumAnalyzeResult.NoPass;
            if(info.jbntArea<info.area)
            {
                result.passJbnt = EnumAnalyzeResult.PartPass;
            }
            jbntResult = "拟建设项目区涉及永久基本农田，共" + Round(info.jbntArea, 2) + "亩。";

        }
        else
        {
            result.passJbnt=EnumAnalyzeResult.AllPass;
            jbntResult ="拟建设项目区不涉及永久基本农田。";
        }
        Paragraph jbntInfo = new Paragraph(jbntResult, normalFont);
        jbntInfo.setFirstLineIndent(firstLineIndent);
        jbntInfo.setExtraParagraphSpace(30);
        jbntInfo.setSpacingBefore(pgSpace);
        jbntInfo.setLeading(lineSpace);
        document.add(jbntInfo);

        if(info.jbntImg!=null)
        {
            String desc = "基本农田："+Round(info.jbntArea,2)+"亩，非基本农田："+Round((info.area - info.jbntArea),2)+"亩";
            AddImg(document,"永久基本农田",info.jbntImg,desc);
        }
        AddRoleDesc(document,"《中华人民共和国土地管理法》第三十五条  基本农田经依法划定后，任何单位和个人不得擅自占用或者改变其用途。国家能源、交通、水利、军事设施等重点建设项目选址确实难以避让永久基本农田，涉及农用地转用或者土地征收的，必须经国务院批准。","政策解读");


        Paragraph tdghPg = new Paragraph("(二)土地利用总体规划情况分析", sectionFont);
        tdghPg.setExtraParagraphSpace(30);
        tdghPg.setSpacingBefore(pgSpace);
        tdghPg.setLeading(lineSpace);
        document.add(tdghPg);

        String tdghDesc ="";
        String tdghResult ="";
        double cantJsArea = 0;
        if(info.tdghInfos!=null)
        {
            int count = 0;
            int leng = info.tdghInfos.size();
            for(TdghInfo item : info.tdghInfos)
            {
                count++;
                tdghDesc+=item.jsmc+"("+item.jsbm+")-"+Round(item.area,2);
                if(count<leng)
                {
                    tdghDesc+="/";
                }
                if("040".equals(item.jsbm))
                {
                    cantJsArea+=item.area;
                }
                if("030".equals(item.jsbm))
                {
                    cantJsArea+=item.area;
                }
            }

        }
        if(cantJsArea>0)
        {
            result.passTdgh=EnumAnalyzeResult.NoPass;
            tdghResult ="拟建设项目区不符合土地利用总体规划。";
        }
        else
        {
            result.passTdgh=EnumAnalyzeResult.AllPass;
            tdghResult ="拟建设项目区符合土地利用总体规划。";
        }
        Paragraph tdghInfo = new Paragraph(tdghResult, normalFont);
        tdghInfo.setFirstLineIndent(firstLineIndent);
        tdghInfo.setExtraParagraphSpace(30);
        tdghInfo.setSpacingBefore(pgSpace);
        tdghInfo.setLeading(lineSpace);
        document.add(tdghInfo);

        if(info.tdghImg!=null)
        {
            AddImg(document,"土地规划",info.tdghImg,tdghDesc);
        }
        AddRoleDesc(document,"《中华人民共和国土地管理法》第二十五条 经批准的土地利用总体规划的修改，须经原审批机关批准；未经批准，不得改变土地利用总体规划确定的土地用途。","政策解读");



        Paragraph spbaPg = new Paragraph("(三)建设用地审批情况分析", sectionFont);
        spbaPg.setExtraParagraphSpace(30);
        spbaPg.setSpacingBefore(pgSpace);
        spbaPg.setLeading(lineSpace);
        document.add(spbaPg);

        String spbaResult="";
        double spbaArea = 0;
        if(info.spbaInfos!=null&&info.spbaInfos.size()>0)
        {

            String license = "";
            int count=0;
            for (SpbaInfo sp: info.spbaInfos)
            {
                count++;
                spbaArea+=sp.area;
                license+=sp.license;
                if(count<info.spbaInfos.size())
                {
                    license+="、";
                }

            }
            result.passSpba=EnumAnalyzeResult.NoPass;
            if(spbaArea>0) {
                if (spbaArea < info.area) {
                    result.passSpba=EnumAnalyzeResult.PartPass;
                    spbaResult = "拟建设项目区有部分审批备案信息。";
                } else
                {
                    result.passSpba=EnumAnalyzeResult.AllPass;
                    spbaResult = "拟建设项目区有审批备案信息。";
                }
                if(license!=null&&!license.isEmpty())
                {
                    spbaResult+="备案号为："+license;
                }
            }
            else
            {
                result.passSpba=EnumAnalyzeResult.NoPass;
                spbaResult ="拟建设项目区无审批备案信息。";
            }

        }
        else
        {
            result.passSpba=EnumAnalyzeResult.NoPass;
            spbaResult ="拟建设项目区无审批备案信息。";
        }
        Paragraph spbaInfo = new Paragraph(spbaResult, normalFont);
        spbaInfo.setFirstLineIndent(firstLineIndent);
        spbaInfo.setExtraParagraphSpace(30);
        spbaInfo.setSpacingBefore(pgSpace);
        spbaInfo.setLeading(lineSpace);
        document.add(spbaInfo);

        if(info.spbaImg!=null)
        {
            String desc = "有审批："+Round(spbaArea,2)+"亩，无审批："+Round((info.area - spbaArea),2)+"亩";
            AddImg(document,"用地审批",info.spbaImg,desc);
        }
        AddRoleDesc(document,"《中华人民共和国土地管理法》第四十四条 建设占土地，涉及农用地转为建设用地的，应当办理农用地转用审批手续。","政策解读");
        AddRoleDesc(document,"永久基本农田转为建设用地的，由国务院审批。",null);
        AddRoleDesc(document,"在土地利用总体规划确定的城市和村庄、集镇建设用地规模范围内，未实施该规定而将永久基本农田以外的农用地转为建设用地的，按土地利用年度计划分批次按照国务院规定由原审批土地利用总体规划的机关单位或者其授权的机关审批。在已批准的农用地转用范围内，具体建设项目用地可以由市、县人民政府批准。",null);
        AddRoleDesc(document,"第四十六条 征收下列土地的，由国务院批准：",null);
        AddRoleDesc(document,"（一）永久基本农田；",null);
        AddRoleDesc(document,"（二）永久基本农田以外的耕地超过三十五公顷的；",null);
        AddRoleDesc(document,"（三）其他土地超过七十公顷的。",null);
        AddRoleDesc(document,"征收前款规定以外的土地的，由省、自治区、直辖市人民政府审批。",null);
        AddRoleDesc(document,"征收农用地的，应当依照本法第四十四条的规定先行办理农用地转用审批。其中，经国务院批准农用地转用的，同时办理征地审批手续，不再另行办理征地审批；经省、自治区、直辖市人民政府在征地批准权限内批准农用地转用的，同时办理征地审批手续，不再另行办理征地审批，超过征地批准权限的，应当依照本条第一款的规定另行办理征地审批。",null);

        Paragraph zrbhqPg = new Paragraph("(四)自然保护地情况分析", sectionFont);
        zrbhqPg.setExtraParagraphSpace(30);
        zrbhqPg.setSpacingBefore(pgSpace);
        zrbhqPg.setLeading(lineSpace);
        document.add(zrbhqPg);

        String zrbhqDesc ="";
        if(info.zrbhqArea>0)
        {
            result.passZrbhq=EnumAnalyzeResult.NoPass;
            if(info.zrbhqArea<info.area)
            {
                result.passZrbhq=EnumAnalyzeResult.PartPass;
            }
            zrbhqDesc ="拟建设项目区涉及自然保护地，共 "+Round(info.zrbhqArea,2)+"亩";
        }
        else
        {
            result.passZrbhq=EnumAnalyzeResult.AllPass;
            zrbhqDesc ="拟建设项目区不涉及自然保护地。";
        }
        Paragraph zrbhqInfo = new Paragraph(zrbhqDesc, normalFont);
        zrbhqInfo.setFirstLineIndent(firstLineIndent);
        zrbhqInfo.setExtraParagraphSpace(30);
        zrbhqInfo.setSpacingBefore(pgSpace);
        zrbhqInfo.setLeading(lineSpace);
        document.add(zrbhqInfo);

        if(info.zrbhqImg!=null)
        {
            String desc = "自然保护地："+Round(info.zrbhqArea,2)+"亩，非自然保护地："+Round((info.area - info.zrbhqArea),2)+"亩";

            AddImg(document,"自然保护用地",info.zrbhqImg,desc);
        }
        AddRoleDesc(document,"《自然保护区土地管理办法》第十五条 自然保护区内土地的使用，不得违反有关环境和资源保护法律的规定。","政策解读");
        AddRoleDesc(document,"依法使用自然保护区内土地的单位和个人必须严格按照土地登记和土地证书规定的用途使用土地，并严格遵守有关法律的规定。改变用途时，需事先征求环境保护及有关自然保护区行政主管部门的意见，由县级以上人民政府土地管理行政主管部门审查，报县级以上人民政府批准。",null);

    }
    private void LegalAnalysisLawless(Document document,LawlessInfo info,ResultInfo result) throws DocumentException
    {
        Paragraph chapter2Pg = new Paragraph("二、合法性分析", chapterFont);
        chapter2Pg.setExtraParagraphSpace(30);
        chapter2Pg.setSpacingBefore(pgSpace);
        chapter2Pg.setLeading(lineSpace);
        document.add(chapter2Pg);

        Paragraph jbntPg = new Paragraph("(一)永久基本农田情况分析", sectionFont);
        jbntPg.setExtraParagraphSpace(30);
        jbntPg.setSpacingBefore(pgSpace);
        jbntPg.setLeading(lineSpace);
        document.add(jbntPg);

        String jbntResult ="";
        if(info.jbntArea>0)
        {
            result.passJbnt = EnumAnalyzeResult.NoPass;
            if(info.jbntArea<info.area)
            {
                result.passJbnt = EnumAnalyzeResult.PartPass;
            }
            jbntResult = "该遥感监测图斑涉及永久基本农田，共" + Round(info.jbntArea, 2) + "亩。";

        }
        else
        {
            result.passJbnt=EnumAnalyzeResult.AllPass;
            jbntResult ="该遥感监测图斑不涉及永久基本农田。";
        }
        Paragraph jbntInfo = new Paragraph(jbntResult, normalFont);
        jbntInfo.setFirstLineIndent(firstLineIndent);
        jbntInfo.setExtraParagraphSpace(30);
        jbntInfo.setSpacingBefore(pgSpace);
        jbntInfo.setLeading(lineSpace);
        document.add(jbntInfo);

        if(info.jbntImg!=null)
        {
            String desc = "基本农田："+Round(info.jbntArea,2)+"亩，非基本农田："+Round((info.area - info.jbntArea),2)+"亩";
            AddImg(document,"永久基本农田",info.jbntImg,desc);
        }
        AddRoleDesc(document,"《中华人民共和国土地管理法》第三十五条  基本农田经依法划定后，任何单位和个人不得擅自占用或者改变其用途。国家能源、交通、水利、军事设施等重点建设项目选址确实难以避让永久基本农田，涉及农用地转用或者土地征收的，必须经国务院批准。","政策解读");


        Paragraph tdghPg = new Paragraph("(二)土地利用总体规划情况分析", sectionFont);
        tdghPg.setExtraParagraphSpace(30);
        tdghPg.setSpacingBefore(pgSpace);
        tdghPg.setLeading(lineSpace);
        document.add(tdghPg);

        String tdghDesc ="";
        String tdghResult ="";
        double cantJsArea = 0;
        if(info.tdghInfos!=null)
        {
            int count = 0;
            int leng = info.tdghInfos.size();
            for(TdghInfo item : info.tdghInfos)
            {
                count++;
                tdghDesc+=item.jsmc+"("+item.jsbm+")-"+Round(item.area,2);
                if(count<leng)
                {
                    tdghDesc+="/";
                }
                if("040".equals(item.jsbm))
                {
                    cantJsArea+=item.area;
                }
                if("030".equals(item.jsbm))
                {
                    cantJsArea+=item.area;
                }
            }

        }
        if(cantJsArea>0)
        {
            result.passTdgh=EnumAnalyzeResult.NoPass;
            tdghResult ="该遥感监测图斑不符合土地利用总体规划。";
        }
        else
        {
            result.passTdgh=EnumAnalyzeResult.AllPass;
            tdghResult ="该遥感监测图斑符合土地利用总体规划。";
        }
        Paragraph tdghInfo = new Paragraph(tdghResult, normalFont);
        tdghInfo.setFirstLineIndent(firstLineIndent);
        tdghInfo.setExtraParagraphSpace(30);
        tdghInfo.setSpacingBefore(pgSpace);
        tdghInfo.setLeading(lineSpace);
        document.add(tdghInfo);

        if(info.tdghImg!=null)
        {
            AddImg(document,"土地规划",info.tdghImg,tdghDesc);
        }
        AddRoleDesc(document,"《中华人民共和国土地管理法》第二十五条 经批准的土地利用总体规划的修改，须经原审批机关批准；未经批准，不得改变土地利用总体规划确定的土地用途。","政策解读");



        Paragraph spbaPg = new Paragraph("(三)建设用地审批情况分析", sectionFont);
        spbaPg.setExtraParagraphSpace(30);
        spbaPg.setSpacingBefore(pgSpace);
        spbaPg.setLeading(lineSpace);
        document.add(spbaPg);

        String spbaResult="";
        double spbaArea = 0;
        if(info.spbaInfos!=null&&info.spbaInfos.size()>0)
        {

            String license = "";
            int count=0;
            for (SpbaInfo sp: info.spbaInfos)
            {
                count++;
                spbaArea+=sp.area;
                license+=sp.license;
                if(count<info.spbaInfos.size())
                {
                    license+="、";
                }

            }
            result.passSpba=EnumAnalyzeResult.NoPass;
            if(spbaArea>0) {
                if (spbaArea < info.area) {
                    result.passSpba=EnumAnalyzeResult.PartPass;
                    spbaResult = "该遥感监测图斑有部分审批备案信息。";
                } else
                {
                    result.passSpba=EnumAnalyzeResult.AllPass;
                    spbaResult = "该遥感监测图斑有审批备案信息。";
                }
                if(license!=null&&!license.isEmpty())
                {
                    spbaResult+="备案号为："+license;
                }
            }
            else
            {
                result.passSpba=EnumAnalyzeResult.NoPass;
                spbaResult ="该遥感监测图斑无审批备案信息。";
            }

        }
        else
        {
            result.passSpba=EnumAnalyzeResult.NoPass;
            spbaResult ="该遥感监测图斑无审批备案信息。";
        }
        Paragraph spbaInfo = new Paragraph(spbaResult, normalFont);
        spbaInfo.setFirstLineIndent(firstLineIndent);
        spbaInfo.setExtraParagraphSpace(30);
        spbaInfo.setSpacingBefore(pgSpace);
        spbaInfo.setLeading(lineSpace);
        document.add(spbaInfo);

        if(info.spbaImg!=null)
        {
            String desc = "有审批："+Round(spbaArea,2)+"亩，无审批："+Round((info.area - spbaArea),2)+"亩";
            AddImg(document,"用地审批",info.spbaImg,desc);
        }
        AddRoleDesc(document,"《中华人民共和国土地管理法》第四十四条 建设占土地，涉及农用地转为建设用地的，应当办理农用地转用审批手续。","政策解读");
        AddRoleDesc(document,"永久基本农田转为建设用地的，由国务院审批。",null);
        AddRoleDesc(document,"在土地利用总体规划确定的城市和村庄、集镇建设用地规模范围内，未实施该规定而将永久基本农田以外的农用地转为建设用地的，按土地利用年度计划分批次按照国务院规定由原审批土地利用总体规划的机关单位或者其授权的机关审批。在已批准的农用地转用范围内，具体建设项目用地可以由市、县人民政府批准。",null);
        AddRoleDesc(document,"第四十六条 征收下列土地的，由国务院批准：",null);
        AddRoleDesc(document,"（一）永久基本农田；",null);
        AddRoleDesc(document,"（二）永久基本农田以外的耕地超过三十五公顷的；",null);
        AddRoleDesc(document,"（三）其他土地超过七十公顷的。",null);
        AddRoleDesc(document,"征收前款规定以外的土地的，由省、自治区、直辖市人民政府审批。",null);
        AddRoleDesc(document,"征收农用地的，应当依照本法第四十四条的规定先行办理农用地转用审批。其中，经国务院批准农用地转用的，同时办理征地审批手续，不再另行办理征地审批；经省、自治区、直辖市人民政府在征地批准权限内批准农用地转用的，同时办理征地审批手续，不再另行办理征地审批，超过征地批准权限的，应当依照本条第一款的规定另行办理征地审批。",null);

        Paragraph zrbhqPg = new Paragraph("(四)自然保护地情况分析", sectionFont);
        zrbhqPg.setExtraParagraphSpace(30);
        zrbhqPg.setSpacingBefore(pgSpace);
        zrbhqPg.setLeading(lineSpace);
        document.add(zrbhqPg);

        String zrbhqDesc ="";
        if(info.zrbhqArea>0)
        {
            result.passZrbhq=EnumAnalyzeResult.NoPass;
            if(info.zrbhqArea<info.area)
            {
                result.passZrbhq=EnumAnalyzeResult.PartPass;
            }
            zrbhqDesc ="该遥感监测图斑涉及自然保护地，共 "+Round(info.zrbhqArea,2)+"亩";
        }
        else
        {
            result.passZrbhq=EnumAnalyzeResult.AllPass;
            zrbhqDesc ="该遥感监测图斑不涉及自然保护地。";
        }
        Paragraph zrbhqInfo = new Paragraph(zrbhqDesc, normalFont);
        zrbhqInfo.setFirstLineIndent(firstLineIndent);
        zrbhqInfo.setExtraParagraphSpace(30);
        zrbhqInfo.setSpacingBefore(pgSpace);
        zrbhqInfo.setLeading(lineSpace);
        document.add(zrbhqInfo);

        if(info.zrbhqImg!=null)
        {
            String desc = "自然保护地："+Round(info.zrbhqArea,2)+"亩，非自然保护地："+Round((info.area - info.zrbhqArea),2)+"亩";

            AddImg(document,"自然保护用地",info.zrbhqImg,desc);
        }
        AddRoleDesc(document,"《自然保护区土地管理办法》第十五条 自然保护区内土地的使用，不得违反有关环境和资源保护法律的规定。","政策解读");
        AddRoleDesc(document,"依法使用自然保护区内土地的单位和个人必须严格按照土地登记和土地证书规定的用途使用土地，并严格遵守有关法律的规定。改变用途时，需事先征求环境保护及有关自然保护区行政主管部门的意见，由县级以上人民政府土地管理行政主管部门审查，报县级以上人民政府批准。",null);

    }

    private void BzInfo(Document document) throws DocumentException
    {
        String content="说明：由于我们不掌握环评信息和城乡规划数据等相关管理数据，云分析报告不涉及环评和城乡规划符合性等分析内容。";

        Paragraph contentPg = new Paragraph(content, redFont);
        contentPg.setFirstLineIndent(firstLineIndent);
        contentPg.setExtraParagraphSpace(30);
        contentPg.setSpacingBefore(pgSpace);
        contentPg.setLeading(lineSpace);
        document.add(contentPg);
    }

    private void SuitableAnalysis(Document document,ProjectInfo info,ResultInfo result,boolean px) throws DocumentException
    {
        Paragraph chapter3Pg = new Paragraph("三、适宜性分析", chapterFont);
        chapter3Pg.setExtraParagraphSpace(30);
        chapter3Pg.setSpacingBefore(pgSpace);
        chapter3Pg.setLeading(lineSpace);
        document.add(chapter3Pg);

        Paragraph jzdjPg = new Paragraph("(一)征地区片地价分析", sectionFont);
        jzdjPg.setExtraParagraphSpace(30);
        jzdjPg.setSpacingBefore(pgSpace);
        jzdjPg.setLeading(lineSpace);
        document.add(jzdjPg);

        String jzdjResult ="";
        if(info.landPrice>0)
        {
            jzdjResult ="拟建设项目区征地区片地价为："+ Round(info.landPrice,2)+"元/亩";
        }
        else
        {
            jzdjResult ="拟建设项目区征地区片地价未知 ";
        }

        Paragraph jzdjInfo = new Paragraph(jzdjResult, normalFont);
        jzdjInfo.setFirstLineIndent(firstLineIndent);
        jzdjInfo.setExtraParagraphSpace(30);
        jzdjInfo.setSpacingBefore(pgSpace);
        jzdjInfo.setLeading(lineSpace);
        document.add(jzdjInfo);


        if(info.landPriceImg!=null)
        {
            AddImg(document,"征地区片地价",info.landPriceImg,Round(info.landPrice,2)+"元/亩");
        }
        AddRoleDesc(document,"《自然保护区土地管理办法》第四十八条第三款  征收农用地的土地补偿费、安置补助费标准由省、自治区、直辖市通过制定公布区片综合地价确定。制定区片综合地价应当综合考虑土地原用途、土地资源条件、土地产值、土地区位、土地供求关系、人口以及经济社会发展水平等因素，并至少每三年调整或者重新公布一次。","政策解读");

        Paragraph dzzhPg = new Paragraph("(二)地质灾害隐患分析", sectionFont);
        dzzhPg.setExtraParagraphSpace(30);
        dzzhPg.setSpacingBefore(pgSpace);
        dzzhPg.setLeading(lineSpace);
        document.add(dzzhPg);

        String dzzhResult ="";
        double dzfxArea = 0;
        String dzfxDesc="";
        if(info.dzfxInfos!=null&&info.dzfxInfos.size()>0) {

            int count = 0;
            for(DzfxInfo dzfx:info.dzfxInfos)
            {
                count++;
                dzfxArea+=dzfx.area;
                dzfxDesc+=dzfx.level+ Round(dzfx.area,2)+"亩";
                if(count<info.dzfxInfos.size())
                {
                    dzfxDesc+="/";
                }
            }
        }
        if(dzfxArea>0)
        {
            result.passDzfx = EnumAnalyzeResult.NoPass;
            if(dzfxArea<info.area)
            {
                result.passDzfx = EnumAnalyzeResult.PartPass;
            }
            dzzhResult ="拟建设项目区存在地质灾害隐患。";
        }
        else
        {
            result.passDzfx = EnumAnalyzeResult.AllPass;
            if(info.provinceCode!=null&&info.provinceCode.startsWith("22"))
            {
                dzzhResult ="拟建设项目区不存在地质灾害隐患。";
            }
            else
            {
                dzzhResult ="拟建设项目区地质灾害隐患情况不详（无该区域地质灾害数据）。";

            }
        }


        Paragraph dzzhInfo = new Paragraph(dzzhResult, normalFont);
        dzzhInfo.setFirstLineIndent(firstLineIndent);
        dzzhInfo.setExtraParagraphSpace(30);
        dzzhInfo.setSpacingBefore(pgSpace);
        dzzhInfo.setLeading(lineSpace);
        document.add(dzzhInfo);

        if(info.dzfxImg!=null)
        {
            AddImg(document,"地质灾害隐患",info.dzfxImg,dzfxDesc);
        }
        AddRoleDesc(document,"《地质灾害防治管理办法》第25条明确规定：“城市建设有可能导致地质灾害发生的工程项目建设和地质灾害易发区内进行的工程建设在申请建设用地之前必须进行地质灾害风险性评估，评估结果由省级以上地质矿产行政主管部门认定。不符合条件的，土地行政主管部门不予办理建设用地审批手续”。","政策解读");


        Paragraph stydPg = new Paragraph("(三)生态用地分析", sectionFont);
        stydPg.setExtraParagraphSpace(30);
        stydPg.setSpacingBefore(pgSpace);
        stydPg.setLeading(lineSpace);
        document.add(stydPg);

        String stydResult ="";
        if(info.sdArea>0)
        {
            result.passSd=EnumAnalyzeResult.NoPass;
            if(info.sdArea<info.area)
            {
                result.passSd=EnumAnalyzeResult.PartPass;
            }
            stydResult ="拟建设项目区占用湿地，共"+Round(info.sdArea,2)+"亩";
        }
        else
        {
            result.passSd=EnumAnalyzeResult.AllPass;
            stydResult ="拟建设项目区不占用湿地。";
        }

        Paragraph stydInfo = new Paragraph(stydResult, normalFont);
        stydInfo.setFirstLineIndent(firstLineIndent);
        stydInfo.setExtraParagraphSpace(30);
        stydInfo.setSpacingBefore(pgSpace);
        stydInfo.setLeading(lineSpace);
        document.add(stydInfo);

        if(info.sdImg!=null)
        {
            AddImg(document,"生态用地",info.sdImg,"生态用地面积："+Round(info.sdArea,2)+"亩");
        }
        AddRoleDesc(document,"《湿地保护管理规定》第十八条 因工程建设等造成国际重要湿地生态特征退化甚至消失的，省、自治区、直辖市人民政府林业主管部门应当会同同级人民政府有关部门督促、指导项目建设单位限期恢复，并向同级人民政府和国家林业局报告；对逾期不予恢复或者确实无法恢复的，由国家林业局会商所在地省、自治区、直辖市人民政府和国务院有关部门后，按照有关规定处理。","政策解读");
        AddRoleDesc(document,"《湿地保护管理规定》第三十条建设项目应当不占或者少占湿地，经批准确需征收、占用湿地并转为其他用途的，用地单位应当按照“先补后占、占补平衡”的原则，依法办理相关手续。",null);
        AddRoleDesc(document,"临时占用湿地的，期限不得超过2年；临时占用期限届满，占用单位应当对所占湿地限期进行生态修复。",null);

        boolean hasGradeInfo=false;
        if(info.gradeInfos!=null&&info.gradeInfos.size()>0)
        {
            hasGradeInfo=true;
            Paragraph zbphPg = new Paragraph("(四)占补平衡情况分析", sectionFont);
            zbphPg.setExtraParagraphSpace(30);
            zbphPg.setSpacingBefore(pgSpace);
            zbphPg.setLeading(lineSpace);
            document.add(zbphPg);

            String zbphResult ="拟建设项目区";
            String first="";
            double area=0;
            String second="";
            HashMap<String,Double> map = new HashMap<String,Double>();
            for(LandUseGrade g : info.gradeInfos)
            {
                if(map.containsKey(g.field2))
                {
                    double gradeArea = map.get(g.field2)+g.area;
                    map.put(g.field2,gradeArea);

                }
                else
                {
                    map.put(g.field2, g.area);
                }
                if(!first.equalsIgnoreCase(g.field1))
                {

                    if(second!=null&&!second.isEmpty())
                    {
                        zbphResult+="占"+ first+ Round(area,2)+"亩，";
                        second = second.substring(0,second.length()-1);
                        zbphResult+=second+"；";
                    }
                    area = 0;
                    second="";
                    first = g.field1;
                }

                area+=g.area;
                if(g.field2==null)
                {
                    second += "耕地等别未知" + Round(g.area, 2) + "亩,";

                }
                else
                {
                    second += "耕地等别为" + g.field2 + "等" + Round(g.area, 2) + "亩，";
                }
            }
            if(second!=null&&!second.isEmpty())
            {
                zbphResult+="占"+ first+Round(area,2)+"亩，";
                second = second.substring(0,second.length()-1);
                zbphResult+=second+"。";
            }




            Paragraph zbphInfo = new Paragraph(zbphResult, normalFont);
            zbphInfo.setFirstLineIndent(firstLineIndent);
            zbphInfo.setExtraParagraphSpace(30);
            zbphInfo.setSpacingBefore(pgSpace);
            zbphInfo.setLeading(lineSpace);
            document.add(zbphInfo);

            if(info.gradeImg!=null)
            {
                String desc="";
                for(Map.Entry m: map.entrySet())
                {
                    Object obj = m.getKey();
                    String key="";
                    if(obj==null)
                    {
                        key="未知";
                    }
                    else
                    {
                        key = obj.toString();
                        key+="等";
                    }
                    desc+=key+Round((double)m.getValue(),2)+"亩/";
                }
                desc = desc.substring(0,desc.length()-1);
                AddImg(document,"耕地等别",info.gradeImg,desc);
            }
            AddRoleDesc(document,"《关于补足耕地数量与提升耕地质量相结合落实占补平衡的指导意见》提出，补改结合、提质改造必须坚持政府主导、因地制宜、维护农民权益和审慎稳妥推进的原则。","政策解读");
            AddRoleDesc(document,"《指导意见》明确了补改结合的适用范围，即对于单独选址建设项目受资源条件限制，无法落实占优补优、占水田补水田的，可通过补改结合落实；各类城市建设涉及占用耕地的，仍坚持先补后占和直接补充优质耕地，落实占优补优、占水田补水田。",null);

        }

        if(px)
        {
            String xh = hasGradeInfo?"五":"四";
            Paragraph sthxPg = new Paragraph("("+xh+")生态保护红线", sectionFont);
            sthxPg.setExtraParagraphSpace(30);
            sthxPg.setSpacingBefore(pgSpace);
            sthxPg.setLeading(lineSpace);
            document.add(sthxPg);

            String sthxResult ="";
            if(info.px_stbhhx_area>0)
            {
                result.passPxSthx=EnumAnalyzeResult.NoPass;
                if(info.px_stbhhx_area<info.area)
                {
                    result.passPxSthx=EnumAnalyzeResult.PartPass;
                }
                sthxResult ="拟建设项目区占用生态保护红线，共"+Round(info.px_stbhhx_area,2)+"亩";
            }
            else
            {
                result.passPxSthx=EnumAnalyzeResult.AllPass;
                sthxResult ="拟建设项目区不占用生态保护红线。";
            }

            Paragraph sthxInfo = new Paragraph(sthxResult, normalFont);
            sthxInfo.setFirstLineIndent(firstLineIndent);
            sthxInfo.setExtraParagraphSpace(30);
            sthxInfo.setSpacingBefore(pgSpace);
            sthxInfo.setLeading(lineSpace);
            document.add(sthxInfo);

            if(info.px_stbhhx_Img!=null)
            {
                AddImg(document,"生态保护红线",info.px_stbhhx_Img,"生态保护红线面积："+Round(info.px_stbhhx_area,2)+"亩");
            }
            AddRoleDesc(document,"《生态保护红线管理办法》第五条【管理原则】生态保护红线原则上按禁止开发区域的要求进行管理。遵循生态优先、严格管控、奖惩并重原则，严禁不符合主题功能定位的各类开发活动。根据主导生态功能定位，实时差别化管理，确保生态红线生态功能不降低、面积不减少、性质不改变。","政策解读");

            xh = hasGradeInfo?"六":"五";
            Paragraph gylPg = new Paragraph("("+xh+")公益林分析", sectionFont);
            gylPg.setExtraParagraphSpace(30);
            gylPg.setSpacingBefore(pgSpace);
            gylPg.setLeading(lineSpace);
            document.add(gylPg);
            String gylResult ="";
            if(info.px_gyl_area>0)
            {
                result.passPxGyl=EnumAnalyzeResult.NoPass;
                if(info.px_gyl_area<info.area)
                {
                    result.passPxGyl=EnumAnalyzeResult.PartPass;
                }
                gylResult ="拟建设项目区占用公益林，共"+Round(info.px_gyl_area,2)+"亩";
            }
            else
            {
                result.passPxGyl=EnumAnalyzeResult.AllPass;
                gylResult ="拟建设项目区不占用公益林。";
            }

            Paragraph gylInfo = new Paragraph(gylResult, normalFont);
            gylInfo.setFirstLineIndent(firstLineIndent);
            gylInfo.setExtraParagraphSpace(30);
            gylInfo.setSpacingBefore(pgSpace);
            gylInfo.setLeading(lineSpace);
            document.add(gylInfo);

            if(info.px_gyl_img!=null)
            {
                AddImg(document,"公益林",info.px_gyl_img,"公益林面积："+Round(info.px_gyl_area,2)+"亩");
            }
            AddRoleDesc(document,"《国家级公益林管理办法》第八条 国家级公益林不得随意调整。确需调整的，必须按照《国家级公益林区划界定办法》规定程序执行。","政策解读");
            AddRoleDesc(document,"第十一条 禁止在国家级公益林地开垦、采石、采沙、取土，严格控制勘查、开采矿藏和工程建设征收、征用、占用国家级公益林地。除国务院有关部门和省级人民政府批准的基础设施建设项目外，不得征收、征用、占用一级国家级公益林地。",null);
            AddRoleDesc(document,"经批准征收、征用、占用的国家级公益林地，由国家林业局进行审核汇总并相应核减国家级公益林总量，财政部根据国家林业局审核结果相应核减下一年度中央财政森林生态效益补偿基金。",null);

            xh = hasGradeInfo?"七":"六";
            Paragraph pdPg = new Paragraph("("+xh+")坡度图分析", sectionFont);
            pdPg.setExtraParagraphSpace(30);
            pdPg.setSpacingBefore(pgSpace);
            pdPg.setLeading(lineSpace);
            document.add(pdPg);

            String pdDesc ="";
            String pdResult ="";
            double cantJsArea = 0;
            if(info.px_pd!=null)
            {
                int count = 0;
                int leng = info.px_pd.size();
                for(Map.Entry item : info.px_pd.entrySet())
                {
                    count++;
                    pdDesc+=item.getKey()+"(级)-"+Round((double)item.getValue(),2)+"亩";
                    if(count<leng)
                    {
                        pdDesc+="/";
                    }
                    if("5".equals(item.getKey()))
                    {
                        cantJsArea+=(double)item.getValue();
                    }

                }

            }
            if(cantJsArea>0)
            {
                result.passPxPd=EnumAnalyzeResult.NoPass;
                pdResult ="拟建设项目区涉及5级坡度。";
            }
            else
            {
                result.passPxPd=EnumAnalyzeResult.AllPass;
                pdResult ="拟建设项目区未涉及5级坡度。";
            }
            Paragraph pdInfo = new Paragraph(pdResult, normalFont);
            pdInfo.setFirstLineIndent(firstLineIndent);
            pdInfo.setExtraParagraphSpace(30);
            pdInfo.setSpacingBefore(pgSpace);
            pdInfo.setLeading(lineSpace);
            document.add(pdInfo);

            if(info.px_pd_img!=null)
            {
                AddImg(document,"坡度分析",info.px_pd_img,pdDesc);
            }
            AddRoleDesc(document,"根据萍乡地形和环境因素，坡度为5级时不允许土地开发。","政策解读");

        }
    }

    private void SummarizeAnalysis(Document document,ProjectInfo info,ResultInfo result,boolean px) throws DocumentException
    {
        Paragraph chapter4Pg = new Paragraph("四、总体分析", chapterFont);
        chapter4Pg.setExtraParagraphSpace(30);
        chapter4Pg.setSpacingBefore(pgSpace);
        chapter4Pg.setLeading(lineSpace);
        document.add(chapter4Pg);
        String content = "根据已有数据分析结果，拟建设项目区";
        if(!px) {
            boolean legal = result.passJbnt == EnumAnalyzeResult.AllPass && result.passTdgh == EnumAnalyzeResult.AllPass && result.passSpba == EnumAnalyzeResult.AllPass && result.passZrbhq == EnumAnalyzeResult.AllPass;
            boolean suitable = result.passDzfx == EnumAnalyzeResult.AllPass && result.passSd == EnumAnalyzeResult.AllPass;
            boolean unlegal = result.passJbnt == EnumAnalyzeResult.NoPass || result.passTdgh == EnumAnalyzeResult.NoPass || result.passSpba == EnumAnalyzeResult.NoPass || result.passZrbhq == EnumAnalyzeResult.NoPass;
            boolean unsuitable = result.passDzfx == EnumAnalyzeResult.NoPass || result.passSd == EnumAnalyzeResult.NoPass;



            boolean allPass = legal && suitable;
            if (allPass) {
                content += "合法、适宜。";
                content += "不涉及占用永久基本农田、";
                content += "符合土地利用总体规划、";
                content += "有合法建设用地审批手续、";
                content += "不涉及占用自然保护地、";
                content += "不存在地质灾害隐患、";
                content += "不占用湿地。";
            } else if (unlegal) {
                content += "不合法、不适宜。";
                switch (result.passJbnt) {
                    case NoPass:
                        content += "涉及占用永久基本农田、";
                        break;
                    case PartPass:
                        content += "部分区域涉及占用永久基本农田、";
                        break;
                }
                switch (result.passTdgh) {
                    case NoPass:
                        content += "不符合土地利用总体规划、";
                        break;
                    case PartPass:
                        content += "部分区域不符合土地利用总体规划、";
                        break;
                }
                switch (result.passSpba) {
                    case NoPass:
                        content += "没有合法建设用地审批手续、";
                        break;
                    case PartPass:
                        content += "部分区域没有合法建设用地审批手续、";
                        break;
                }
                switch (result.passZrbhq) {
                    case NoPass:
                        content += "涉及占用自然保护地、";
                        break;
                    case PartPass:
                        content += "部分区域涉及占用自然保护地、";
                        break;
                }
                switch (result.passDzfx) {
                    case NoPass:
                        content += "存在地质灾害隐患、";
                        break;
                    case PartPass:
                        content += "部分区域存在地质灾害隐患、";
                        break;
                }
                switch (result.passSd) {
                    case NoPass:
                        content += "占用湿地、";
                        break;
                    case PartPass:
                        content += "部分区域占用湿地、";
                        break;
                }
                content = content.substring(0, content.length() - 1) + "。";
            } else if (unsuitable) {
                content += "不适宜。";
                switch (result.passDzfx) {
                    case NoPass:
                        content += "存在地质灾害隐患、";
                        break;
                    case PartPass:
                        content += "部分区域存在地质灾害隐患、";
                        break;
                }
                switch (result.passSd) {
                    case NoPass:
                        content += "占用湿地、";
                        break;
                    case PartPass:
                        content += "部分区域占用湿地、";
                        break;
                }
                content = content.substring(0, content.length() - 1) + "。";
            } else {
                boolean partlegal = false;
                boolean partsuitable = false;
                String tmp = "";
                if (result.passJbnt == EnumAnalyzeResult.PartPass) {
                    tmp += "涉及占用永久基本农田、";
                    partlegal = true;
                }
                if (result.passTdgh == EnumAnalyzeResult.PartPass) {
                    tmp += "不符合土地利用总体规划、";
                    partlegal = true;
                }
                if (result.passSpba == EnumAnalyzeResult.PartPass) {
                    tmp += "没有合法建设用地审批手续、";
                    partlegal = true;
                }
                if (result.passZrbhq == EnumAnalyzeResult.PartPass) {
                    tmp += "涉及占用自然保护地、";
                    partlegal = true;
                }
                if (result.passDzfx == EnumAnalyzeResult.PartPass) {
                    tmp += "存在地质灾害隐患、";
                    partsuitable = true;
                }
                if (result.passSd == EnumAnalyzeResult.PartPass) {
                    tmp += "占用湿地、";
                    partsuitable = true;
                }
                tmp = tmp.substring(0, tmp.length() - 1) + "。";
                if (partlegal && partsuitable) {
                    content += "存在不符合法律法规、不适宜的情况。比如部分地区";
                } else if (partlegal) {
                    content += "存在不符合法律法规的情况。比如部分地区";
                } else if (partsuitable) {
                    content += "存在不适宜的情况。比如部分地区";
                }
                content += tmp;
            }
        }
        else
        {
            boolean legal = result.passJbnt == EnumAnalyzeResult.AllPass && result.passTdgh == EnumAnalyzeResult.AllPass && result.passSpba == EnumAnalyzeResult.AllPass && result.passZrbhq == EnumAnalyzeResult.AllPass;
            boolean suitable = result.passDzfx == EnumAnalyzeResult.AllPass && result.passSd == EnumAnalyzeResult.AllPass&&result.passPxSthx == EnumAnalyzeResult.AllPass&&result.passPxGyl == EnumAnalyzeResult.AllPass&&result.passPxPd == EnumAnalyzeResult.AllPass;
            boolean unlegal = result.passJbnt == EnumAnalyzeResult.NoPass || result.passTdgh == EnumAnalyzeResult.NoPass || result.passSpba == EnumAnalyzeResult.NoPass || result.passZrbhq == EnumAnalyzeResult.NoPass;
            boolean unsuitable = result.passDzfx == EnumAnalyzeResult.NoPass || result.passSd == EnumAnalyzeResult.NoPass||result.passPxSthx == EnumAnalyzeResult.NoPass||result.passPxGyl == EnumAnalyzeResult.NoPass||result.passPxPd == EnumAnalyzeResult.NoPass;

            boolean allPass = legal && suitable;
            if (allPass) {
                content += "合法、适宜。";
                content += "不涉及占用永久基本农田、";
                content += "符合土地利用总体规划、";
                content += "有合法建设用地审批手续、";
                content += "不涉及占用自然保护地、";
                content += "不存在地质灾害隐患、";
                content += "不占用湿地、";
                content += "不占用生态保护红线、";
                content += "不占用公益林、";
                content += "不涉及5级坡度。";
            } else if (unlegal) {
                content += "不合法、不适宜。";
                switch (result.passJbnt) {
                    case NoPass:
                        content += "涉及占用永久基本农田、";
                        break;
                    case PartPass:
                        content += "部分区域涉及占用永久基本农田、";
                        break;
                }
                switch (result.passTdgh) {
                    case NoPass:
                        content += "不符合土地利用总体规划、";
                        break;
                    case PartPass:
                        content += "部分区域不符合土地利用总体规划、";
                        break;
                }
                switch (result.passSpba) {
                    case NoPass:
                        content += "没有合法建设用地审批手续、";
                        break;
                    case PartPass:
                        content += "部分区域没有合法建设用地审批手续、";
                        break;
                }
                switch (result.passZrbhq) {
                    case NoPass:
                        content += "涉及占用自然保护地、";
                        break;
                    case PartPass:
                        content += "部分区域涉及占用自然保护地、";
                        break;
                }
                switch (result.passDzfx) {
                    case NoPass:
                        content += "存在地质灾害隐患、";
                        break;
                    case PartPass:
                        content += "部分区域存在地质灾害隐患、";
                        break;
                }
                switch (result.passSd) {
                    case NoPass:
                        content += "占用湿地、";
                        break;
                    case PartPass:
                        content += "部分区域占用湿地、";
                        break;
                }
                switch (result.passPxSthx) {
                    case NoPass:
                        content += "占用生态保护红线、";
                        break;
                    case PartPass:
                        content += "部分区域占用生态保护红线、";
                        break;
                }
                switch (result.passPxGyl) {
                    case NoPass:
                        content += "占用公益林、";
                        break;
                    case PartPass:
                        content += "部分区域占用公益林、";
                        break;
                }
                switch (result.passPxPd) {
                    case NoPass:
                        content += "涉及5级坡度、";
                        break;
                    case PartPass:
                        content += "部分区域涉及5级坡度、";
                        break;
                }
                content = content.substring(0, content.length() - 1) + "。";
            } else if (unsuitable) {
                content += "不适宜。";
                switch (result.passDzfx) {
                    case NoPass:
                        content += "存在地质灾害隐患、";
                        break;
                    case PartPass:
                        content += "部分区域存在地质灾害隐患、";
                        break;
                }
                switch (result.passSd) {
                    case NoPass:
                        content += "占用湿地、";
                        break;
                    case PartPass:
                        content += "部分区域占用湿地、";
                        break;
                }
                switch (result.passPxSthx) {
                    case NoPass:
                        content += "占用生态保护红线、";
                        break;
                    case PartPass:
                        content += "部分区域占用生态保护红线、";
                        break;
                }
                switch (result.passPxGyl) {
                    case NoPass:
                        content += "占用公益林、";
                        break;
                    case PartPass:
                        content += "部分区域占用公益林、";
                        break;
                }
                switch (result.passPxPd) {
                    case NoPass:
                        content += "涉及5级坡度、";
                        break;
                    case PartPass:
                        content += "部分区域涉及5级坡度、";
                        break;
                }
                content = content.substring(0, content.length() - 1) + "。";
            } else {
                boolean partlegal = false;
                boolean partsuitable = false;
                String tmp = "";
                if (result.passJbnt == EnumAnalyzeResult.PartPass) {
                    tmp += "涉及占用永久基本农田、";
                    partlegal = true;
                }
                if (result.passTdgh == EnumAnalyzeResult.PartPass) {
                    tmp += "不符合土地利用总体规划、";
                    partlegal = true;
                }
                if (result.passSpba == EnumAnalyzeResult.PartPass) {
                    tmp += "没有合法建设用地审批手续、";
                    partlegal = true;
                }
                if (result.passZrbhq == EnumAnalyzeResult.PartPass) {
                    tmp += "涉及占用自然保护地、";
                    partlegal = true;
                }
                if (result.passDzfx == EnumAnalyzeResult.PartPass) {
                    tmp += "存在地质灾害隐患、";
                    partsuitable = true;
                }
                if (result.passSd == EnumAnalyzeResult.PartPass) {
                    tmp += "占用湿地、";
                    partsuitable = true;
                }
                if (result.passPxSthx == EnumAnalyzeResult.PartPass) {
                    tmp += "占用生态保护红线、";
                    partsuitable = true;
                }
                if (result.passPxGyl == EnumAnalyzeResult.PartPass) {
                    tmp += "占用公益林、";
                    partsuitable = true;
                }
                if (result.passPxPd == EnumAnalyzeResult.PartPass) {
                    tmp += "涉及5级坡度、";
                    partsuitable = true;
                }
                tmp = tmp.substring(0, tmp.length() - 1) + "。";
                if (partlegal && partsuitable) {
                    content += "存在不符合法律法规、不适宜的情况。比如部分地区";
                } else if (partlegal) {
                    content += "存在不符合法律法规的情况。比如部分地区";
                } else if (partsuitable) {
                    content += "存在不适宜的情况。比如部分地区";
                }
                content += tmp;
            }
        }

        Paragraph contentPg = new Paragraph(content, normalFont);
        contentPg.setFirstLineIndent(firstLineIndent);
        contentPg.setExtraParagraphSpace(30);
        contentPg.setSpacingBefore(pgSpace);
        contentPg.setLeading(lineSpace);
        document.add(contentPg);
    }

    private void SummarizeAnalysisLawless(Document document,LawlessInfo info,ResultInfo result) throws DocumentException
    {
        Paragraph chapter4Pg = new Paragraph("四、总体分析", chapterFont);
        chapter4Pg.setExtraParagraphSpace(30);
        chapter4Pg.setSpacingBefore(pgSpace);
        chapter4Pg.setLeading(lineSpace);
        document.add(chapter4Pg);
        boolean legal = result.passJbnt == EnumAnalyzeResult.AllPass &&result.passTdgh == EnumAnalyzeResult.AllPass&&result.passSpba == EnumAnalyzeResult.AllPass &&result.passZrbhq== EnumAnalyzeResult.AllPass;
        String content="根据已有数据分析结果，该遥感监测图斑";
        content+=legal?"属于合法建设用地":"属于违法建设用地";
        content+="，";

        content+=(result.passJbnt!= EnumAnalyzeResult.AllPass ? "涉及":"不涉及")+"占用永久基本农田、";
        content+=(result.passTdgh!= EnumAnalyzeResult.AllPass?"不符合":"符合")+"土地利用总体规划、";
        switch(result.passSpba)
        {
            case AllPass:
                content+="有合法建设用地审批手续、";
                break;
            case PartPass:
                content+="部分区域没有合法建设用地审批手续、";
                break;
            case NoPass:
                content+="没有合法建设用地审批手续、";
                break;
        }

        content+=(result.passZrbhq!= EnumAnalyzeResult.AllPass?"涉及":"不涉及")+"占用自然保护地，";
        if(!legal) {
            content += "建议变更为临时用地或设施农用地。";
        }
        else
        {
            content = content.substring(0, content.length() - 1) + "。";
        }


        Paragraph contentPg = new Paragraph(content, normalFont);
        contentPg.setFirstLineIndent(firstLineIndent);
        contentPg.setExtraParagraphSpace(30);
        contentPg.setSpacingBefore(pgSpace);
        contentPg.setLeading(lineSpace);
        document.add(contentPg);




    }

    private void AddBaseImg(Document document,ProjectInfo info) throws DocumentException {
        int col=2;
        if(info.narrow)
        {
            col = 1;
            PdfPTable t1 = new PdfPTable(col);//创建一个1列表格对象
            t1.setSpacingBefore(pgSpace);//设置上边距
            t1.setSpacingAfter(pgSpace);//设置下边距
            t1.setWidthPercentage(80);

            PdfPTable t2 = new PdfPTable(col);//创建一个1列表格对象
            t2.setSpacingBefore(pgSpace);//设置上边距
            t2.setSpacingAfter(pgSpace);//设置下边距
            t2.setWidthPercentage(80);

            PdfPCell c1 = null;
            c1 = new PdfPCell(new Phrase("项目区位置", normalFont));
            c1.setVerticalAlignment(Element.ALIGN_CENTER);
            c1.setHorizontalAlignment(Element.ALIGN_CENTER);
            c1.setPaddingTop(5);
            c1.setPaddingBottom(5);

            PdfPCell c2 = null;
            c2 = new PdfPCell(new Phrase("土地利用现状图", normalFont));
            c2.setVerticalAlignment(Element.ALIGN_CENTER);
            c2.setHorizontalAlignment(Element.ALIGN_CENTER);
            c2.setPaddingTop(5);
            c2.setPaddingBottom(5);

            PdfPCell cimg1 = null;

            float width = 250;
            float heigth = 250;
            if (info.ownershipImg != null) {
                heigth = info.ownershipImg.getHeight() / (info.ownershipImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.ownershipImg.scaleToFit(width, heigth);
                cimg1 = new PdfPCell(info.ownershipImg);
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            } else {
                cimg1 = new PdfPCell();
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            }
            PdfPCell cimg2 = null;
            if (info.landtypeImg != null) {
                heigth = info.landtypeImg.getHeight() / (info.landtypeImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.landtypeImg.scaleToFit(width, heigth);
                cimg2 = new PdfPCell(info.landtypeImg);
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            } else {
                cimg2 = new PdfPCell();
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            }
            PdfPCell meta1 = null;
            meta1 = new PdfPCell(new Phrase(info.ownershipImgMeta, roleFont));
            meta1.setVerticalAlignment(Element.ALIGN_CENTER);
            meta1.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta1.setPaddingTop(5);
            meta1.setPaddingBottom(5);


            PdfPCell meta2 = null;
            meta2 = new PdfPCell(new Phrase(info.landtypeImgMeta, roleFont));
            meta2.setVerticalAlignment(Element.ALIGN_CENTER);
            meta2.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta2.setPaddingTop(5);
            meta2.setPaddingBottom(5);

            if (info.ownershipImg != null) {
                t1.addCell(c1);
                t1.addCell(cimg1);
                t1.addCell(meta1);

            }
            if (info.landtypeImg != null) {
                t2.addCell(c2);
                t2.addCell(cimg2);
                t2.addCell(meta2);

            }

            if (info.photo1 != null || info.photo2 != null) {
                PdfPCell photo1 = new PdfPCell(new Phrase("实地照片1", normalFont));
                photo1.setVerticalAlignment(Element.ALIGN_CENTER);
                photo1.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo1.setPaddingTop(5);
                photo1.setPaddingBottom(5);


                PdfPCell photo2 = new PdfPCell(new Phrase("实地照片2", normalFont));
                photo2.setVerticalAlignment(Element.ALIGN_CENTER);
                photo2.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo2.setPaddingTop(5);
                photo2.setPaddingBottom(5);


                PdfPCell cp1 = null;
                if (info.photo1 != null) {
                    Image img1 = Image.getInstance(info.photo1);
                    heigth = img1.getHeight() / (img1.getWidth() / width);
                    img1.scaleToFit(width, heigth);
                    cp1 = new PdfPCell(img1);
                } else {
                    cp1 = new PdfPCell();
                }
                cp1.setVerticalAlignment(Element.ALIGN_CENTER);
                cp1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp1.setPaddingTop(5);
                cp1.setPaddingBottom(5);
                cp1.setPaddingLeft(5);
                cp1.setPaddingRight(5);


                PdfPCell cp2 = null;
                if (info.photo2 != null) {
                    Image img2 = Image.getInstance(info.photo2);
                    heigth = img2.getHeight() / (img2.getWidth() / width);
                    img2.scaleToFit(width, heigth);
                    cp2 = new PdfPCell(img2);
                } else {
                    cp2 = new PdfPCell();
                }
                cp2.setVerticalAlignment(Element.ALIGN_CENTER);
                cp2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp2.setPaddingTop(5);
                cp2.setPaddingBottom(5);
                cp2.setPaddingLeft(5);
                cp2.setPaddingRight(5);

                PdfPCell pmeta1 = new PdfPCell(new Phrase(info.photoMeta1, normalFont));
                pmeta1.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta1.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta1.setPaddingTop(5);
                pmeta1.setPaddingBottom(5);

                PdfPCell pmeta2 = new PdfPCell(new Phrase(info.photoMeta2, normalFont));
                pmeta2.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta2.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta2.setPaddingTop(5);
                pmeta2.setPaddingBottom(5);

                t2.addCell(photo1);
                t2.addCell(photo2);
                t2.addCell(cp1);
                t2.addCell(cp2);
                t2.addCell(pmeta1);
                t2.addCell(pmeta2);


            }

            document.add(t1);
            document.newPage();
            document.add(t2);
        }
        else {
            PdfPTable t = new PdfPTable(col);//创建一个2列表格对象
            t.setSpacingBefore(pgSpace);//设置上边距
            t.setSpacingAfter(pgSpace);//设置下边距
            t.setWidthPercentage(100);

            PdfPCell c1 = null;
            c1 = new PdfPCell(new Phrase("项目区位置", normalFont));
            c1.setVerticalAlignment(Element.ALIGN_CENTER);
            c1.setHorizontalAlignment(Element.ALIGN_CENTER);
            c1.setPaddingTop(5);
            c1.setPaddingBottom(5);

            PdfPCell c2 = null;
            c2 = new PdfPCell(new Phrase("土地利用现状图", normalFont));
            c2.setVerticalAlignment(Element.ALIGN_CENTER);
            c2.setHorizontalAlignment(Element.ALIGN_CENTER);
            c2.setPaddingTop(5);
            c2.setPaddingBottom(5);

            PdfPCell cimg1 = null;

            float width = 250;
            float heigth = 250;
            if (info.ownershipImg != null) {
                heigth = info.ownershipImg.getHeight() / (info.ownershipImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.ownershipImg.scaleToFit(width, heigth);
                cimg1 = new PdfPCell(info.ownershipImg);
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            } else {
                cimg1 = new PdfPCell();
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            }
            PdfPCell cimg2 = null;
            if (info.landtypeImg != null) {
                heigth = info.landtypeImg.getHeight() / (info.landtypeImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.landtypeImg.scaleToFit(width, heigth);
                cimg2 = new PdfPCell(info.landtypeImg);
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            } else {
                cimg2 = new PdfPCell();
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            }
            PdfPCell meta1 = null;
            meta1 = new PdfPCell(new Phrase(info.ownershipImgMeta, roleFont));
            meta1.setVerticalAlignment(Element.ALIGN_CENTER);
            meta1.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta1.setPaddingTop(5);
            meta1.setPaddingBottom(5);


            PdfPCell meta2 = null;
            meta2 = new PdfPCell(new Phrase(info.landtypeImgMeta, roleFont));
            meta2.setVerticalAlignment(Element.ALIGN_CENTER);
            meta2.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta2.setPaddingTop(5);
            meta2.setPaddingBottom(5);
            t.addCell(c1);
            t.addCell(c2);
            t.addCell(cimg1);
            t.addCell(cimg2);
            t.addCell(meta1);
            t.addCell(meta2);

            if (info.photo1 != null || info.photo2 != null) {
                PdfPCell photo1 = new PdfPCell(new Phrase("实地照片1", normalFont));
                photo1.setVerticalAlignment(Element.ALIGN_CENTER);
                photo1.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo1.setPaddingTop(5);
                photo1.setPaddingBottom(5);


                PdfPCell photo2 = new PdfPCell(new Phrase("实地照片2", normalFont));
                photo2.setVerticalAlignment(Element.ALIGN_CENTER);
                photo2.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo2.setPaddingTop(5);
                photo2.setPaddingBottom(5);


                PdfPCell cp1 = null;
                if (info.photo1 != null) {
                    Image img1 = Image.getInstance(info.photo1);
                    heigth = img1.getHeight() / (img1.getWidth() / width);
                    img1.scaleToFit(width, heigth);
                    cp1 = new PdfPCell(img1);
                } else {
                    cp1 = new PdfPCell();
                }
                cp1.setVerticalAlignment(Element.ALIGN_CENTER);
                cp1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp1.setPaddingTop(5);
                cp1.setPaddingBottom(5);
                cp1.setPaddingLeft(5);
                cp1.setPaddingRight(5);


                PdfPCell cp2 = null;
                if (info.photo2 != null) {
                    Image img2 = Image.getInstance(info.photo2);
                    heigth = img2.getHeight() / (img2.getWidth() / width);
                    img2.scaleToFit(width, heigth);
                    cp2 = new PdfPCell(img2);
                } else {
                    cp2 = new PdfPCell();
                }
                cp2.setVerticalAlignment(Element.ALIGN_CENTER);
                cp2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp2.setPaddingTop(5);
                cp2.setPaddingBottom(5);
                cp2.setPaddingLeft(5);
                cp2.setPaddingRight(5);

                PdfPCell pmeta1 = new PdfPCell(new Phrase(info.photoMeta1, normalFont));
                pmeta1.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta1.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta1.setPaddingTop(5);
                pmeta1.setPaddingBottom(5);

                PdfPCell pmeta2 = new PdfPCell(new Phrase(info.photoMeta2, normalFont));
                pmeta2.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta2.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta2.setPaddingTop(5);
                pmeta2.setPaddingBottom(5);

                t.addCell(photo1);
                t.addCell(photo2);
                t.addCell(cp1);
                t.addCell(cp2);
                t.addCell(pmeta1);
                t.addCell(pmeta2);


            }


            document.add(t);
        }
    }

    private void AddBaseImgLawless(Document document,LawlessInfo info) throws DocumentException {
        if(info.ownershipImg!=null) {
            AddImg(document,"位置图", info.ownershipImg, info.ownershipImgMeta);
        }
    }

    private void AddBaseImgLawless2(Document document,LawlessInfo info) throws DocumentException {
        int col=2;
        if(info.narrow)
        {
            col = 1;
            PdfPTable t1 = new PdfPTable(col);//创建一个1列表格对象
            t1.setSpacingBefore(pgSpace);//设置上边距
            t1.setSpacingAfter(pgSpace);//设置下边距
            t1.setWidthPercentage(80);

            PdfPTable t2 = new PdfPTable(col);//创建一个1列表格对象
            t2.setSpacingBefore(pgSpace);//设置上边距
            t2.setSpacingAfter(pgSpace);//设置下边距
            t2.setWidthPercentage(80);

            PdfPCell c1 = null;
            c1 = new PdfPCell(new Phrase("项目区位置", normalFont));
            c1.setVerticalAlignment(Element.ALIGN_CENTER);
            c1.setHorizontalAlignment(Element.ALIGN_CENTER);
            c1.setPaddingTop(5);
            c1.setPaddingBottom(5);

            PdfPCell c2 = null;
            c2 = new PdfPCell(new Phrase("土地利用现状图", normalFont));
            c2.setVerticalAlignment(Element.ALIGN_CENTER);
            c2.setHorizontalAlignment(Element.ALIGN_CENTER);
            c2.setPaddingTop(5);
            c2.setPaddingBottom(5);

            PdfPCell cimg1 = null;

            float width = 250;
            float heigth = 250;
            if (info.ownershipImg != null) {
                heigth = info.ownershipImg.getHeight() / (info.ownershipImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.ownershipImg.scaleToFit(width, heigth);
                cimg1 = new PdfPCell(info.ownershipImg);
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            } else {
                cimg1 = new PdfPCell();
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            }
            PdfPCell cimg2 = null;
            if (info.landtypeImg != null) {
                heigth = info.landtypeImg.getHeight() / (info.landtypeImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.landtypeImg.scaleToFit(width, heigth);
                cimg2 = new PdfPCell(info.landtypeImg);
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            } else {
                cimg2 = new PdfPCell();
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            }
            PdfPCell meta1 = null;
            meta1 = new PdfPCell(new Phrase(info.ownershipImgMeta, roleFont));
            meta1.setVerticalAlignment(Element.ALIGN_CENTER);
            meta1.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta1.setPaddingTop(5);
            meta1.setPaddingBottom(5);


            PdfPCell meta2 = null;
            meta2 = new PdfPCell(new Phrase(info.landtypeImgMeta, roleFont));
            meta2.setVerticalAlignment(Element.ALIGN_CENTER);
            meta2.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta2.setPaddingTop(5);
            meta2.setPaddingBottom(5);

            if (info.ownershipImg != null) {
                t1.addCell(c1);
                t1.addCell(cimg1);
                t1.addCell(meta1);

            }
            if (info.landtypeImg != null) {
                t2.addCell(c2);
                t2.addCell(cimg2);
                t2.addCell(meta2);

            }

            if (info.photo1 != null || info.photo2 != null) {
                PdfPCell photo1 = new PdfPCell(new Phrase("实地照片1", normalFont));
                photo1.setVerticalAlignment(Element.ALIGN_CENTER);
                photo1.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo1.setPaddingTop(5);
                photo1.setPaddingBottom(5);


                PdfPCell photo2 = new PdfPCell(new Phrase("实地照片2", normalFont));
                photo2.setVerticalAlignment(Element.ALIGN_CENTER);
                photo2.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo2.setPaddingTop(5);
                photo2.setPaddingBottom(5);


                PdfPCell cp1 = null;
                if (info.photo1 != null) {
                    Image img1 = Image.getInstance(info.photo1);
                    heigth = img1.getHeight() / (img1.getWidth() / width);
                    img1.scaleToFit(width, heigth);
                    cp1 = new PdfPCell(img1);
                } else {
                    cp1 = new PdfPCell();
                }
                cp1.setVerticalAlignment(Element.ALIGN_CENTER);
                cp1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp1.setPaddingTop(5);
                cp1.setPaddingBottom(5);
                cp1.setPaddingLeft(5);
                cp1.setPaddingRight(5);


                PdfPCell cp2 = null;
                if (info.photo2 != null) {
                    Image img2 = Image.getInstance(info.photo2);
                    heigth = img2.getHeight() / (img2.getWidth() / width);
                    img2.scaleToFit(width, heigth);
                    cp2 = new PdfPCell(img2);
                } else {
                    cp2 = new PdfPCell();
                }
                cp2.setVerticalAlignment(Element.ALIGN_CENTER);
                cp2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp2.setPaddingTop(5);
                cp2.setPaddingBottom(5);
                cp2.setPaddingLeft(5);
                cp2.setPaddingRight(5);

                PdfPCell pmeta1 = new PdfPCell(new Phrase(info.photoMeta1, normalFont));
                pmeta1.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta1.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta1.setPaddingTop(5);
                pmeta1.setPaddingBottom(5);

                PdfPCell pmeta2 = new PdfPCell(new Phrase(info.photoMeta2, normalFont));
                pmeta2.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta2.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta2.setPaddingTop(5);
                pmeta2.setPaddingBottom(5);

                t2.addCell(photo1);
                t2.addCell(photo2);
                t2.addCell(cp1);
                t2.addCell(cp2);
                t2.addCell(pmeta1);
                t2.addCell(pmeta2);


            }

            document.add(t1);
            document.newPage();
            document.add(t2);
        }
        else {
            PdfPTable t = new PdfPTable(col);//创建一个2列表格对象
            t.setSpacingBefore(pgSpace);//设置上边距
            t.setSpacingAfter(pgSpace);//设置下边距
            t.setWidthPercentage(100);

            PdfPCell c1 = null;
            c1 = new PdfPCell(new Phrase("项目区位置", normalFont));
            c1.setVerticalAlignment(Element.ALIGN_CENTER);
            c1.setHorizontalAlignment(Element.ALIGN_CENTER);
            c1.setPaddingTop(5);
            c1.setPaddingBottom(5);

            PdfPCell c2 = null;
            c2 = new PdfPCell(new Phrase("土地利用现状图", normalFont));
            c2.setVerticalAlignment(Element.ALIGN_CENTER);
            c2.setHorizontalAlignment(Element.ALIGN_CENTER);
            c2.setPaddingTop(5);
            c2.setPaddingBottom(5);

            PdfPCell cimg1 = null;

            float width = 250;
            float heigth = 250;
            if (info.ownershipImg != null) {
                heigth = info.ownershipImg.getHeight() / (info.ownershipImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.ownershipImg.scaleToFit(width, heigth);
                cimg1 = new PdfPCell(info.ownershipImg);
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            } else {
                cimg1 = new PdfPCell();
                cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg1.setPaddingTop(5);
                cimg1.setPaddingBottom(5);
                cimg1.setPaddingLeft(5);
                cimg1.setPaddingRight(5);
            }
            PdfPCell cimg2 = null;
            if (info.landtypeImg != null) {
                heigth = info.landtypeImg.getHeight() / (info.landtypeImg.getWidth() / width);
                System.out.println("width:" + width);
                System.out.println("heigth:" + heigth);
                info.landtypeImg.scaleToFit(width, heigth);
                cimg2 = new PdfPCell(info.landtypeImg);
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            } else {
                cimg2 = new PdfPCell();
                cimg2.setVerticalAlignment(Element.ALIGN_CENTER);
                cimg2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cimg2.setPaddingTop(5);
                cimg2.setPaddingBottom(5);
                cimg2.setPaddingLeft(5);
                cimg2.setPaddingRight(5);
            }
            PdfPCell meta1 = null;
            meta1 = new PdfPCell(new Phrase(info.ownershipImgMeta, roleFont));
            meta1.setVerticalAlignment(Element.ALIGN_CENTER);
            meta1.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta1.setPaddingTop(5);
            meta1.setPaddingBottom(5);


            PdfPCell meta2 = null;
            meta2 = new PdfPCell(new Phrase(info.landtypeImgMeta, roleFont));
            meta2.setVerticalAlignment(Element.ALIGN_CENTER);
            meta2.setHorizontalAlignment(Element.ALIGN_CENTER);
            meta2.setPaddingTop(5);
            meta2.setPaddingBottom(5);
            t.addCell(c1);
            t.addCell(c2);
            t.addCell(cimg1);
            t.addCell(cimg2);
            t.addCell(meta1);
            t.addCell(meta2);

            if (info.photo1 != null || info.photo2 != null) {
                PdfPCell photo1 = new PdfPCell(new Phrase("实地照片1", normalFont));
                photo1.setVerticalAlignment(Element.ALIGN_CENTER);
                photo1.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo1.setPaddingTop(5);
                photo1.setPaddingBottom(5);


                PdfPCell photo2 = new PdfPCell(new Phrase("实地照片2", normalFont));
                photo2.setVerticalAlignment(Element.ALIGN_CENTER);
                photo2.setHorizontalAlignment(Element.ALIGN_CENTER);
                photo2.setPaddingTop(5);
                photo2.setPaddingBottom(5);


                PdfPCell cp1 = null;
                if (info.photo1 != null) {
                    Image img1 = Image.getInstance(info.photo1);
                    heigth = img1.getHeight() / (img1.getWidth() / width);
                    img1.scaleToFit(width, heigth);
                    cp1 = new PdfPCell(img1);
                } else {
                    cp1 = new PdfPCell();
                }
                cp1.setVerticalAlignment(Element.ALIGN_CENTER);
                cp1.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp1.setPaddingTop(5);
                cp1.setPaddingBottom(5);
                cp1.setPaddingLeft(5);
                cp1.setPaddingRight(5);


                PdfPCell cp2 = null;
                if (info.photo2 != null) {
                    Image img2 = Image.getInstance(info.photo2);
                    heigth = img2.getHeight() / (img2.getWidth() / width);
                    img2.scaleToFit(width, heigth);
                    cp2 = new PdfPCell(img2);
                } else {
                    cp2 = new PdfPCell();
                }
                cp2.setVerticalAlignment(Element.ALIGN_CENTER);
                cp2.setHorizontalAlignment(Element.ALIGN_CENTER);
                cp2.setPaddingTop(5);
                cp2.setPaddingBottom(5);
                cp2.setPaddingLeft(5);
                cp2.setPaddingRight(5);

                PdfPCell pmeta1 = new PdfPCell(new Phrase(info.photoMeta1, normalFont));
                pmeta1.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta1.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta1.setPaddingTop(5);
                pmeta1.setPaddingBottom(5);

                PdfPCell pmeta2 = new PdfPCell(new Phrase(info.photoMeta2, normalFont));
                pmeta2.setVerticalAlignment(Element.ALIGN_CENTER);
                pmeta2.setHorizontalAlignment(Element.ALIGN_CENTER);
                pmeta2.setPaddingTop(5);
                pmeta2.setPaddingBottom(5);

                t.addCell(photo1);
                t.addCell(photo2);
                t.addCell(cp1);
                t.addCell(cp2);
                t.addCell(pmeta1);
                t.addCell(pmeta2);


            }


            document.add(t);
        }
    }

    private void AddImg(Document document,String title,Image img,String desc) throws DocumentException
    {
        PdfPTable t = new PdfPTable(1);
        //t.setSplitLate(false);
        //t.setSplitRows(true);
        t.setKeepTogether(true);
        t.setSpacingBefore(pgSpace);//设置上边距
        t.setSpacingAfter(pgSpace);//设置下边距
        t.setWidthPercentage(80);

        PdfPCell titleCell = new PdfPCell(new Phrase(title,normalFont));
        titleCell.setVerticalAlignment(Element.ALIGN_CENTER);
        titleCell.setHorizontalAlignment(Element.ALIGN_CENTER);
        titleCell.setPaddingTop(5);
        titleCell.setPaddingBottom(5);
        t.addCell(titleCell);

        float width = 250;
        float heigth =img.getHeight()/(img.getWidth()/width);
        img.scaleToFit(width, heigth);
        PdfPCell cimg1 = new PdfPCell(img);
        cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
        cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
        cimg1.setPaddingTop(5);
        cimg1.setPaddingBottom(5);
        cimg1.setPaddingLeft(5);
        cimg1.setPaddingRight(5);
        t.addCell(cimg1);

        PdfPCell descCell = new PdfPCell(new Phrase(desc,roleFont));
        descCell.setVerticalAlignment(Element.ALIGN_CENTER);
        descCell.setHorizontalAlignment(Element.ALIGN_CENTER);
        descCell.setPaddingTop(5);
        descCell.setPaddingBottom(5);
        t.addCell(descCell);
        document.add(t);
    }

    private void AddImg2(Document document,String title,Image img,String desc,boolean first,boolean last) throws DocumentException
    {
        PdfPTable t = new PdfPTable(1);
        //t.setSplitLate(false);
        //t.setSplitRows(true);
        t.setKeepTogether(true);
        if(first) {
            t.setSpacingBefore(pgSpace);//设置上边距
            //t.setSpacingAfter(pgSpace);//设置下边距
        }
        else if(last)
        {
            t.setSpacingAfter(pgSpace);//设置下边距
        }
        t.setWidthPercentage(75);

        PdfPCell titleCell = new PdfPCell(new Phrase(title,normalFont));
        titleCell.setVerticalAlignment(Element.ALIGN_CENTER);
        titleCell.setHorizontalAlignment(Element.ALIGN_CENTER);
        titleCell.setPaddingTop(5);
        titleCell.setPaddingBottom(5);
        t.addCell(titleCell);

        float width = 200;
        float heigth =img.getHeight()/(img.getWidth()/width);
        img.scaleToFit(width, heigth);
        PdfPCell cimg1 = new PdfPCell(img);
        cimg1.setVerticalAlignment(Element.ALIGN_CENTER);
        cimg1.setHorizontalAlignment(Element.ALIGN_CENTER);
        cimg1.setPaddingTop(5);
        cimg1.setPaddingBottom(5);
        cimg1.setPaddingLeft(5);
        cimg1.setPaddingRight(5);
        t.addCell(cimg1);

        PdfPCell descCell = new PdfPCell(new Phrase(desc,roleFont));
        descCell.setVerticalAlignment(Element.ALIGN_CENTER);
        descCell.setHorizontalAlignment(Element.ALIGN_CENTER);
        descCell.setPaddingTop(5);
        descCell.setPaddingBottom(5);
        t.addCell(descCell);
        document.add(t);
    }

    private void AddRoleDesc(Document document,String desc,String role) throws DocumentException
    {
        float spaceBefore = 0;
        if(role!=null&&!role.isEmpty()) {
            String roleInfo = role + "：";
            Paragraph rolePg = new Paragraph(roleInfo, roleFont);
            rolePg.setFirstLineIndent(firstLineIndent);
            rolePg.setExtraParagraphSpace(30);
            rolePg.setSpacingBefore(7);
            rolePg.setLeading(lineSpace);
            document.add(rolePg);
            spaceBefore=5;
        }
        Paragraph descPg = new Paragraph( desc, roleFont);
        descPg.setFirstLineIndent(firstLineIndent);
        descPg.setExtraParagraphSpace(30);
        descPg.setSpacingBefore(spaceBefore);
        descPg.setLeading(lineSpace);
        document.add(descPg);
    }



    private void AddSign(Document document,ProjectInfo info) throws DocumentException {
        String sign="报告人： "+info.userName;
        Paragraph signPg = new Paragraph(sign, normalFont);
        signPg.setAlignment(Element.ALIGN_LEFT);
        if(info.narrow)
        {
            signPg.setFirstLineIndent(200);
        }
        else {
            signPg.setFirstLineIndent(300);
        }
        signPg.setExtraParagraphSpace(30);
        signPg.setSpacingBefore(24);
        signPg.setLeading(22);
        document.add(signPg);


        String timeStr="时  间： "+ new SimpleDateFormat("yyyy-MM-dd").format(new Date()).toString();
        Paragraph timePg = new Paragraph(timeStr, normalFont);
        timePg.setAlignment(Element.ALIGN_LEFT);
        if(info.narrow)
        {
            timePg.setFirstLineIndent(200);
        }
        else {
            timePg.setFirstLineIndent(300);
        }
        timePg.setExtraParagraphSpace(30);
        timePg.setSpacingBefore(24);
        timePg.setLeading(22);
        document.add(timePg);

    }

    private void AddSign(Document document,LawlessInfo info) throws DocumentException {
        String sign="报告人： "+info.userName;
        Paragraph signPg = new Paragraph(sign, normalFont);
        signPg.setAlignment(Element.ALIGN_LEFT);
        if(info.narrow)
        {
            signPg.setFirstLineIndent(200);
        }
        else {
            signPg.setFirstLineIndent(300);
        }
        signPg.setExtraParagraphSpace(30);
        signPg.setSpacingBefore(24);
        signPg.setLeading(22);
        document.add(signPg);


        String timeStr="时  间： "+ new SimpleDateFormat("yyyy-MM-dd").format(new Date()).toString();
        Paragraph timePg = new Paragraph(timeStr, normalFont);
        timePg.setAlignment(Element.ALIGN_LEFT);
        if(info.narrow)
        {
            timePg.setFirstLineIndent(200);
        }
        else {
            timePg.setFirstLineIndent(300);
        }
        timePg.setExtraParagraphSpace(30);
        timePg.setSpacingBefore(24);
        timePg.setLeading(22);
        document.add(timePg);

    }

    private void AddSign(Document document,YnbhInfo info) throws DocumentException {
        String sign="报告人： "+info.userName;
        Paragraph signPg = new Paragraph(sign, normalFont);
        signPg.setAlignment(Element.ALIGN_LEFT);
        if(info.narrow)
        {
            signPg.setFirstLineIndent(200);
        }
        else {
            signPg.setFirstLineIndent(300);
        }
        signPg.setExtraParagraphSpace(30);
        signPg.setSpacingBefore(24);
        signPg.setLeading(22);
        document.add(signPg);


        String timeStr="时  间： "+ new SimpleDateFormat("yyyy-MM-dd").format(new Date()).toString();
        Paragraph timePg = new Paragraph(timeStr, normalFont);
        timePg.setAlignment(Element.ALIGN_LEFT);
        if(info.narrow)
        {
            timePg.setFirstLineIndent(200);
        }
        else {
            timePg.setFirstLineIndent(300);
        }
        timePg.setExtraParagraphSpace(30);
        timePg.setSpacingBefore(24);
        timePg.setLeading(22);
        document.add(timePg);

    }

    private String GetDirction(double azid)
    {
        String direction="";
        if(azid>0&&azid<90)
        {
            direction="东偏北";
        }
        else if(azid==90)
        {
            direction="正东";
        }
        else if(azid>90&&azid<180)
        {
            direction="东偏南";
        }
        else if(azid==180)
        {
            direction="正南";
        }
        else if(azid>180&&azid<270)
        {
            direction="西偏南";
        }
        else if(azid==270)
        {
            direction="正西";
        }
        else if(azid>270&&azid<360)
        {
            direction="西偏北";
        }
        else if(azid==360)
        {
            direction="正北";
        }
        return direction;
    }



}
