package com.geoway.landteam.landcloud.service.pub;

import com.geoway.landteam.landcloud.common.util.orm.QueryParamUtil;
import com.geoway.landteam.landcloud.common.util.orm.QuerySpecification;
import com.geoway.landteam.landcloud.core.model.base.entity.Region;
import com.geoway.landteam.landcloud.core.model.base.entity.RegionDetail;
import com.geoway.landteam.landcloud.core.model.base.entity.RegionVersion;
import com.geoway.landteam.landcloud.core.model.pub.entity.InterestGroup;
import com.geoway.landteam.landcloud.core.model.pub.entity.InterestPoint;
import com.geoway.landteam.landcloud.core.repository.base.RegionRepository;
import com.geoway.landteam.landcloud.core.repository.base.RegionVersionRepository;
import com.geoway.landteam.landcloud.core.repository.pub.InterestGroupRepository;
import com.geoway.landteam.landcloud.core.repository.pub.InterestPointRepository;
import com.geoway.landteam.landcloud.core.servface.region.RegionService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

@Service
@Transactional(rollbackFor = Exception.class)
public class MInterestPointService {

    @Autowired
	RegionService regionService;
	
	@Autowired
	InterestPointRepository interestPointDao;

	@Autowired
	RegionVersionRepository regionVersionRepository;
	
	@Autowired
	InterestGroupRepository interestGroupDao;
	
	@Autowired
	RegionRepository regionRepository;

	public Page<InterestGroup> getMyInterestGroups(Long userId, int start, int limit) throws Exception{
		String filterParam = "Q_userId_N_EQ=" + userId + ";Q_isDeleted_N_EQ=0";
		String sortParam = "SORT_createTime_DESC";
		Page<InterestGroup> interestGroups = interestGroupDao.findAll(new QuerySpecification<InterestGroup>(filterParam),
				 PageRequest.of(start, limit, QueryParamUtil.parseSortParams(sortParam)));
		for (InterestGroup interestGroup : interestGroups) {
			Page<InterestPoint> interestPoints = getInterestPointsByGroupId(interestGroup.getId(), 0, 1000, "", "");
			interestGroup.setInterestPoints(interestPoints.getContent());
		}
		return interestGroups;
	}

	public InterestGroup createInterestGroup(Long userId, String groupName) throws Exception{
		Page<InterestGroup> myInterestGroups = getMyInterestGroups(userId, 0, 1000);
		for (InterestGroup interestGroup : myInterestGroups) {
			if (interestGroup.getGroupName().equals(groupName)) {
				throw new RuntimeException("兴趣组名重复");
			}
		}
		InterestGroup interestGroup = new InterestGroup();
		interestGroup.setId(UUID.randomUUID().toString());
		interestGroup.setUserId(userId);
		interestGroup.setGroupName(groupName);
		interestGroup.setCreateTime(new Date());
		interestGroup.setIsDeleted(0);
		interestGroup.setIsDefault(0);
		interestGroupDao.save(interestGroup);
		return interestGroup;
	}

	public void deleteInterestGroup(String groupId, Long userid) throws Exception{
		InterestGroup interestGroup = interestGroupDao.gwSearchByPK(groupId);
		if (userid.equals(interestGroup.getUserId())) {
			interestGroup.setIsDeleted(1);
			// 删除对应兴趣组的所有兴趣点
			List<InterestPoint> list = interestPointDao.getInterestPointsByGroupId(groupId);
			if (list.size() > 0) {
				for (InterestPoint interestPoint : list) {
					interestPoint.setIsDeleted(1);
				}
				interestPointDao.saveAll(list);
			}
		} else {
			throw new RuntimeException("操作失败");
		}
	}

	public void renameInterestGroup(String groupId, String groupName, Long userid) {
		InterestGroup interestGroup = interestGroupDao.gwSearchByPK(groupId);
		if (userid.equals(interestGroup.getUserId())) {
			interestGroup.setGroupName(groupName);
			interestGroupDao.save(interestGroup);
		} else {
			throw new RuntimeException("操作失败");
		}
	}

	public InterestPoint createInterestPoint(Long userId, String name, String groupId, String lat, String lon, Integer drawIndex) throws Exception {
		RegionVersion defaultVersion = regionVersionRepository.getRegionVersionByDefault();//获取政区默认版本
		//  根据经纬度获取对应的行政区信息
        double dLon = Double.parseDouble(lon);
        double dLat = Double.parseDouble(lat);
        String wkt = "POINT(" + dLon + " " + dLat + ")";
        InterestPoint interestPoint = new InterestPoint();
      // 查询市级行政区代码
        RegionDetail region = regionService.queryRegionByGeom(wkt, 4490, 2);
        if (region != null) {
            interestPoint.setCity(region.getCode());
            interestPoint.setCityName(region.getName());
        } else {
        	// 查找不到对应的市,先查找所在的区
        	region = regionService.queryRegionByGeom(wkt, 4490, 3);
        	if (region != null) {
        		// 获取其所在的市信息
                interestPoint.setCity(region.getPcode());
                Region parentRegion = regionRepository.findByCodeAndVersion(region.getPcode(),defaultVersion!=null?defaultVersion.getVersion():"2018");
                interestPoint.setCityName(parentRegion.getName());
            } else {
            	throw new RuntimeException("请确认经纬度数据是否正确");
            }
        }
		interestPoint.setId(UUID.randomUUID().toString());
		interestPoint.setUserId(userId);
		interestPoint.setName(name);
		interestPoint.setDrawIndex(drawIndex);
		if (!"default".equalsIgnoreCase(groupId)) {
			interestPoint.setTheGroup(groupId);
		} else {
			// 获取兴趣点默认分组
			InterestGroup interestGroup = interestGroupDao.findDefaultWorkGroup(userId);
			if (interestGroup == null) {
				interestGroup = createInterestGroup(userId, "默认兴趣组");
				interestGroup.setIsDefault(1);
				interestGroupDao.save(interestGroup);
			}
			interestPoint.setTheGroup(interestGroup.getId());
		}
		// 判断该兴趣组是否有同名兴趣点
		Page<InterestPoint> interestPoints = getInterestPointsByGroupId(interestPoint.getTheGroup(), 0, 1000, "", "");
		for (InterestPoint point : interestPoints) {
			if (point.getName().equals(name)) {
				throw new RuntimeException("兴趣点名重复");
			}
		}
		interestPoint.setLat(Double.valueOf(lat));
		interestPoint.setLon(Double.valueOf(lon));
		interestPoint.setCreateTime(new Date());
		interestPoint.setIsDeleted(0);
		interestPointDao.save(interestPoint);
		return interestPoint;
	}

	public Page<InterestPoint> getInterestPointsByGroupId(String groupId, int start, int limit, String filterParam, String sortType) throws Exception {
		if (!"".equals(filterParam)) {
			filterParam += ";";
		}
		filterParam += "Q_theGroup_S_EQ=" + groupId + ";Q_isDeleted_N_EQ=0";
		String sortParam;
		if ("asc".equals(sortType)){
                sortParam = "SORT_createTime_DESC";
            } else {
                sortParam = "SORT_createTime_ASC";
            }
		return interestPointDao.findAll(new QuerySpecification<InterestPoint>(filterParam),
			 PageRequest.of(start, limit, QueryParamUtil.parseSortParams(sortParam)));
	}

	public void deleteInterestPoint(String ids, Long userId) throws Exception {
		String[] idArr = ids.split(",");
		for (String id : idArr) {
			InterestPoint interestPoint = interestPointDao.gwSearchByPK(id);
			if (userId.equals(interestPoint.getUserId())) {
				interestPoint.setIsDeleted(1);
				interestPointDao.save(interestPoint);
			} else {
				throw new RuntimeException("操作失败");
			}
		}
	}

	public void updateInterestPoint(Long userId, String id, String name, String groupId, Integer drawIndex) throws Exception {
		InterestPoint interestPoint = interestPointDao.gwSearchByPK(id);
		if (userId.equals(interestPoint.getUserId())) {
			if (name != null && !"".equals(name)) {
				interestPoint.setName(name);
			}
			if (groupId != null && !"".equals(groupId)) {
				if (!"default".equals(groupId)) {
					interestPoint.setTheGroup(groupId);
				} else {
					// 获取兴趣点默认分组
					InterestGroup interestGroup = interestGroupDao.findDefaultWorkGroup(userId);
					if (interestGroup == null) {
						interestGroup = createInterestGroup(userId, "默认兴趣组");
						interestGroup.setIsDefault(1);
						interestGroupDao.save(interestGroup);
					}
					interestPoint.setTheGroup(interestGroup.getId());
				}
				// 判断该兴趣组是否有同名兴趣点
				Page<InterestPoint> interestPoints = getInterestPointsByGroupId(interestPoint.getTheGroup(), 0, 1000, "", "");
				for (InterestPoint point : interestPoints) {
					if (point.getName().equals(name)) {
						throw new RuntimeException("兴趣点名重复");
					}
				}
			}
			if (drawIndex != null && drawIndex > 0){
                interestPoint.setDrawIndex(drawIndex);
            }
			interestPointDao.save(interestPoint);
		} else {
			throw new RuntimeException("操作失败");
		}
	}

    public List<Map<String, Object>> getInterestPointCities(String groupId) throws Exception{
        List<Map<String, Object>> theList = new ArrayList<Map<String,Object>>();
        List<Object> list = interestPointDao.getInterestPointCities(groupId);
        for (Object object : list) {
            Object[] arr = (Object[]) object;
            Map<String, Object> map = new HashMap<String, Object>();
            map.put("city", arr[0]);
            map.put("cityName", arr[1]);
            map.put("num", arr[2]);
            map.put("createTime", arr[3]);
            theList.add(map);
        }
        return theList;
    }

    public Page<InterestPoint> getInterestPointByCity(String groupId, String city, String sortType, int start, int limit) {
        String filterParam ="Q_theGroup_S_EQ=" + groupId +";Q_city_S_EQ=" + city + ";Q_isDeleted_N_EQ=0";
        String sortParam;
        if ("asc".equals(sortType)){
            sortParam = "SORT_createTime_ASC";
        } else {
            sortParam = "SORT_createTime_DESC";
        }
        return interestPointDao.findAll(new QuerySpecification<InterestPoint>(filterParam),
             PageRequest.of(start, limit, QueryParamUtil.parseSortParams(sortParam)));
    }
}
