package com.geoway.landteam.landcloud.service.pub;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.geoway.landteam.landcloud.common.util.http.HttpsUtil;
import com.geoway.landteam.landcloud.core.service.base.TemporarySignedUrlService;
import com.itextpdf.text.*;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfCell;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.validator.routines.UrlValidator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.io.InputStream;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Service
public class PDFService {

    @Autowired
    TemporarySignedUrlService temporarySignedUrlService;

    private final Map<String, String> errInfo;

    public PDFService(){
        errInfo = new HashMap<>();
        errInfo.put("photo", "无现场照片");
        errInfo.put("image_Analyze", "查询范围内无当年影像信息");
        errInfo.put("landType_Analyze", "查询范围内无当年现状信息");
        errInfo.put("ownership_Analyze", "查询范围无权属关系信息");
        errInfo.put("plan_Analyze", "查询范围内无基本土地规划信息");
        errInfo.put("landGrade_Analyze", "查询范围内无耕地等别信息");
        errInfo.put("primeFarm_Analyze", "查询范围内不涉及永久基本农田");
        errInfo.put("spba_Analyze", "查询范围内无审批备案信息");
        errInfo.put("natureReserve_Analyze", "查询范围内不涉及自然保护区");
        errInfo.put("jsydgmbj_Analyze", "查询范围内不涉及建设用地边界");
        errInfo.put("cxydgh_Analyze", "查询范围内不涉及城乡规划");
        errInfo.put("gqzrbhq_Analyze", "查询范围内不涉及保护区(国情)");
        errInfo.put("keyArea_Analyze", "查询范围内不涉及保护区(报备)");
    }


    /***
     * 获取图片表格单元
     *
     * @param columnwidth
     *            宽度
     * @param minColumnHeight
     *            最小高度
     * @param imgb
     *            图片内容
     * @param isBorder
     *            是否有边界
     * @return
     */
    public PdfPCell getImageCell(Integer columnwidth, Integer minColumnHeight,
                                        final byte imgb[], Boolean isBorder) {
        try {
            if (imgb == null) {
                return null;
            }
            Image image = Image.getInstance(imgb);

            if (null != minColumnHeight && columnwidth != null) {
                //image.scaleToFit(columnwidth - 1, minColumnHeight - 1);
                //image.scaleToFit(100,400);
                //image.setScaleToFitHeight(true);
                image.scaleToFit(columnwidth * 4 - 5, minColumnHeight - 5);
                //image.scaleAbsoluteHeight(minColumnHeight);

            }

            PdfPCell vcell = new PdfPCell(image);

            vcell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
            vcell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
            if (!isBorder) {
                vcell.setBorder(PdfPCell.NO_BORDER);
            }
            if (null != minColumnHeight) {
                vcell.setMinimumHeight(minColumnHeight);
            }
            return vcell;

        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }


    /**
     * 获得PDFCell
     *
     * @param font
     * @param columnHeight
     * @param cellValue
     * @param isBorder
     * @param align
     * @return
     */
    public PdfPCell getPDFCell(Font font, Integer columnHeight, String cellValue, Boolean isBorder, int align) {
        PdfPCell dcell = new PdfPCell(new Phrase(cellValue, font));
        dcell.setHorizontalAlignment(align);
        dcell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);

        if (!isBorder) {
            dcell.setBorder(PdfPCell.NO_BORDER);
        }
        //dcell.setBackgroundColor(BaseColor.LIGHT_GRAY);
        dcell.setMinimumHeight(columnHeight);

        return dcell;
    }

    /**
     * 根据模板取值
     *
     * @param jsonObject
     * @param template
     * @return
     */
    public String getCellValue(JSONObject jsonObject, String template, String format) {
        String val = "";
        try {
            List<String> extraValue = getExtraValue(template);
            for (int i = 0; i < extraValue.size(); i++) {
                Object o = getJSONValue(jsonObject, extraValue.get(i));
                if (o == null) {
                    template = template.replace("{" + extraValue.get(i) + "}", "");
                } else {
                    String valStr = o.toString();
                    if (StringUtils.isNotEmpty(format)) {
                        Double valDouble = Double.valueOf(valStr);
                        DecimalFormat decimalFormat = new DecimalFormat(format);
                        valStr = decimalFormat.format(valDouble);
                    }
                    template = template.replace("{" + extraValue.get(i) + "}", valStr);
                }
            }
        } catch (Exception e) {
            return "";
        }
        return template;
    }


    public String getCellValue(JSONObject jsonObject, String template) {
        return getCellValue(jsonObject, template, null);
    }

    /**
     * 根据模板内容获取json的值
     *
     * @param jsonObject
     * @param template
     * @return
     */
    private Object getJSONValue(JSONObject jsonObject, String template) {
        try {
            if (jsonObject == null) {
                return null;
            }

            String[] arr = template.split("\\.");
            if (arr.length < 2) {
                return jsonObject.getString(template);
            }
            for (int j = 0; j <arr.length - 1; j++) {
            	
                // 从数组中取值
                if (arr[j].contains("[")) {
                    String key = arr[j].substring(0, arr[j].indexOf("["));
                    JSONArray jsonArray = jsonObject.getJSONArray(key);
                    Integer index = Integer.valueOf(getExtraValue1(template).get(0));
                    JSONObject temp = jsonArray.getJSONObject(index);
                    template = template.substring(arr[j].length());
                    if (template.startsWith(".")) {
                        template = template.substring(1);
                    }
                    return getJSONValue(temp, template);
                } else {
                    if (j == arr.length - 1) {
                        return getJSONValue(jsonObject, arr[j]);
                    } else {
                        JSONObject temp = jsonObject.getJSONObject(arr[j]);
                        template = template.substring(arr[j].length());
                        if (template.startsWith(".")) {
                            template = template.substring(1);
                        }
                        return getJSONValue(temp, template);
                    }
                }
            }
        } catch (Exception e) {
            return "";
        }
        return "";
    }

    /**
     * 提取花括号中的内容
     *
     * @param value
     * @return
     */
    public List<String> getExtraValue(String value) {
        try {
            List<String> list = new ArrayList<String>();
            Pattern p = Pattern.compile("(?<=\\{)(.+?)(?=\\})");
            Matcher m = p.matcher(value);
            while (m.find()) {
                list.add(m.group());
            }
            return list;
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 提取中括号中的内容
     *
     * @param value
     * @return
     */
    public List<String> getExtraValue1(String value) {
        List<String> list = new ArrayList<String>();
        Pattern p = Pattern.compile("(?<=\\[)(.+?)(?=\\])");
        Matcher m = p.matcher(value);
        while (m.find()) {
            list.add(m.group());
        }
        return list;
    }

    public PdfPTable getImageTable(JSONObject result, JSONObject object, Font font, Font imgTitleFont, Font kai, Integer width) {
        int[] widths = {width};
        PdfPTable imgTable = new PdfPTable(1);
        //获取模板json中的基本信息
        JSONObject title = object.getJSONObject("title");
        String titleVal = title.getString("value");
        Integer titleHt = title.getInteger("height");
        JSONObject img = object.getJSONObject("img");
        Integer imgHeight = img.getInteger("height");
        String mode = object.getString("mode");
        byte[] bytes = null;
        try {
//            设置表格宽度
            imgTable.setWidths(widths);
//            添加标题栏样式设置标题栏
            if (StringUtils.isNotEmpty(titleVal)) {
                PdfPCell titleCell = getPDFCell(imgTitleFont, titleHt, titleVal, false, PdfCell.ALIGN_CENTER);
                titleCell.setBorder(PdfCell.BOTTOM);
                titleCell.setBorderWidth(0.5f);
                titleCell.setBorderColor(BaseColor.LIGHT_GRAY);
                imgTable.addCell(titleCell);
            }
//            解析出真实的url地址
            String url = parseImgUrl(mode, result, object);
//            请求图片
            String year = "";
            if("image_Analyze".equalsIgnoreCase(mode)){
                year = url.split(",")[1];
                url = url.split(",")[0];
            }

            if(url.contains("obs.cn") && !url.contains("AWSAccessKeyId")){
                //url = temporarySignedUrlService.getTemporarySignedUrl("",url,null);
            }

            bytes = HttpsUtil.httpRequest(url, "GET", "");
            Image image = Image.getInstance(bytes);
            if (null != imgHeight && null != width) {
                image.scaleToFit(width * 4.5f - 3, imgHeight - 3);
            }
            PdfPCell imgCell = new PdfPCell(image);
            imgCell.setHorizontalAlignment(PdfPCell.ALIGN_CENTER);
            imgCell.setVerticalAlignment(PdfPCell.ALIGN_MIDDLE);
            imgCell.setBorder(PdfPCell.NO_BORDER);
            imgCell.setPadding(0);
            if (null != imgHeight) {
                imgCell.setMinimumHeight(imgHeight);
            }
            imgTable.addCell(imgCell);
//            解析描述信息
            JSONObject attr = object.getJSONObject("attr");
            if (attr != null && attr.getBoolean("show")) {
                String key = attr.getString("key");
                String desc = "";
                if ("photo".equals(mode)) {
                    JSONObject basic = result.getJSONObject("basic");
                    desc = parseAttr(basic, key);
                } else {
//                    当mode为影像和现状时，url为年份
                    String yearTmp = "";
                    if("image_Analyze".equalsIgnoreCase(mode)){
                        yearTmp = year;
                    }
                    else {
                        yearTmp = object.getJSONObject("img").getString("url");
                    }
                    JSONObject detail = result.getJSONObject("detail");
                    desc = parseAttr(detail, mode, key, yearTmp);
                }
//                设置描述信息格式，添加描述信息到表格
                PdfPCell descCell = getPDFCell(kai, 18, desc, false, PdfCell.ALIGN_CENTER);
                descCell.setBorder(PdfCell.TOP);
                descCell.setBorderColor(BaseColor.LIGHT_GRAY);
                descCell.setBorderWidth(0.5f);
                imgTable.addCell(descCell);
            }
        } catch (Exception e) {
            //e.printStackTrace();
            String msg = errInfo.get(mode);
            PdfPCell noResult = getPDFCell(font, imgHeight, msg, false, PdfCell.ALIGN_CENTER);
            imgTable.addCell(noResult);
        }
        return imgTable;
    }

    private String parseImgUrl(String mode, JSONObject result, JSONObject object) {
        String urlStr = object.getJSONObject("img").getString("url");
        if (StringUtils.isEmpty(urlStr)) {
            throw new RuntimeException("url为空");
        }
        String imgUrl = urlStr;
        String imgMode = object.getJSONObject("img").getString("yearMode");
        switch (mode) {
            //因为云盘的原因随手拍图片的url地址不唯一,故输入的url为第几张
            case "photo": {
                JSONArray jsonArray = result.getJSONObject("basic").getJSONArray("list");
                if("task".equalsIgnoreCase(object.getString("target"))){
                    jsonArray = result.getJSONObject("basic").getJSONObject("prj").getJSONObject("tb").getJSONArray
                            ("list");
                }
                else{
                    jsonArray = result.getJSONObject("basic").getJSONArray("list");
                }
                Integer index = Integer.valueOf(urlStr);
                Integer fromIndex = -1;
                JSONObject photoObject = null;
                for (int i = 0; i < jsonArray.size(); i++) {
                    JSONObject obj = jsonArray.getJSONObject(i);
                    if(obj.getInteger("type")!=1){
                        continue;
                    }
                    fromIndex++;
                    if(fromIndex.equals(index)){
                        photoObject = obj;
                        break;
                    }
                }
                JSONObject photo = photoObject;
                /*if (StringUtils.isNotEmpty(photo.getString("serverpath"))) {
                    imgUrl = photo.getString("serverpath");
                } else*/ if (StringUtils.isNotEmpty(photo.getString("downloadUrl"))) {
                    imgUrl = photo.getString("downloadUrl");
                } else if (StringUtils.isNotEmpty(photo.getString("serverpath"))) {
                    imgUrl = photo.getString("serverpath");
                } else {
                    imgUrl = "";
                }
                break;
            }
            //影像分析与土地现状涉及到多年的情况需要根据年份筛选，故输入的url为年份
            case "landType_Analyze":{
                boolean match=false;
                JSONArray history = result.getJSONObject("detail").getJSONArray(mode + "_History");
                for (int i = 0; i < history.size(); i++) {
                    JSONObject landType = history.getJSONObject(i);
                    String date = landType.getString("date");
                    if (date.startsWith(urlStr)) {
                        imgUrl = landType.getString("images");
                        match=true;
                        break;
                    }
                }
                if(!match)
                {
                    imgUrl = getCellValue(result, urlStr);
                }
                break;
            }
            case "image_Analyze": {
                JSONArray history = result.getJSONObject("detail").getJSONArray(mode + "_History");
                int lastYear = 0;
                for (int i = 0; i < history.size(); i++) {
                    JSONObject landType = history.getJSONObject(i);
                    String date = landType.getString("date");
                    int year = Integer.parseInt(date.substring(0,4));

                    if("maxYear".equalsIgnoreCase(imgMode)) {
                        if (year > lastYear) {
                            lastYear = year;
                            imgUrl = landType.getString("images") + "," + lastYear;
                        }
                    }
                    else {
                        if (String.valueOf(year).equalsIgnoreCase(imgUrl)) {
                            imgUrl = landType.getString("images") + "," + year;
                        } else {
                            continue;
                        }
                    }
//                    if (date.startsWith(urlStr)) {
//                        imgUrl = landType.getString("images");
//                        break;
//                    }
                }
                break;
            }
            case "cxydgh_Analyze":{
                if(urlStr.contains("Cxydgh_Analyze")){
                    imgUrl = getCellValue(result, urlStr.replace("Cxydgh_Analyze","cxydgh_Analyze"));
                }else{
                    imgUrl = getCellValue(result, urlStr);
                }
                break;
            }
            case "jsydgmbj_Analyze":{
                if(urlStr.contains("Jsydgmbj_Analyze")){
                    imgUrl =getCellValue(result,  urlStr.replace("Jsydgmbj_Analyze","jsydgmbj_Analyze"));
                }else{
                    imgUrl = getCellValue(result, urlStr);
                }
                break;
            }
            case "keyArea_Analyze":{
                if(urlStr.contains("KeyArea_Analyze")){
                    imgUrl = getCellValue(result, urlStr.replace("KeyArea_Analyze","keyArea_Analyze"));
                }else{
                    imgUrl = getCellValue(result, urlStr);
                }
                break;
            }
            case "ownership_Analyze":
            case "plan_Analyze":
            case "landGrade_Analyze":
            case "primeFarm_Analyze":
            case "spba_Analyze":
            case "natureReserve_Analyze":
            default:
                imgUrl = getCellValue(result, urlStr);
        }
        UrlValidator urlValidator = new UrlValidator();
        if (urlValidator.isValid(imgUrl)) {
            return imgUrl;
        } else {
            throw new RuntimeException("图片url不合法");
        }
    }

    //    本描述解析提供给云查询使用
    private String parseAttr(JSONObject detail, String mode, String key, String sj){
        String result = "";
        JSONObject modeDetail = detail.getJSONObject(mode);
        JSONArray attrs = modeDetail.getJSONArray("attributes");
        JSONObject base = detail.getJSONObject("data");
        switch (mode) {
            case "image_Analyze": {
                JSONArray imgAttrs = detail.getJSONArray("image_Analyze_History");
                JSONObject attr = new JSONObject();
                for (int i = 0; i < imgAttrs.size(); i++) {
                    if (imgAttrs.getJSONObject(i).getString("date").startsWith(sj)) {
                        attr = imgAttrs.getJSONObject(i).getJSONArray("attributes").getJSONObject(0);
                        break;
                    }
                }
                String sjy = attr.getString("sjy");
                String sx = attr.getString("sx");
                String year = sx.substring(0, 4);
                String month = sx.substring(4, 6);
                String day = sx.substring(6, 8);
                //result = StringFormatter.format("卫星：%s   日期：%s-%s-%s", sjy, year, month, day).getValue();
                result = String.format("卫星：%s   日期：%s-%s-%s", sjy, year, month, day);
                break;

            }
            case "landType_Analyze": {
                JSONArray landTypeAttrs = detail.getJSONArray("landType_Analyze_History");
                for (int i = 0; i < landTypeAttrs.size(); i++) {
                    if (landTypeAttrs.getJSONObject(i).getString("date").startsWith(sj)) {
                        attrs = landTypeAttrs.getJSONObject(i).getJSONArray("attributes");
                        break;
                    }
                }
                float max1 = 0, max2 = 0, total = 0;
                String max1Code = "", max2Code = "", max1Str, max2Str, otherStr;
                List<String> codes = new ArrayList<>();
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    Float mj = attr.getFloat("mj");
                    String code = attr.getString(key);
                    codes.add(code);
                    total += mj;
                    if (mj > max1) {
                        max2 = max1;
                        max2Code = max1Code;
                        max1 = mj;
                        max1Code = code;
                    } else if (mj >= max2) {
                        max2 = mj;
                        max2Code = code;
                    }
                }
                float other = total - max1 - max2;
                codes.remove(max1Code);
                codes.remove(max2Code);
                DecimalFormat decimalFormat = new DecimalFormat("0.00");
                max1Str = decimalFormat.format(max1);
                max2Str = decimalFormat.format(max2);
                otherStr = decimalFormat.format(other);
                if (max2 <= 0) {
                    //result = StringFormatter.format("%s-%s亩", max1Code, decimalFormat.format(max1)).getValue();
                    result = String.format("%s-%s亩", max1Code, decimalFormat.format(max1));
                } else if (codes.size() == 0) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str).getValue();
                    result = String.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str);
                } else if (codes.size() == 1) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr);
                } else {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr);
                }
                break;
            }

            case "landGrade_Analyze":
            case "plan_Analyze":
            case "ownership_Analyze": {
                float max1 = 0, max2 = 0, total = 0;
                String max1Code = "", max2Code = "", max1Str, max2Str, otherStr;
                List<String> codes = new ArrayList<>();
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    Float mj = attr.getFloat("mj");
                    String code = attr.getString(key);
                    codes.add(code);
                    total += mj;
                    if (mj > max1) {
                        max2 = max1;
                        max2Code = max1Code;
                        max1 = mj;
                        max1Code = code;
                    } else if (mj >= max2) {
                        max2 = mj;
                        max2Code = code;
                    }
                }
                float other = total - max1 - max2;
                codes.remove(max1Code);
                codes.remove(max2Code);
                DecimalFormat decimalFormat = new DecimalFormat("0.00");
                max1Str = decimalFormat.format(max1);
                max2Str = decimalFormat.format(max2);
                otherStr = decimalFormat.format(other);
                if (max2 <= 0) {
                    //result = StringFormatter.format("%s-%s亩", max1Code, decimalFormat.format(max1)).getValue();
                    result = String.format("%s-%s亩", max1Code, decimalFormat.format(max1));
                } else if (codes.size() == 0) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str).getValue();
                    result = String.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str);
                } else if (codes.size() == 1) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr);
                } else {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr);
                }
                break;
            }
            case "natureReserve_Analyze":{
                Float total = base.getFloat("area");
                float yba = 0;
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    Float mj = attr.getFloat("mj");
                    yba += mj;
                }
                DecimalFormat decimalFormat = new DecimalFormat("0.00");
                String wbaStr = decimalFormat.format(total - yba);
                String ybaStr = decimalFormat.format(yba);
                //result = StringFormatter.format("自然保护区-%s亩/非自然保护区-%s亩", ybaStr, wbaStr).getValue();
                result = String.format("自然保护区-%s亩/非自然保护区-%s亩", ybaStr, wbaStr);
                break;
            }
            case "primeFarm_Analyze":{
                Float total = base.getFloat("area");
                float yba = 0;
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    Float mj = attr.getFloat("mj");
                    yba += mj;
                }
                DecimalFormat decimalFormat = new DecimalFormat("0.00");
                String wbaStr = decimalFormat.format(total - yba);
                String ybaStr = decimalFormat.format(yba);
                //result = StringFormatter.format("基本农田-%s亩/非基本农田-%s亩", ybaStr, wbaStr).getValue();
                result = String.format("基本农田-%s亩/非基本农田-%s亩", ybaStr, wbaStr);
                break;
            }
            case "spba_Analyze": {
                Float total = base.getFloat("area");
                float yba = 0;
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    Float mj = attr.getFloat("mj");
                    yba += mj;
                }
                DecimalFormat decimalFormat = new DecimalFormat("0.00");
                String wbaStr = decimalFormat.format(total - yba);
                String ybaStr = decimalFormat.format(yba);
                //result = StringFormatter.format("已备案-%s亩/未备案-%s亩", ybaStr, wbaStr).getValue();
                result = String.format("已备案-%s亩/未备案-%s亩", ybaStr, wbaStr);
                break;
            }
            case "gqzrbhq_Analyze": {
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    String type = attr.getString("datatype");
                    Float mj = attr.getFloat("area");
                    DecimalFormat decimalFormat = new DecimalFormat("0.00");

                    if(StringUtils.isNotBlank(result)){
                        result+="/";
                    }

                    result+=type+"-"+decimalFormat.format(mj)+"亩";
                }
                break;
            }
            case "cxydgh_Analyze": {
                float max1 = 0, max2 = 0, total = 0;
                String max1Code = "", max2Code = "", max1Str, max2Str, otherStr;
                List<String> codes = new ArrayList<>();
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    Float mj = attr.getFloat("mj");
                    String code = attr.getString(key);
                    codes.add(code);
                    total += mj;
                    if (mj > max1) {
                        max2 = max1;
                        max2Code = max1Code;
                        max1 = mj;
                        max1Code = code;
                    } else if (mj >= max2) {
                        max2 = mj;
                        max2Code = code;
                    }
                }
                float other = total - max1 - max2;
                codes.remove(max1Code);
                codes.remove(max2Code);
                DecimalFormat decimalFormat = new DecimalFormat("0.00");
                max1Str = decimalFormat.format(max1);
                max2Str = decimalFormat.format(max2);
                otherStr = decimalFormat.format(other);
                if (max2 <= 0) {
                    //result = StringFormatter.format("%s-%s亩", max1Code, decimalFormat.format(max1)).getValue();
                    result = String.format("%s-%s亩", max1Code, decimalFormat.format(max1));
                } else if (codes.size() == 0) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str).getValue();
                    result = String.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str);
                } else if (codes.size() == 1) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr);
                } else {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr);
                }
                break;
            }
            case "keyArea_Analyze": {
                float max1 = 0, max2 = 0, total = 0;
                String max1Code = "", max2Code = "", max1Str, max2Str, otherStr;
                List<String> codes = new ArrayList<>();
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    Float mj = attr.getFloat("mj");
                    String code = attr.getString(key);
                    codes.add(code);
                    total += mj;
                    if (mj > max1) {
                        max2 = max1;
                        max2Code = max1Code;
                        max1 = mj;
                        max1Code = code;
                    } else if (mj >= max2) {
                        max2 = mj;
                        max2Code = code;
                    }
                }
                float other = total - max1 - max2;
                codes.remove(max1Code);
                codes.remove(max2Code);
                DecimalFormat decimalFormat = new DecimalFormat("0.00");
                max1Str = decimalFormat.format(max1);
                max2Str = decimalFormat.format(max2);
                otherStr = decimalFormat.format(other);
                if (max2 <= 0) {
                    //result = StringFormatter.format("%s-%s亩", max1Code, decimalFormat.format(max1)).getValue();
                    result = String.format("%s-%s亩", max1Code, decimalFormat.format(max1));
                } else if (codes.size() == 0) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str).getValue();
                    result = String.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str);
                } else if (codes.size() == 1) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr);
                } else {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr);
                }
                break;
            }
            case "jsydgmbj_Analyze": {
                float max1 = 0, max2 = 0, total = 0;
                String max1Code = "", max2Code = "", max1Str, max2Str, otherStr;
                List<String> codes = new ArrayList<>();
                for (int i = 0; i < attrs.size(); i++) {
                    JSONObject attr = attrs.getJSONObject(i);
                    Float mj = attr.getFloat("mj");
                    String code = attr.getString(key);
                    codes.add(code);
                    total += mj;
                    if (mj > max1) {
                        max2 = max1;
                        max2Code = max1Code;
                        max1 = mj;
                        max1Code = code;
                    } else if (mj >= max2) {
                        max2 = mj;
                        max2Code = code;
                    }
                }
                float other = total - max1 - max2;
                codes.remove(max1Code);
                codes.remove(max2Code);
                DecimalFormat decimalFormat = new DecimalFormat("0.00");
                max1Str = decimalFormat.format(max1);
                max2Str = decimalFormat.format(max2);
                otherStr = decimalFormat.format(other);
                if (max2 <= 0) {
                    //result = StringFormatter.format("%s-%s亩", max1Code, decimalFormat.format(max1)).getValue();
                    result = String.format("%s-%s亩", max1Code, decimalFormat.format(max1));
                } else if (codes.size() == 0) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str).getValue();
                    result = String.format("%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str);
                } else if (codes.size() == 1) {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/%s-%s亩", max1Code, max1Str, max2Code, max2Str, codes.get(0), otherStr);
                } else {
                    //result = StringFormatter.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr).getValue();
                    result = String.format("%s-%s亩/%s-%s亩/其他-%s亩", max1Code, max1Str, max2Code, max2Str, otherStr);
                }
                break;
            }
            default:
                result = "";
        }
        return result;
    }

    /**
     * 本描述解析专给现场图片使用；
     *
     * @param basic 图集的基本信息
     * @param index 第几章图片
     * @return
     */
    private String parseAttr(JSONObject basic, String index) {
        String username = "";
        if (basic.getString("username") != null) {
            username = basic.getString("username");
        }
        JSONObject attr = null;
        if(basic.getJSONArray("list")==null){
            attr = basic.getJSONObject("prj").getJSONObject("tb").getJSONArray("list").getJSONObject(Integer.valueOf
                    (index));
        }
        else{
            attr = basic.getJSONArray("list").getJSONObject(Integer.valueOf(index));
        }
        Date date = attr.getDate("time");
        SimpleDateFormat formater = new SimpleDateFormat("yyyy-MM-dd");
        String dateStr = formater.format(date);
        Double lat = attr.getDouble("lat");
        Double lon = attr.getDouble("lon");
        Float dAzimuthDeg = attr.getFloat("azimuth");
        int azimuthDeg = Math.round(dAzimuthDeg);
        Double direction = Math.floor((azimuthDeg + 15) / 30);
        String azimuth = "正北";
        switch (direction.intValue()) {
            case 0: {
                azimuth = "正北";
                break;
            }
            case 1: {
                azimuth = "北偏东";
                break;
            }
            case 2: {
                azimuth = "东偏北";
                break;
            }
            case 3: {
                azimuth = "正东";
                break;
            }
            case 4: {
                azimuth = "东偏南";
                break;
            }
            case 5: {
                azimuth = "南偏东";
                break;
            }
            case 6: {
                azimuth = "正南";
                break;
            }
            case 7: {
                azimuth = "南偏西";
                break;
            }
            case 8: {
                azimuth = "西偏南";
                break;
            }
            case 9: {
                azimuth = "正西";
                break;
            }
            case 10: {
                azimuth = "西偏北";
                break;
            }
            case 11: {
                azimuth = "北偏西";
                break;
            }
            default:
                azimuth = "正北";
        }

        DecimalFormat decimalFormat = new DecimalFormat("0.00");
        String latStr = decimalFormat.format(lat);
        String lonStr = decimalFormat.format(lon);
        //return StringFormatter.format("%s  %s°E %s°N  %s  %s", username, lonStr, latStr, azimuth, dateStr).getValue();
        return String.format("%s  %s°E %s°N  %s  %s", username, lonStr, latStr, azimuth, dateStr);
    }

    public Image addStamp(InputStream in) throws IOException, BadElementException {
        byte[] bytes = new byte[in.available()];
        in.read(bytes);
        Image img = Image.getInstance(bytes);
        img.scaleToFit(75, 75);
        img.setAbsolutePosition(510, 756);
        return img;
    }
}
