package com.geoway.landteam.landcloud.service.thirddata.utils;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

/***
 * 文件操作相关类
 * 
 * @author Walt Chiang
 *
 */
public class FileUtil {
	/**
	 * 递归删除目录下的所有文件及子目录下所有文件
	 * 
	 * @param dir
	 *            将要删除的文件目录
	 * @return boolean Returns "true" if all deletions were successful. If a
	 *         deletion fails, the method stops attempting to delete and returns
	 *         "false".
	 */
	public static boolean deleteFileAndDir(File dir) {
		if (dir.isDirectory()) {
			String[] children = dir.list();
			// 递归删除目录中的子目录下
			for (int i = 0; i < children.length; i++) {
				boolean success = deleteFileAndDir(new File(dir, children[i]));
				if (!success) {
					return false;
				}
			}
		}
		// 目录此时为空，可以删除
		return dir.delete();
	}

	/**
	 * 递归查找文件
	 * 
	 * @param baseDirName
	 *            查找的文件夹路径
	 * @param targetFileName
	 *            需要查找的文件名
	 */
	public static List<String> findFiles(String baseDirName, String targetFileName) {
		List<String> fileList = new ArrayList<String>();
		/**
		 * 算法简述： 从某个给定的需查找的文件夹出发，搜索该文件夹的所有子文件夹及文件，
		 * 若为文件，则进行匹配，匹配成功则加入结果集，若为子文件夹，则进队列。 队列不空，重复上述操作，队列为空，程序结束，返回结果。
		 */
		String tempName = null;
		// 判断目录是否存在
		File baseDir = new File(baseDirName);
		if (!baseDir.exists() || !baseDir.isDirectory()) {
			System.out.println("文件查找失败：" + baseDirName + "不是一个目录！");
		} else {
			String[] filelist = baseDir.list();
			for (int i = 0; i < filelist.length; i++) {
				File readfile = new File(baseDirName + File.separator + filelist[i]);
				// System.out.println(readfile.getName());
				if (!readfile.isDirectory()) {
					tempName = readfile.getName();
					if (FileUtil.wildcardMatch(targetFileName, tempName)) {
						// 匹配成功，将文件名添加到结果集
						fileList.add(readfile.getAbsolutePath());
					}
				} else if (readfile.isDirectory()) {
					fileList.addAll(findFiles(baseDirName + File.separator + filelist[i],
							targetFileName));
				}
			}
		}
		return fileList;
	}

	/**
	 * 通配符匹配
	 * 
	 * @param pattern
	 *            通配符模式
	 * @param str
	 *            待匹配的字符串
	 * @return 匹配成功则返回true，否则返回false
	 */
	private static boolean wildcardMatch(String pattern, String str) {
		int patternLength = pattern.length();
		int strLength = str.length();
		int strIndex = 0;
		char ch;
		for (int patternIndex = 0; patternIndex < patternLength; patternIndex++) {
			ch = pattern.charAt(patternIndex);
			if (ch == '*') {
				// 通配符星号*表示可以匹配任意多个字符
				while (strIndex < strLength) {
					if (wildcardMatch(pattern.substring(patternIndex + 1),
							str.substring(strIndex))) {
						return true;
					}
					strIndex++;
				}
			} else if (ch == '?') {
				// 通配符问号?表示匹配任意一个字符
				strIndex++;
				if (strIndex > strLength) {
					// 表示str中已经没有字符匹配?了。
					return false;
				}
			} else {
				if ((strIndex >= strLength) || (ch != str.charAt(strIndex))) {
					return false;
				}
				strIndex++;
			}
		}
		return (strIndex == strLength);
	}

	public static List<String> findDirectorys(String baseDirName, String targetFileName) {
		List<String> fileList = new ArrayList<String>();
		String tempName = null;
		File baseDir = new File(baseDirName);
		if (!baseDir.exists() || !baseDir.isDirectory()) {
			System.out.println("文件查找失败：" + baseDirName + "不是一个目录！");
		} else {
			String[] filelist = baseDir.list();
			for (int i = 0; i < filelist.length; i++) {
				File readfile = new File(baseDirName + File.separator + filelist[i]);
				tempName = readfile.getName();
				if (FileUtil.wildcardMatch(targetFileName, tempName)) {
					fileList.add(readfile.getAbsolutePath());
				}else if (readfile.isDirectory()) {
					fileList.addAll(findDirectorys(baseDirName + File.separator + filelist[i],
							targetFileName));
				}
			}
		}
		return fileList;
	}

	/**
	 * 复制文件
	 * @param inFilePath
	 * @param outFilePath
	 * @throws IOException
	 */
	public static void copyFile(String inFilePath, String outFilePath) throws IOException {
		File in = new File(inFilePath);
		File out = new File(outFilePath);
		if (!out.getParentFile().exists()) {
			out.getParentFile().mkdirs();
			out.createNewFile();
		}
		copyFileUsingFileStreams(in, out);
	}

	public static void copyFileUsingFileStreams(File in, File out) throws IOException {
		InputStream input = null;
		OutputStream output = null;
		try {
			input = new FileInputStream(in);
			output = new FileOutputStream(out);
			byte[] buf = new byte[1024];
			int bytesRead;
			while ((bytesRead = input.read(buf)) > 0) {
				output.write(buf, 0, bytesRead);
			}
		} finally {
			input.close();
			output.close();
		}
	}

	public static byte[] readFileByte(String file) throws IOException {
		byte[] bytes = null;
		ByteArrayOutputStream byteOut = null;
		FileInputStream fileInsS = null;
		try {
			File f = new File(file);
			fileInsS = new FileInputStream(f);
			byte[] buffer = new byte[1024];
			byteOut = new ByteArrayOutputStream();
			for (int len; (len = fileInsS.read(buffer)) != -1; ) {
				byteOut.write(buffer, 0, len);
			}
			byteOut.flush();
			bytes = byteOut != null ? byteOut.toByteArray() : null;
		} finally {
			if (fileInsS != null) {
				fileInsS.close();
			}
			if (byteOut != null) {
				byteOut.close();
			}
		}
		return bytes;
	}

	public static void downloadExcelFile(HttpServletResponse resp, byte[] fileBytes, String fileName) throws IOException {
		resp.setContentType("application/vnd.ms-excel;charset=UTF-8");
		resp.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(fileName, "UTF-8"));
		processingByte(resp, fileBytes);
	}

	public static void downloadJsonFile(HttpServletResponse resp, byte[] fileBytes, String fileName) throws IOException {
		resp.setHeader("content-type", "application/json");
		resp.setContentType("application/octet-stream");
		resp.setHeader("Content-Disposition", "attachment;filename=" + fileName);
		processingByte(resp, fileBytes);
	}

	public static void downloadXmlFile(HttpServletResponse resp, byte[] fileBytes, String fileName) throws IOException {
		resp.setHeader("content-type", "application/octet-stream");
		resp.setContentType("application/octet-stream");
		resp.setHeader("Content-Disposition", "attachment;filename=" + fileName);
		processingByte(resp, fileBytes);
	}

	public static void processingByte(HttpServletResponse resp, byte[] fileBytes) throws IOException{
		byte[] buff = new byte[1024];
		BufferedInputStream bufferedInputStream = null;
		OutputStream outputStream = null;
		try {
			outputStream = resp.getOutputStream();
			bufferedInputStream = new BufferedInputStream(new ByteArrayInputStream(fileBytes), fileBytes.length);
			int num = bufferedInputStream.read(buff);
			while (num != -1) {
				outputStream.write(buff, 0, num);
				outputStream.flush();
				num = bufferedInputStream.read(buff);
			}
		} catch (IOException e) {
			throw new RuntimeException(e.getMessage());
		} finally {
			if (bufferedInputStream != null) {
				bufferedInputStream.close();
			}
		}
	}
}
