package com.geoway.landteam.landcloud.service.util;

import com.geoway.landteam.landcloud.common.util.geometry.GeometryUtil;
import com.geoway.landteam.landcloud.common.util.geometry.WKTUtil;
import org.apache.commons.lang3.StringUtils;
import org.geotools.data.FeatureWriter;
import org.geotools.data.Transaction;
import org.geotools.data.shapefile.ShapefileDataStore;
import org.geotools.data.shapefile.ShapefileDataStoreFactory;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.geotools.referencing.CRS;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.io.WKTReader;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

public class ShpUtil {
    public static Map<String,String> getShpToTableColumns(ShapefileDataStore shpDataStore) throws IOException {
        String[] typeNames = shpDataStore.getTypeNames();
        String typeName = typeNames[0];
        SimpleFeatureType featureType = shpDataStore.getSchema(typeName);
        String shapeField = featureType.getGeometryDescriptor().getLocalName();
        List<AttributeDescriptor> ads = featureType.getAttributeDescriptors();
        List<String> fieldNames = ads.stream().map(descriptor -> descriptor.getLocalName()).collect(Collectors.toList());
        Map<String,String>mapFields = new HashMap<>();
        for(String f:fieldNames)
        {
            if(f.equals(shapeField))
            {
                mapFields.put("f_shape","public.geometry NOT NULL");
            }
            else
            {
                mapFields.put(f,"varchar(100) NULL");
            }
        }
        return mapFields;
    }

    public static Map<String,Object> getFromSimpleFeature(SimpleFeature sf, List<String>fields)
    {
        Map<String,Object> map = new HashMap<>();
        Geometry geo = (Geometry) sf.getDefaultGeometry();
        geo.setSRID(4490);
        for(String f:fields)
        {
            if(f.equals("f_shape"))
            {
                continue;
            }
            map.put(f,sf.getAttribute(f));

        }
        map.put("f_shape","SRID=4490;"+ WKTUtil.geomToWkt(geo));

        return map;

    }
    public static Map<String,Object> getFromSimpleFeatureWithoutShape(SimpleFeature sf, List<String>fields)
    {
        Map<String,Object> map = new HashMap<>();

        for(String f:fields)
        {
            if(f.equals("f_shape"))
            {
                continue;
            }
            map.put(f,sf.getAttribute(f));

        }


        return map;

    }
    public static void writeShape(String shpPath, String encode, String geoType, String shpKey, List<String> attrKeys, List<Map<String, Object>> data, int consumerCount, Consumer<Integer> consumer) throws Exception {
        if (data == null || data.size() == 0) {
            return;
        }
        Map<String, Serializable> params = null;
        ShapefileDataStore ds = null;
        FeatureWriter<SimpleFeatureType, SimpleFeature> writer = null;
        SimpleFeatureTypeBuilder tb = null;
        SimpleFeature feature = null;
        Map<String, Object> row = null;
        Geometry geom = null;
        WKTReader wktReader = new WKTReader();
        try {

            //创建shape文件对象
            File file = new File(shpPath);
            params = new HashMap<>();
            params.put(ShapefileDataStoreFactory.URLP.key, file.toURI().toURL());
            ds = (ShapefileDataStore) new ShapefileDataStoreFactory().createNewDataStore(params);

            //定义图形信息和属性信息
            tb = new SimpleFeatureTypeBuilder();
            //生成84坐标系
//            tb.setCRS(DefaultGeographicCRS.WGS84);
//            tb.setCRS(CRS.decode("EPSG:4490"));
            //生成2000坐标系
            String coorWkt = "GEOGCS[\"GCS_China_Geodetic_Coordinate_System_2000\",DATUM[\"D_China_2000\",SPHEROID[\"CGCS2000\",6378137.0,298.257222101]],PRIMEM[\"Greenwich\",0.0],UNIT[\"Degree\",0.0174532925199433],AUTHORITY[\"EPSG\",4490]]";
            CoordinateReferenceSystem coordinateReferenceSystem = CRS.parseWKT(coorWkt);
            tb.setCRS(coordinateReferenceSystem);
            tb.setDefaultGeometry("the_geom");
            tb.setName("shapefile");
            tb = GeometryUtil.createType(tb, geoType);
            for (String field : attrKeys) {
                tb.add(field, String.class);
            }

            ds.createSchema(tb.buildFeatureType());
            //设置编码
            Charset charset = Charset.forName(encode);
            ds.setCharset(charset);
            //设置Writer
            writer = ds.getFeatureWriter(ds.getTypeNames()[0], Transaction.AUTO_COMMIT);

            //写入文件信息
            for (int i = 0; i < data.size(); i++) {
                feature = writer.next();
                row = data.get(i);
                String wkt = row.get(shpKey).toString();
                geom = wktReader.read(wkt);
                feature.setAttribute("the_geom", geom);
                for (String key : row.keySet()) {
                    if (!key.equals(shpKey)) {
                        if (row.get(key) != null && StringUtils.isNotBlank(row.get(key).toString())) {
                            if (key.length() >= 10) {
                                feature.setAttribute(key.substring(0, 10), row.get(key).toString());
                            } else {
                                feature.setAttribute(key, row.get(key).toString());
                            }
                        } else {
                            if (key.length() >= 10) {
                                feature.setAttribute(key.substring(0, 10), "");
                            } else {
                                feature.setAttribute(key, "");
                            }
                        }
                    }
                }
                if(i%consumerCount==0)
                {
                    consumer.accept(i);
                }
            }
            writer.write();
            writer.close();
            ds.dispose();


            consumer.accept(data.size());

            //添加到压缩文件
            //zipShapeFile(shpPath);
        } catch (Exception e) {
            e.printStackTrace();
            throw e;
        } finally {
            if (params != null) {
                params = null;
            }
            if (tb != null) {
                tb = null;
            }
            if (feature != null) {
                feature = null;
            }
            if (row != null) {
                row = null;
            }
            if (geom != null) {
                geom = null;
            }
            if (ds != null) {
                ds.dispose();
                ds = null;
            }
            if (writer != null) {
                writer.close();
                writer = null;
            }

        }
    }
    public static ShpObject createShape(String shpPath, String encode, String geoType, String shpKey, List<String> attrKeys) throws Exception {

        ShpObject shpObject = new ShpObject();
        Map<String, Serializable> params = null;
        ShapefileDataStore ds = null;
        SimpleFeatureTypeBuilder tb = null;
        try {

            //创建shape文件对象
            File file = new File(shpPath);
            params = new HashMap<>();
            params.put(ShapefileDataStoreFactory.URLP.key, file.toURI().toURL());
            ds = (ShapefileDataStore) new ShapefileDataStoreFactory().createNewDataStore(params);

            //定义图形信息和属性信息
            tb = new SimpleFeatureTypeBuilder();
            //生成84坐标系
//            tb.setCRS(DefaultGeographicCRS.WGS84);
//            tb.setCRS(CRS.decode("EPSG:4490"));
            //生成2000坐标系
            String coorWkt = "GEOGCS[\"GCS_China_Geodetic_Coordinate_System_2000\",DATUM[\"D_China_2000\",SPHEROID[\"CGCS2000\",6378137.0,298.257222101]],PRIMEM[\"Greenwich\",0.0],UNIT[\"Degree\",0.0174532925199433],AUTHORITY[\"EPSG\",4490]]";
            CoordinateReferenceSystem coordinateReferenceSystem = CRS.parseWKT(coorWkt);
            tb.setCRS(coordinateReferenceSystem);
            tb.setDefaultGeometry(shpKey);
            tb.setName("the_geom");
            tb = GeometryUtil.createType(tb, geoType);
            for (String field : attrKeys) {
                tb.add(field, String.class);
            }
            Charset charset = Charset.forName(encode);
            ds.setCharset(charset);
            ds.createSchema(tb.buildFeatureType());

            //设置Writer
            FeatureWriter<SimpleFeatureType, SimpleFeature> writer = ds.getFeatureWriter(ds.getTypeNames()[0], Transaction.AUTO_COMMIT);
            shpObject.setDataStore(ds);
            shpObject.setWriter(writer);
            return shpObject;
        } catch (Exception e) {
            e.printStackTrace();
            throw e;
        } finally {
            if (params != null) {
                params = null;
            }
            if (tb != null) {
                tb = null;
            }
        }
    }
    public static void appendShape(ShpObject shpObject, String shpKey, List<Map<String, Object>> data,List<String>fields, int consumerCount, Consumer<Integer> consumer) throws Exception {
        if (data == null || data.size() == 0) {
            return;
        }

        FeatureWriter<SimpleFeatureType, SimpleFeature> writer = shpObject.getWriter();
        SimpleFeature feature = null;
        Map<String, Object> row = null;
        Geometry geom = null;
        WKTReader wktReader = new WKTReader();
        try {

            //写入文件信息
            for (int i = 0; i < data.size(); i++) {
                feature = writer.next();
                row = data.get(i);
                String wkt = row.get(shpKey).toString();
                geom = wktReader.read(wkt);
                feature.setAttribute("the_geom", geom);
                for (String key : row.keySet()) {
                    if(!fields.contains(key))
                    {
                        continue;
                    }
                    if (!key.equals(shpKey)) {
                        if (row.get(key) != null && StringUtils.isNotBlank(row.get(key).toString())) {
                            if (key.length() >= 10) {
                                feature.setAttribute(key.substring(0, 10), row.get(key).toString());
                            } else {
                                feature.setAttribute(key, row.get(key).toString());
                            }
                        } else {
                            if (key.length() >= 10) {
                                feature.setAttribute(key.substring(0, 10), "");
                            } else {
                                feature.setAttribute(key, "");
                            }
                        }
                    }
                }
                if(i%consumerCount==0)
                {
                    consumer.accept(i);
                }
            }
            consumer.accept(data.size());
        } catch (Exception e) {
            e.printStackTrace();
            throw e;
        } finally {
            if (feature != null) {
                feature = null;
            }
            if (row != null) {
                row = null;
            }
            if (geom != null) {
                geom = null;
            }
        }
    }
    public static void dispose(ShpObject shpObject) throws IOException {

        shpObject.getWriter().write();
        shpObject.getWriter().close();
        shpObject.getDataStore().dispose();

    }
}
