package com.geoway.landteam.landcloud.service.util.hdfs;

import java.util.function.Function;

/**
 * try-catch结果的包装类
 * @Author Waves
 * @Date 2023/4/4
 * @Description 抄的Rust里面的异常处理机制
 **/
public abstract class Result<T> {

    private Result() {}

    public static <T> Result<T> ok(T value) {
        return new OK<>(value);
    }

    public static Result err(Throwable error) {
        return new ERROR(error);
    }

    /**
     * 返回是否正确处理
     * @return
     */
    public boolean isOk() {
        return this instanceof OK;
    }

    /**
     * 返回是否抛出了异常
     * @return
     */
    public boolean isErr() {
        return this instanceof ERROR;
    }

    /**
     * 获取被包装的结果或抛出运行时异常
     * @return
     */
    public T unwrap(){
        if (isOk()) {
            return this.get();
        } else {
            throw new RuntimeException((Throwable)this.get());
        }
    }

    public<R> Result<R> map(Function<T,R> mapper){
        if (isOk()) {
            return Result.ok(mapper.apply(this.get()));
        } else {
            throw new RuntimeException((Throwable)this.get());
        }
    }
    /**
     * 获取被包装的结果或默认值
     * @return
     */
    public T unwrapElse(Function<Throwable,T> func) {
        if (isOk()) {
            return this.get();
        } else {
            return func.apply((Throwable)this.get());
        }
    }
    protected abstract T get();

    public static class OK<T> extends Result<T>{
        private final T value;

        public OK(T value) {
            this.value = value;
        }

        @Override
        protected T get() {
            return value;
        }
    }
    public static class ERROR<T> extends Result<Throwable>{
        private final Throwable e;

        public ERROR(Throwable e) {
            this.e = e;
        }

        @Override
        protected Throwable get() {
            return e;
        }
    }
}
