package com.geoway.design.base.base.controller;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.geoway.design.base.base.domain.RestResult;
import com.geoway.design.base.support.query.MyBatisQueryMapperUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.*;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.IService;

import cn.hutool.core.bean.BeanUtil;
import io.swagger.annotations.ApiOperation;

/**
 * @description: 基础接口
 * @author: wujing
 * @date: 2024-08-28 10:42
 */
public class BaseCrudController<T, S extends IService<T>> {

    protected final S service;
    protected final MyBatisQueryMapperUtils<T> qmu;

    public BaseCrudController(S s, T t) {
        this.service = s;
        this.qmu = new MyBatisQueryMapperUtils<>(t);
    }

    @ApiOperation("根据id获取内容")
    @GetMapping("/{id}")
    public RestResult<T> queryById(@PathVariable String id) throws Exception {
        return RestResult.success(service.getById(id));
    }

    @ApiOperation("根据条件获取单个内容")
    @GetMapping("/filter")
    public RestResult<T> queryByFilter(String filterParam) throws Exception {
        QueryWrapper<T> wrapper = getWrapper(filterParam);
        wrapper.last("limit 1");
        return RestResult.success(service.getOne(wrapper));
    }

    @ApiOperation("获取列表")
    @GetMapping("/list")
    public RestResult<List<T>> queryList(String filterParam, String sortParam) throws Exception {
        QueryWrapper<T> wrapper = getWrapper(filterParam, sortParam);
        return RestResult.success(service.list(wrapper));
    }

    @ApiOperation("分页查询")
    @GetMapping("/page")
    public RestResult<IPage<T>> queryList(String filterParam, String sortParam, Integer page, Integer size)
        throws Exception {
        QueryWrapper<T> wrapper = getWrapper(filterParam, sortParam);
        IPage<T> pages = new Page<>(page, size);
        return RestResult.success(service.page(pages, wrapper));
    }

    @ApiOperation("根据id保存或更新")
    @PostMapping("/saveOrUpdate")
    public RestResult<String> saveOrUpdate(@RequestBody T t) throws Exception {
        service.saveOrUpdate(t);
        Map<String, Object> map = BeanUtil.beanToMap(t);
        return RestResult.success(map.get("id").toString());
    }

    @ApiOperation("根据id删除")
    @DeleteMapping("/{id}")
    public RestResult<String> delete(@PathVariable String id) throws Exception {
        service.removeById(id);
        return RestResult.success("删除成功！");
    }

    @ApiOperation("根据ids批量删除")
    @DeleteMapping("/deleteByIds")
    public RestResult<String> deleteBatch(@RequestParam String ids) throws Exception {
        service.removeByIds(Arrays.asList(ids.split(",")));
        return RestResult.success("删除成功！");
    }

    private QueryWrapper<T> getWrapper(String filterParam, String sortParam) throws Exception {
        if (StringUtils.isNotBlank(filterParam) && StringUtils.isNotBlank(sortParam)) {
            return qmu.queryMapper(filterParam, sortParam);
        } else if (StringUtils.isNotBlank(filterParam)) {
            return qmu.queryMapper(filterParam);
        } else if (StringUtils.isNotBlank(sortParam)) {
            return qmu.queryMapperSort(sortParam);
        }
        return new QueryWrapper<>();
    }

    private QueryWrapper<T> getWrapper(String filterParam) throws Exception {
        return qmu.queryMapper(filterParam);
    }

}
