package com.geoway.design.base.base.domain;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.geoway.design.base.exception.ServiceException;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.util.Objects;

/**
 * Description: 请求统一返回结果<br>
 *
 * @author linkang
 * @version 1.0
 * @date 2022-05-19 15:51
 * @since JDK 1.8
 */
@Data
public class RestResult<T> implements Serializable {

    /**
     * 错误码
     */
    @ApiModelProperty(value = "请求状态码", example = "200")
    private Integer code;

    /**
     * 返回数据
     */
    @ApiModelProperty(value = "返回数据")
    private T data;

    /**
     * 错误提示，用户可阅读
     */
    @ApiModelProperty(value = "错误提示，用户可阅读")
    private String msg;

    /**
     * 将传入的 result 对象，转换成另外一个泛型结果的对象
     *
     * <p>因为 A 方法返回的 RestResult 对象，不满足调用其的 B 方法的返回，所以需要进行转换。
     *
     * @param result 传入的 result 对象
     * @param <T>    返回的泛型
     * @return 新的 RestResult 对象
     */
    public static <T> RestResult<T> error(RestResult<?> result) {
        return error(result.getCode(), result.getMsg());
    }

    public static <T> RestResult<T> error(Integer code, String message) {
        Assert.isTrue(!ResultCodeEnum.SUCCESS.getCode().equals(code), "code 必须是错误的！");
        RestResult<T> result = new RestResult<>();
        result.code = code;
        result.msg = message;
        return result;
    }

    public static <T> RestResult<T> error(ResultCodeEnum resultCodeEnum) {
        return error(resultCodeEnum.getCode(), resultCodeEnum.getMessage());
    }

    public static <T> RestResult<T> success(T data) {
        RestResult<T> result = new RestResult<>();
        result.code = ResultCodeEnum.SUCCESS.getCode();
        result.data = data;
        result.msg = "操作成功！";
        return result;
    }

    public static <T> RestResult<T> success(T data, String msg) {
        RestResult<T> result = new RestResult<>();
        result.code = ResultCodeEnum.SUCCESS.getCode();
        result.data = data;
        result.msg = msg;
        return result;
    }

    public static boolean isSuccess(Integer code) {
        return Objects.equals(code, ResultCodeEnum.SUCCESS.getCode());
    }

    public static <T> RestResult<T> error(ServiceException serviceException) {
        return error(serviceException.getCode(), serviceException.getMessage());
    }

    /**
     * 避免 jackson 序列化
     *
     * @return 布尔值
     */
    @JsonIgnore
    public boolean isSuccess() {
        return isSuccess(code);
    }

    /**
     * 避免 jackson 序列化
     *
     * @return 布尔值
     */
    @JsonIgnore
    public boolean isError() {
        return !isSuccess();
    }

    /**
     * 判断是否有异常。如果有，则抛出 {@link ServiceException} 异常
     */
    public void checkError() throws ServiceException {
        if (isSuccess()) {
            return;
        }
        // 业务异常
        throw new ServiceException(msg, code);
    }
}
