package com.geoway.design.base.support;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.lang.tree.Tree;
import cn.hutool.core.lang.tree.TreeNodeConfig;
import cn.hutool.core.lang.tree.TreeUtil;
import cn.hutool.core.util.ReflectUtil;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author: linkang
 * @Date: 2021/7/8 11:33
 * @Description: list数据转为tree
 */
public class BaseTreeUtil {
    public static <T> List<Tree<String>> ListBuildTree(
            List<T> list, Class<?> clazz, TreeNodeConfig treeNodeConfig) {
        // 转换器
        List<Tree<String>> treeNodes =
                TreeUtil.build(
                        list,
                        "-1",
                        treeNodeConfig,
                        (t, tree) -> {
                            Field[] fields = ReflectUtil.getFieldsDirectly(clazz, true);
                            for (Field field : fields) {
                                String fieldName = field.getName();
                                Object fieldValue = ReflectUtil.getFieldValue(t, field);
                                tree.putExtra(fieldName, fieldValue);
                            }
                        });
        return treeNodes;
    }
    /**
     * 返回树状数据
     *
     * @param tList 全部数据
     * @param treeNodeConfig 配置信息
     * @param parentId 根目录id
     * @param <T> 泛型类型
     * @return 树状数据
     */
    public static <T> List<T> listToTree(List<T> tList, TreeNodeConfig treeNodeConfig, String parentId) {
        // 查出所有节点
        if (tList != null && !tList.isEmpty()) {
            // 组装父子树形结构
            return tList.stream()
                    // 获取一级节点
                    .filter(root -> parentId.equals(ReflectUtil.getFieldValue(root, treeNodeConfig.getParentIdKey())))
                    // 插入子节点到children属性
                    .peek(children -> ReflectUtil.setFieldValue(children, treeNodeConfig.getChildrenKey(),
                            getChildren(children, tList, treeNodeConfig)))
                    // 根据sort属性排序（空值居后）
                    .sorted(Comparator.nullsLast(Comparator
                            .comparingInt(o -> Convert.toInt(ReflectUtil.getFieldValue(o, treeNodeConfig.getWeightKey()), 0))))
                    // 将流数据转成List类型
                    .collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    /**
     * 根据全部节点，获取当前节点的子节点
     * @param root 当前根节点
     * @param all 全部数据
     * @param treeNodeConfig 配置信息
     * @param <T> 实体泛型
     * @return 树形结构
     */
    private static <T> List<T> getChildren(T root, List<T> all, TreeNodeConfig treeNodeConfig) {
        return all.stream()
                // 获取当前节点的子节点
                .filter(menu -> ReflectUtil.getFieldValue(root, treeNodeConfig.getIdKey())
                        .equals(ReflectUtil.getFieldValue(menu, treeNodeConfig.getParentIdKey())))
                // 递归调用，查找子节点插入至children属性
                .peek(children -> ReflectUtil.setFieldValue(children, treeNodeConfig.getChildrenKey(),
                        getChildren(children, all, treeNodeConfig)))
                // 对子节点根据sort属性排序（空值居后）
                .sorted(Comparator.nullsLast(Comparator
                        .comparingInt(o -> Convert.toInt(ReflectUtil.getFieldValue(o, treeNodeConfig.getWeightKey()), 0))))
                // 将流数据转成List类型
                .collect(Collectors.toList());
    }

    public static <T> List<Tree<String>> ListBuildTree(
            List<T> list, Class<?> clazz, TreeNodeConfig treeNodeConfig, String parentId) {
        // 转换器
        List<Tree<String>> treeNodes =
                TreeUtil.build(
                        list,
                        parentId,
                        treeNodeConfig,
                        (t, tree) -> {
                            Field[] fields = ReflectUtil.getFieldsDirectly(clazz, true);
                            for (Field field : fields) {
                                String fieldName = field.getName();
                                Object fieldValue = ReflectUtil.getFieldValue(t, field);
                                tree.putExtra(fieldName, fieldValue);
                            }
                        });
        return treeNodes;
    }
}
