package com.geoway.design.base.support.query;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.annotation.TableId;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringPool;
import com.github.yulichang.query.MPJQueryWrapper;
import com.github.yulichang.wrapper.MPJLambdaWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;
import org.springframework.format.annotation.DateTimeFormat;

import java.lang.reflect.Field;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * myBatis条件工具类 ( 仅适用于queryMapper )
 *
 * @author  daidd
 * @date 2021年10月14日14:07:47
 */
@Configuration
@Slf4j
public class MPJQueryMapperUtil<T> {

    /**
     * @param filterParams ( 查询字段_条件类型_值1_值2;OR;查询字段_条件类型_值1_值2; 注：值和OR可省略 )
     *                     兼容老式写法（Q_name_S_EQ=mike)
     * 	 从请求对象获取查询参数,并进行构造
     * 	 <p>
     * 	 参数名格式必须为: Q_firstName_S_EQ 其中Q_表示该参数为查询的参数，firstName查询的字段名称，和实体类的字段名称一致
     * 	 S代表该参数的类型为字符串类型,该位置的其他值有： D=日期，N=number,S=字符串 EQ代表等于。 该位置的其他值有：<br/>
     * 	 LT，GT，EQ，LE，GE,LK,NE<br/>
     * 	要别代表<,>,=,<=,>=,like的条件查询
     * 	 <p>
     * 	支持带刮号的复杂表达式：如
     * 	 Q_fieldA_N_LE=100;Q_fieldB_N_LT=200;(QOR_fieldC_N_LE=300;QOR_fieldD_N_LT=400;(Q_fieldE_N_LE=500;QOR_fieldF_N_LT=600;))
     *
     * @param sortParams   (sort_字段_排序条件(asc,desc))
     * @param entityClass  ( 实体类.class )
     * @return QueryWrapper
     */
    public MPJLambdaWrapper<T> queryMapper(String filterParams, String sortParams, Class entityClass) throws Exception {
        MPJLambdaWrapper<T> queryWrapper = this.queryMapper(filterParams, entityClass);
        if (StrUtil.isNotEmpty(sortParams)) {
            String[] sortParamArr = sortParams.split(";");
            for (String sortParam : sortParamArr) {
                String[] sortCondition = sortParam.split("_");
                String sortFieldName = sortCondition[1];
                String sortType = sortCondition[2];

                // 根据字段名获取该属性
                Field field = entityClass.getDeclaredField(sortFieldName);
                // 通过对注解的反射获取数据库对应字段
                String fieldName = findTableFiled(field);
                String column = queryWrapper.getAlias() + StringPool.DOT + fieldName;

                if ("desc".equalsIgnoreCase(sortType)) {
                    queryWrapper.orderByDesc(column);
                } else {
                    queryWrapper.orderByAsc(column);
                }
            }
        }


        return queryWrapper;
    }


    public MPJLambdaWrapper<T> queryMapper(String filterParams, Class entityClass) throws Exception {
        MPJLambdaWrapper<T> queryWrapper = new MPJLambdaWrapper<>();
        if (StrUtil.isBlank(filterParams)) {
            return queryWrapper;
        }
        try {
            filterParams = nestingManage(filterParams,queryWrapper,entityClass);
            //是否原始写法
            boolean isOldFilter = filterParams.startsWith("Q_")  || filterParams.startsWith("QOR_") ;
            if (isOldFilter) {
                this.bulidQueryWrapper2(filterParams, queryWrapper, entityClass);
            }else{
                this.bulidQueryWrapper(filterParams, queryWrapper, entityClass);
            }
            return queryWrapper;
        } catch (Exception e) {
            log.error("构建查询表达式出错:",e);
            return queryWrapper;
        }
    }

    /**
     * QueryWrapper 通过循环追加不同的查询条件
     *
     * @param filterParams  ( Q_name_S_EQ=mike;QOR_ )
     * @param queryWrapper (Entity对象封装操作类)
     * @param entityClass  (实体类的Class)
     * @return
     */
    private void bulidQueryWrapper(String filterParams, MPJLambdaWrapper<T> queryWrapper, Class entityClass) throws Exception {

        // 解析filter，获取所有条件集
        String[] filtersSplit = filterParams.split(";");
        // 循环对queryWrapper进行添加操作
        for (String filterParam : filtersSplit) {
            this.buildMybatisQuery(filterParam, queryWrapper, entityClass);
        }
    }

    /**
     * QueryWrapper 通过循环追加不同的查询条件
     *
     * @param filterParam  ( 查询字段_条件类型_值1_值2 注：值2可省略 )
     * @param queryWrapper (Entity对象封装操作类)
     * @param entityClass  (实体类的Class)
     * @return
     */
    private void buildMybatisQuery(String filterParam, MPJLambdaWrapper<T> queryWrapper, Class entityClass) throws Exception {
        if ("OR".equalsIgnoreCase(filterParam)) {
            queryWrapper.or();
            return;
        }
        String[] filters = filterParam.split("_");
        if (filters.length < 3) {
            return;
        }

        // 根据字段名获取该属性
        Field field = entityClass.getDeclaredField(filters[0]);
        // 通过对注解的反射获取数据库对应字段，并加上别名
        String fieldName = findTableFiled(field);
        String column = queryWrapper.getAlias() + StringPool.DOT + fieldName;
        // 获得查询条件类型
        String type = filters[1];
        String value1 = null;
        String value2 = null;
        // 判断是否存在字段值1
        if (filters.length >= 3) {
            value1 = filters[2];
        }
        // 判断是否存在字段值2，一般用于between
        if (filters.length >= 4) {
            value2 = filters[3];
        }
        Object result1=value1;
        Object result2=value2;
        if(!type.contains("IN")) {
            // 类型转换
            result1 = valueOfType(field, value1);
            result2 = valueOfType(field, value2);
        }else{
            result1 = Arrays.stream(value1.split(",")).map(f->valueOfType(field, f)).collect(Collectors.toList());
        }
        this.buildQueryWrapper(queryWrapper, column, type, result1, result2);
    }

    /**
     * QueryWrapper 通过循环追加不同的查询条件
     *
     * @param filterParams  ( Q_name_S_EQ=mike;QOR_ )
     * @param queryWrapper (Entity对象封装操作类)
     * @param entityClass  (实体类的Class)
     * @return
     */
    private void bulidQueryWrapper2(String filterParams, MPJLambdaWrapper<T> queryWrapper, Class entityClass) throws Exception {


        List<QueryFilterParam>  queryFilterParams = QueryParamUtil.parseQueryFilterParams(filterParams);
        for(QueryFilterParam queryFilterParam: queryFilterParams){
            this.bulidQueryWrapper(queryFilterParam,queryWrapper,entityClass);
        }

    }

    /**
     *
     * @param queryFilterParam 查询参数
     * @param queryWrapper
     * @param entityClass
     * @throws Exception
     */
    private void bulidQueryWrapper( QueryFilterParam queryFilterParam,MPJLambdaWrapper<T> queryWrapper, Class entityClass){
        // 判断是否子查询
        if(!queryFilterParam.isNest()){

            Field field = null;
            try {
                field = entityClass.getDeclaredField(queryFilterParam.getParamName());
            } catch (NoSuchFieldException e) {
                e.printStackTrace();
            }
            if(field == null){
                return;
            }
            // 通过对注解的反射获取数据库对应字段,并加上别名
            String fieldName = findTableFiled(field);
            String column = queryWrapper.getAlias() + StringPool.DOT + fieldName;

            String value1 = queryFilterParam.getParamValue();
            String value2 = null;
            if( queryFilterParam.getOperationMode() == OperationModel.BETWEEN
                    || queryFilterParam.getOperationMode() == OperationModel.BTW
                    || queryFilterParam.getOperationMode() == OperationModel.NOTBETWEEN
                    || queryFilterParam.getOperationMode() == OperationModel.NOTBTW){
                String[] values = queryFilterParam.getParamValue().split(",");
                // 判断是否存在字段值1
                if (values.length >= 1) {
                    value1 = values[0];
                }
                // 判断是否存在字段值2，一般用于between
                if (values.length >= 2) {
                    value2 = values[1];
                }
            }
            Object result1=value1;
            Object result2=value2;
            if(queryFilterParam.getOperationMode() == OperationModel.IN
                    || queryFilterParam.getOperationMode() == OperationModel.NOTIN) {
                // 类型转换
                Field finalField = field;
                result1 = Arrays.stream(value1.split(",")).map(f->valueOfType(finalField, f)).collect(Collectors.toList());
            }else{
                result1 = valueOfType(field, value1);
                result2 = valueOfType(field, value2);
            }
            String type = queryFilterParam.getOperation();
            buildQueryWrapper(queryWrapper, column, type, result1, result2);
            if (queryFilterParam.getJoinModel() == QueryFilterParam.JoinModel.OR) {
                queryWrapper.or();
            }
        }else{
            //如果是子查询，
            List<QueryFilterParam> subQueryFilterParams = queryFilterParam.getChildrens();
            if(subQueryFilterParams == null || subQueryFilterParams.size() == 0){
                return;
            }
            if (queryFilterParam.getJoinModel() == QueryFilterParam.JoinModel.OR) {
                queryWrapper.or(subQueryWrapper -> {
                    for(QueryFilterParam subFilterParam: subQueryFilterParams){
                        this.bulidQueryWrapper(subFilterParam, subQueryWrapper,entityClass);
                    }
                });
            }else{
                queryWrapper.and(subQueryWrapper -> {
                    for(QueryFilterParam subFilterParam: subQueryFilterParams){
                        this.bulidQueryWrapper(subFilterParam, subQueryWrapper,entityClass);
                    }
                });
            }
        }
    }

    /**
     * 创建查询条件
     *
     * @param queryWrapper
     * @param column       字段名
     * @param type         查询类型
     * @param result1      查询值1
     * @param result2      查询值2
     * @return
     */
    private MPJLambdaWrapper<T> buildQueryWrapper(MPJLambdaWrapper<T> queryWrapper, String column, String type, Object result1, Object result2) {
        // 判断具体条件
        switch (type) {
            // 等于
            case "EQ":
                queryWrapper.eq(column, result1);
                break;
            // 不等于
            case "NE":
                queryWrapper.ne(column, result1);
                break;
            // 大于
            case "GT":
                queryWrapper.gt(column, result1);
                break;
            // 大于等于
            case "GE":
                queryWrapper.ge(column, result1);
                break;
            // 小于
            case "LT":
                queryWrapper.lt(column, result1);
                break;
            // 小于等于
            case "LE":
                queryWrapper.le(column, result1);
                break;
            // ...之间
            case "BTW":
            case "BETWEEN":
                queryWrapper.between(column, result1, result2);
                break;
            // 不属于...之间
            case "NOTBETWEEN":
                queryWrapper.notBetween(column, result1, result2);
                break;
            // 模糊查询 存在%值%
            case "LK":
            case "LIKE":
                queryWrapper.like(column, result1);
                break;
            // 模糊查询 不存在%值%
            case "NLK":
            case "NOTLIKE":
                queryWrapper.notLike(column, result1);
                break;
            // 模糊查询 %值
            case "LFK":
            case "LIKELEFT":
                queryWrapper.likeLeft(column, result1);
                break;
            // 模糊查询 值%
            case "RHK":
            case "LIKERIGHT":
                queryWrapper.likeRight(column, result1);
                break;
            // 为空的
            case "ISNULL":
                queryWrapper.isNull(column);
                break;
            // 不为空的
            case "ISNOTNULL":
                queryWrapper.isNotNull(column);
                break;
            // in查询
            case "IN":
                queryWrapper.in(column, ((List)result1).toArray());
                break;
            // notIn查询
            case "NOTIN":
                queryWrapper.notIn(column, ((List)result1).toArray());
                break;
            default:
                break;
        }
        return queryWrapper;
    }

    /**
     * valueOfType 根据实体类属性的类型进行转换
     *
     * @param field
     * @param value
     * @return
     * @throws ParseException
     */
    private Object valueOfType(Field field, String value) {
        try {
            // 获取字段的类型
            String valueType = field.getType().getName();
            // 返回的结果
            Object result = value;
            if (value != null) {
                // 进行类型转换
                switch (valueType) {
                    case "java.lang.String":
                        return result;
                    case "java.lang.int":
                    case "java.lang.Integer":
                        result = Integer.parseInt(value);
                        break;
                    case "java.lang.long":
                    case "java.lang.Long":
                        result = Long.parseLong(value);
                        break;
                    case "java.lang.short":
                    case "java.lang.Short":
                        result = Short.parseShort(value);
                        break;
                    case "java.lang.double":
                    case "java.lang.Double":
                        result = Double.parseDouble(value);
                        break;
                    case "java.lang.float":
                    case "java.lang.Float":
                        result = Float.parseFloat(value);
                        break;
                    case "java.lang.byte":
                    case "java.lang.Byte":
                        result = Byte.parseByte(value);
                        break;
                    case "java.lang.Boolean":
                    case "java.lang.boolean":
                        result = Boolean.parseBoolean(value);
                        break;
                    case "java.util.Date":
                        // 默认
                        String dateType = "yyyy-MM-dd HH:mm:ss";
                        // 对日期格式进行一个判断
                        boolean isTableField = field.isAnnotationPresent(DateTimeFormat.class);
                        if (isTableField) {
                            DateTimeFormat tableField = field.getAnnotation(DateTimeFormat.class);
                            dateType = tableField.pattern();
                        }
                        switch (dateType) {
                            case "yyyy-MM-dd HH:mm:ss": {
                                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                                result = sdf.parse(value);
                                break;
                            }
                            case "yyyy-MM-dd": {
                                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
                                result = sdf.parse(value);
                                break;
                            }
                            case "yyyy年MM月dd日 HH时mm分ss秒": {
                                SimpleDateFormat sdf = new SimpleDateFormat("yyyy年MM月dd日 HH时mm分ss秒");
                                result = sdf.parse(value);
                                break;
                            }
                            case "yyyy年MM月dd日": {
                                SimpleDateFormat sdf = new SimpleDateFormat("yyyy年MM月dd日");
                                result = sdf.parse(value);
                                break;
                            }
                            default:
                                break;
                        }
                        break;
                    default:
                        break;
                }
            }
            return result;
        }catch (Exception e){
            throw new RuntimeException(e);
        }
    }

    /**
     * 获取数据库字段 （需要在实体类的属性上用到@TableField）
     *
     * @param field
     * @return
     */
    private String findTableFiled(Field field) {
        // 返回的字段
        String fieldValue = null;
        // 判断是否存在@TableField注释
        boolean isTableField = field.isAnnotationPresent(TableField.class);
        // 判断是否存在@TableId注释
        boolean isTableId = field.isAnnotationPresent(TableId.class);
        // 如果存在，则获取注释上的value
        if (isTableField) {
            TableField tableField = field.getAnnotation(TableField.class);
            fieldValue = tableField.value();
        }
        // 如果存在，则获取注释上的value
        if (isTableId) {
            TableId tableId = field.getAnnotation(TableId.class);

            fieldValue =  tableId.value();
        }
        return fieldValue;
    }

    /**
     * 嵌套执行
     * @param filterParams
     * @param queryWrapper
     * @param entityClass
     * @return
     */
    private String nestingManage(String filterParams, MPJLambdaWrapper<T> queryWrapper, Class entityClass){
        int count = (filterParams.length() - filterParams.replace(";(","").length()) / 2;
        for (int i = 0; i < count ; ++i) {
            int start = filterParams.indexOf(";(");
            int end = filterParams.indexOf(");");
            String deleteStr = filterParams.substring(start, end+1);
            String subFilter = filterParams.substring(start + 2, end);
            queryWrapper.and(a->{
                try {
                    boolean isOldFilter = subFilter.startsWith("Q_")  || subFilter.startsWith("QOR_");
                    if (isOldFilter) {
                        this.bulidQueryWrapper2(subFilter, a, entityClass);
                    }else{
                        this.bulidQueryWrapper(subFilter, a, entityClass);
                    }
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
            filterParams = filterParams.replace(deleteStr,"");
        }
        return filterParams;
    }

}
