package com.geoway.design.biz.service.sys.impl;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.geoway.design.base.exception.ServiceException;
import com.geoway.design.base.support.query.MyBatisQueryMapperUtils;
import com.geoway.design.biz.entity.*;
import com.geoway.design.biz.mapper.SysOrganizationMapper;
import com.geoway.design.biz.mapper.SysRegionMapper;
import com.geoway.design.biz.mapper.SysUserMapper;
import com.geoway.design.biz.service.dev.IOrganizationService;
import com.geoway.design.biz.service.sys.*;
import com.geoway.design.biz.mapper.BizManagerRangeMapper;
import com.geoway.design.biz.vo.BizManagerVO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author wangqiang
 * @description 针对表【sys_biz_manager_range】的数据库操作Service实现
 * @createDate 2024-08-29 14:19:11
 */
@Service
public class BizManagerRangeServiceImpl extends ServiceImpl<BizManagerRangeMapper, BizManagerRange>
        implements BizManagerRangeService {

    @Autowired
    private SysUserMapper sysUserMapper;
    @Autowired
    private ISysRegionService sysRegionService;
    @Autowired
    private IOrganizationService organizationService;
    @Autowired
    private ISysRoleService sysRoleService;
    @Autowired
    private ISysPermissionService sysPermissionService;
    @Autowired
    private ISysPermissionGroupService sysPermissionGroupService;
    @Autowired
    private SysOrganizationMapper sysOrganizationMapper;

    @Override
    public void addBizManager(String userids) {
        List<SysUser> sysUsers = sysUserMapper.selectBatchIds(Arrays.asList(userids.split(",")));
        sysUsers.forEach(f -> {
                    if (f.getCatalog() != 1) {
                        // 除管理员外可设置为业务管理员
                        f.setCatalog(2);
                        sysUserMapper.updateById(f);
                    }
                }
        );
    }

    @Override
    public void delBizManager(String userids) {
        List<SysUser> sysUsers = sysUserMapper.selectBatchIds(Arrays.asList(userids.split(",")));
        sysUsers.forEach(f -> {
                    // 设为普通用户
                    f.setCatalog(3);
                    sysUserMapper.updateById(f);

                    // 删除配置的管理项
                    this.remove(new LambdaQueryWrapper<BizManagerRange>().eq(BizManagerRange::getUserid, f.getId()));
                }
        );
    }

    @Override
    public IPage<BizManagerVO> queryBizManagerPage(String filterParam, Integer page, Integer size) throws Exception {
        IPage<BizManagerVO> resultPage = new Page<>();

        // 查询用户
        MyBatisQueryMapperUtils<SysUser> qmu = new MyBatisQueryMapperUtils<>();
        QueryWrapper queryWrapper = qmu.queryMapper(filterParam, SysUser.class);
        queryWrapper.orderByDesc("f_updatetime");
        Page<SysUser> pages = new Page<>(page, size);
        IPage<SysUser> pageUsers = this.sysUserMapper.selectPage(pages, queryWrapper);
        List<SysUser> users = pageUsers.getRecords();
        if (users.size() == 0) return resultPage;
        List<String> userIds = users.stream().map(SysUser::getId).collect(Collectors.toList());

        // 查询用户配置的管理项
        List<BizManagerRange> ranges = this.lambdaQuery().in(BizManagerRange::getUserid, userIds).list();
        List<BizManagerVO> bizManagerVOS = users.stream().map(user -> {
            List<BizManagerRange> userRanges = ranges.stream().filter(f -> user.getId().equals(f.getUserid())).collect(Collectors.toList());
            BizManagerVO bizManagerVO = BizManagerVO.builder().userid(user.getId()).accout(user.getAccout()).aname(user.getAname()).build();

            buildBizManagerRange(bizManagerVO, userRanges);
            return bizManagerVO;
        }).collect(Collectors.toList());

        resultPage.setRecords(bizManagerVOS);
        resultPage.setTotal(pageUsers.getTotal());
        return resultPage;
    }

    @Transactional
    @Override
    public void updateManage(String userid, Integer type, String values) {
        // 先删除再添加
        this.remove(new LambdaQueryWrapper<BizManagerRange>().eq(BizManagerRange::getUserid, userid).eq(BizManagerRange::getManageType, type));
        if(StrUtil.isNotBlank(values)){
            List<BizManagerRange> ranges = Arrays.asList(values.split(",")).stream().map(value ->
                    BizManagerRange.builder().userid(userid).manageType(type).manageValue(value).build()
            ).collect(Collectors.toList());
            this.saveBatch(ranges);
        }

        // 更新用户表的更新时间，让它排在前面
        SysUser sysUser = sysUserMapper.selectById(userid);
        sysUser.setUpdatetime(new Date());
        sysUserMapper.updateById(sysUser);
    }

    @Override
    public List<BizManagerRange> queryManage(String userid, Integer type) {
        return this.lambdaQuery().eq(BizManagerRange::getUserid, userid).eq(BizManagerRange::getManageType, type).list();
    }

    @Override
    public List<String> queryAllManageOrgs(String userid) {
        List<BizManagerRange>  orgs = this.lambdaQuery().eq(BizManagerRange::getUserid, userid).eq(BizManagerRange::getManageType, 2).list();
        if(orgs.size()==0) return new ArrayList<>();

        Set<String> resultSet = new HashSet<>();
        // 查找下级
        orgs.forEach(org->{
            List<SysOrganization> children = sysOrganizationMapper.getOrgListById(org.getManageValue());
            resultSet.addAll(children.stream().map(SysOrganization::getId).collect(Collectors.toList()));
        });
        return new ArrayList<>(resultSet);
    }

    public List<SysOrganization> queryOrgTree(String userid){
        List<BizManagerRange> orgRanges = this.lambdaQuery().eq(BizManagerRange::getUserid,userid).eq(BizManagerRange::getManageType,2).list();
        if(orgRanges.size()==0) return new ArrayList<>();
        List<SysOrganization> orgs = new ArrayList<>();
        for(BizManagerRange org : orgRanges){
            // 查找上级
            List<SysOrganization> parents = sysOrganizationMapper.getAllParentOrg(org.getManageValue());
            parents.forEach(f->{
                Optional<SysOrganization> opt = orgs.stream().filter(m->m.getId().equals(f.getId())).findAny();
                if(!opt.isPresent()){
                    // 上级的不可编辑
                    f.setEditable(false);
                    orgs.add(f);
                }
            });
            // 查找下级
            List<SysOrganization> children = sysOrganizationMapper.getOrgListById(org.getManageValue());
            children.forEach(f->{
                boolean exist = false;
                // 下级的可编辑
                for(SysOrganization m : orgs){
                    if(f.getId().equals(m.getId())){
                        m.setEditable(true);
                        exist = true;
                        break;
                    }
                }
                if(!exist){
                    f.setEditable(true);
                    orgs.add(f);
                }
            });
        }

        // 关联有权限组织的用户
        List<SysOrganization> editableOrgs = orgs.stream().filter( f->Boolean.TRUE.equals(f.getEditable())).collect(Collectors.toList());
        organizationService.bindUsersToOrg(editableOrgs);
        editableOrgs.forEach(f->{
            for(SysOrganization org : orgs){
                if(org.getId().equals(f.getId())){
                    org.setBindUsers(f.getBindUsers());
                    break;
                }
            }
        });

        return organizationService.constructTree(orgs);
    }

    /**
     * 根据管理类型的编码查找其全路径中文展示值
     *
     * @param bizManagerVO
     * @param ranges
     */
    private void buildBizManagerRange(BizManagerVO bizManagerVO, List<BizManagerRange> ranges) {
        // 按type分组
        Map<Integer, List<BizManagerRange>> groups = ranges.stream().collect(Collectors.groupingBy(BizManagerRange::getManageType));
        groups.keySet().forEach(type -> {
            List<String> typeValues = groups.get(type).stream().map(BizManagerRange::getManageValue).collect(Collectors.toList());
            switch (type) {
                case 1: // 辖区
                    bizManagerVO.setRegions(typeValues);
                    try {
                        List<SysRegion> regions = sysRegionService.queryList("regionCode_IN_"+StrUtil.join(",",typeValues),null);
                        regions.forEach(m->{
                            m.setAllname(StrUtil.join("/",sysRegionService.queryAllParentRegion(m.getRegionCode())
                                    .stream().map(SysRegion::getName).collect(Collectors.toList())));
                        });

                        bizManagerVO.setRegionNames(regions.stream().map(SysRegion::getAllname).collect(Collectors.toList()));
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    break;
                case 2: // 组织
                    bizManagerVO.setOrgs(typeValues);
                    bizManagerVO.setOrgNames(typeValues.stream().map(orgId -> {
                        List<SysOrganization> orgs = organizationService.queryAllParentOrg(orgId);
                        if (orgs.size() == 0) return "";
                        else
                            return StrUtil.join("/", orgs.stream().map(SysOrganization::getName).collect(Collectors.toList()));
                    }).collect(Collectors.toList()));
                    break;
                case 3: // 角色
                    bizManagerVO.setRoles(typeValues);
                    bizManagerVO.setRoleNames(typeValues.stream().map(roleId -> {
                        List<SysRole> roles = sysRoleService.queryAllParentRole(roleId);
                        if (roles.size() == 0) return "";
                        else
                            return StrUtil.join("/", roles.stream().map(SysRole::getName).collect(Collectors.toList()));
                    }).collect(Collectors.toList()));
                    break;
                case 4: // 权限
                    bizManagerVO.setPermissions(typeValues);
                    List<SysPermission> permissions = sysPermissionService.listByIds(typeValues);
                    List<String> groupIds = permissions.stream().map(SysPermission::getGroupid).filter(Objects::nonNull).collect(Collectors.toList());
                    List<SysPermissionGroup> permissionGroups = sysPermissionGroupService.listByIds(groupIds);
                    bizManagerVO.setPermissionNames(permissions.stream().map(permission -> {
                        String str = permission.getName();
                        if (permission.getGroupid() != null) {
                            Optional<SysPermissionGroup> groupOptional = permissionGroups.stream().filter(f -> permission.getGroupid().equals(f.getId())).findAny();
                            if (groupOptional.isPresent()) {
                                str = groupOptional.get().getName() + "/" + str;
                            }
                        }
                        return str;
                    }).collect(Collectors.toList()));
                    break;
                default:
                    break;
            }
        });
    }
}




