package com.geoway.design.biz.service.dev.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.geoway.design.base.support.encryption.AESUtils;
import com.geoway.design.biz.entity.SysRegion;
import com.geoway.design.biz.entity.SysUser;
import com.geoway.design.biz.entity.SysUserSecurity;
import com.geoway.design.biz.mapper.SysRegionMapper;
import com.geoway.design.biz.mapper.SysUserMapper;
import com.geoway.design.biz.mapper.SysUserSecurityMapper;
import com.geoway.design.biz.util.PasswordUtil;
import com.geoway.sso.client.rpc.Result;
import com.geoway.sso.client.rpc.SsoUser;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author ALMJ
 * @desc 用户登录验证逻辑(本地登录)
 */
@Service("userService")
@Slf4j
public class LocalLoginServiceImpl extends AbstractLoginServiceImpl {

    @Autowired
    private SysUserMapper sysUserMapper;
    @Autowired
    private SysUserSecurityMapper sysUserSecurityMapper;
    @Autowired
    private SysRegionMapper sysRegionMapper;
    @Autowired
    private PasswordUtil passwordUtil;
    @Value("${password.forceChangeInitPWD:true}")
    private Boolean forceChangeInitPWD;

    @Value("${password.security.enable:false}")
    private Boolean securityEnable;
    @Value("${password.security.expireDays:90}")
    private int expireDays;
    @Value("${password.security.failureTimes:5}")
    private int allowFailureTimes;
    @Value("${password.security.autoUnlockTime:30}")
    private int autoUnlockMinute;

    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    @Override
    public Result<SsoUser> login(String username, String password,String tel, String uuid, String code, HttpServletRequest req) {
        //校验验证码
        validateCaptcha(uuid,code);
        //账号用户名验证
        return checkLogin(username,password,tel,null);
    }


    @Override
    public boolean checkIsResetPassword(String userId){
        SysUserSecurity sysUserSecurity = sysUserSecurityMapper.selectById(userId);
        if(sysUserSecurity == null){
            return  false;
        }
        if(forceChangeInitPWD && StrUtil.isNotBlank(sysUserSecurity.getResetPassword()) ){
            return  true;
        }
        return  false;
    }

    protected Result<SsoUser> checkLogin(String username,String password,String tel,List<String> sources){

        SsoUser ssoUser = new SsoUser();
        try {
            LambdaQueryWrapper<SysUser> queryWrapper = Wrappers.lambdaQuery();
            queryWrapper.eq(SysUser::getAccout, username);
            if(StrUtil.isNotBlank(tel)){
                queryWrapper.eq(SysUser::getTelEncrypt, AESUtils.encrypt(tel,AESUtils.KEY));
            }
            int iCount = sysUserMapper.selectCount(queryWrapper);
            if (iCount == 0){
                queryWrapper.clear();
                queryWrapper.and(gw->gw.eq(SysUser::getName, username)
                        .or()
                        .eq(SysUser::getAccout, username)
                        .or()
                        .eq(SysUser::getAname, username));
                if(StrUtil.isNotBlank(tel)){
                    queryWrapper.eq(SysUser::getTel,tel);
                }
            }

            if(sources!=null&&sources.size()!=0){
                queryWrapper.and(gw->gw.in(SysUser::getSource,sources));
            }
            queryWrapper.last(" limit 1");
            SysUser user = sysUserMapper.selectOne(queryWrapper);
            if (user == null || user.getId() == null) {
                return Result.createError("用户不存在或者密码错误");
            }
            if (!user.getStatus().toString().equals("1")) {
                return Result.createError("用户已停用或者状态异常，请联系管理员！");
            }

            Date now = new Date();

            //先判断用户是否已上锁
            SysUserSecurity sysUserSecurity = sysUserSecurityMapper.selectById(user.getId());
            if (sysUserSecurity == null){
                return Result.createError("用户不存在或者密码错误。");
            }

            if (sysUserSecurity.getLock() != null && sysUserSecurity.getLock() == 1) {
                if(sysUserSecurity.getUnlocktime() != null && sysUserSecurity.getUnlocktime().before(now)){

                    sysUserSecurity.unlock();
                    LambdaUpdateWrapper<SysUserSecurity> updateWrapper = Wrappers.lambdaUpdate();
                    updateWrapper.set(SysUserSecurity::getLock, 0);
                    updateWrapper.set(SysUserSecurity::getFailedtimes, 0);
                    updateWrapper.set(SysUserSecurity::getUnlocktime,null);
                    updateWrapper.eq(SysUserSecurity::getId,user.getId());

                    this.sysUserSecurityMapper.update(null,updateWrapper);
                }else if(sysUserSecurity.getUnlocktime() != null){
                    String timeMsg =  DateUtil.format(sysUserSecurity.getUnlocktime(),"yyyy-MM-dd HH:mm:ss");
                    return Result.createError("用户已被锁定，系统将于" + timeMsg + "后自动解锁");
                }else{
                    return Result.createError("用户被锁定，请联系管理员!");
                }
            }

            boolean errTimesArrived = false;
            if ( !passwordUtil.getMD5Password(password).equals(sysUserSecurity.getPassword())) {
                if(securityEnable){
                    sysUserSecurity.setLastfailedtime(new Date());
                    sysUserSecurity.addOneFailureTimes();
                    this.sysUserSecurityMapper.updateById(sysUserSecurity);

                    errTimesArrived = this.checkedIsErrpasswordInDuration(user.getId());

                    //连续输错则自动锁定
                    if(errTimesArrived){
                        sysUserSecurity.lock(autoUnlockMinute);
                        this.sysUserSecurityMapper.updateById(sysUserSecurity);
                        String key = "errPwdTimes-" + user.getId();
                        redisTemplate.delete(key);
                    }
                }
                return Result.createError("用户不存在或者密码错误。");
            }



//            if(forceChangeInitPWD && StrUtil.isNotBlank(sysUserSecurity.getResetPassword()) ){
//                 return  Result.create(Oauth2Constant.RESETPASSWORD_CODE,"密码已重置，需重新修改密码");
//            }

            if(securityEnable){
                long timeT = System.currentTimeMillis() - sysUserSecurity.getCreatetime().getTime();
                //如果密码超过设置的有效天或者超过过期时间则过期
                long expireT = expireDays * 24L * 3600000;
                boolean isExipred = timeT > expireT;
                if (sysUserSecurity.getExpiretime() != null && sysUserSecurity.getExpiretime().getTime() >  System.currentTimeMillis()) {
                    isExipred = true;
                }
                if(isExipred){

                    return Result.createResetPassword("密码已过期或超过" + this.expireDays + "天未修改，需要修改密码才能使用!");
                }
            }

            List<SysRegion> regions = sysRegionMapper.queryUserRegions(user.getId());
            String regionCode = regions.stream().map(i -> i.getCode()).collect(Collectors.joining(","));
            String regionName = regions.stream().map(i -> i.getName()).collect(Collectors.joining(","));
            ssoUser.setId(user.getId());
            ssoUser.setUserid(user.getId());
            ssoUser.setUserName(user.getAname());
            ssoUser.setLoginName(user.getAccout());
            ssoUser.setRegionCode(regionCode);
            ssoUser.setRegionName(regionName);
            ssoUser.setUserCatalog(user.getCatalog());
            ssoUser.setSource(getUserSys());
            return Result.createSuccess(ssoUser);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e.getMessage());
        }
    }

    private boolean checkedIsErrpasswordInDuration(String useId) {
        boolean tag = false;
        String key = "errPwdTimes-" + useId;
        boolean hasKey = redisTemplate.hasKey(key);
        if (hasKey) {
            String  strNum = redisTemplate.opsForValue().get(key);
            Integer num = Integer.parseInt(strNum);
            if (num >= this.allowFailureTimes) {
                tag = true;
            } else {
                num++;
                redisTemplate.opsForValue().set(key, num + "");
            }
        } else {
            redisTemplate.opsForValue().set(key, "1");
            redisTemplate.expire(key, 5, TimeUnit.MINUTES);
        }
        return tag;
    }

}
