package com.geoway.design.biz.service.dev.impl;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.geoway.design.base.exception.ServiceException;
import com.geoway.design.base.support.StringUtils;
import com.geoway.design.base.support.query.MyBatisQueryMapperUtils;
import com.geoway.design.biz.entity.SysOrganization;
import com.geoway.design.biz.entity.SysUser;
import com.geoway.design.biz.entity.SysUserOrganization;
import com.geoway.design.biz.mapper.SysOrganizationMapper;
import com.geoway.design.biz.service.dev.IOrganizationService;
import com.geoway.design.biz.service.dev.IUserOrganizationService;
import com.geoway.design.biz.service.dev.IUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author daidongdong
 * @description 针对表【sys_organization】的数据库操作Service实现
 * @createDate 2021-12-02 17:19:38
 */
@Service
public class LocalOrganizationServiceImpl extends ServiceImpl<SysOrganizationMapper,SysOrganization> implements IOrganizationService {

    @Autowired
    SysOrganizationMapper sysOrganizationMapper;

    @Autowired
    IUserOrganizationService sysUserOrganizationService;

    @Autowired
    @Qualifier("localUserServiceImpl")
    IUserService sysUserService;


    @Override
    public void saveOrUp(SysOrganization org) throws Exception {

        //获取级别信息
        if (StrUtil.isBlank(org.getPid())) {
            org.setPid("0");
        }
        SysOrganization pOrg = sysOrganizationMapper.selectById(org.getPid());
        boolean isAdd=false;
        if (StrUtil.isBlank(org.getId())) {
            isAdd=true;
            Integer maxOrder = sysOrganizationMapper.queryMaxOrderByPid(org.getPid());
            int order = maxOrder == null ? 1 : maxOrder + 1;
            org.setOrder(order);

            int level = pOrg == null ? 1 : pOrg.getLevel() + 1;
            org.setLevel(level);

            String parents = "";
            if (pOrg != null) {
                parents = StrUtil.isNotBlank(pOrg.getParents()) ? pOrg.getId() : pOrg.getParents() + "/" + pOrg.getId();
            }
            org.setParents(parents);
        } else {
            isAdd=false;
            SysOrganization oldOrg = this.sysOrganizationMapper.selectById(org.getId());
            org.setParents(oldOrg.getParents());
            org.setLevel(oldOrg.getLevel());
            if (!org.getName().equals(oldOrg.getName())) {
                this.buildChildALlName(org);
            }
        }

        this.buildAllName(org);
        if(isAdd) {
            this.sysOrganizationMapper.insert(org);
        }else{
            this.sysOrganizationMapper.updateById(org);
        }

    }

    @Override
    public void batchSave(List<SysOrganization> orgs) throws Exception {
        for (SysOrganization org : orgs) {
            this.saveOrUp(org);
        }
    }

    @Override
    public List<SysOrganization> queryTree(String filterParam) throws Exception {
        List<SysOrganization> list = this.queryList(filterParam);
        bindUsersToOrg(list);
        return constructTree(list);
    }


    @Override
    public List<SysOrganization> queryTreeNoUser(String filterParam) throws Exception {
        List<SysOrganization> list = this.queryList(filterParam);
        return constructTree(list);
    }

    @Override
    public List<SysOrganization> queryList(String filterParam) throws Exception {
        if (filterParam == null) {
            filterParam = "";
        }

        MyBatisQueryMapperUtils<SysOrganization> qmu = new MyBatisQueryMapperUtils<>();
        QueryWrapper queryWrapper = qmu.queryMapper(filterParam, SysOrganization.class);
        List<SysOrganization> list = this.sysOrganizationMapper.selectList(queryWrapper);
        return list;
    }

    @Override
    public List<SysOrganization> queryListById(String id) throws Exception {
        if (id == null) {
            throw new Exception("机构ID不能为空!");
        }
        List<SysOrganization> list = sysOrganizationMapper.getOrgListById(id);
        bindUsersToOrg(list);
        return constructTree(list);
    }

    @Override
    public List<SysOrganization> queryListByName(String name) throws Exception {
        List<SysOrganization> list = sysOrganizationMapper.getOrgListByName(name);
        bindUsersToOrg(list);
        return constructTree(list);
    }

    @Override
    public List<SysOrganization> queryAllParentOrg(String id) {
        if (id == null) {
            throw new ServiceException("机构ID不能为空!");
        }
        List<SysOrganization> result = sysOrganizationMapper.getAllParentOrg(id);
        Collections.reverse(result);
        return result;
    }

    @Override
    public IPage<SysOrganization> queryPage(String filterParam, int page, int size) throws Exception {
        if (filterParam == null) {
            filterParam = "";
        }

        MyBatisQueryMapperUtils<SysOrganization> qmu = new MyBatisQueryMapperUtils<>();
        QueryWrapper queryWrapper = qmu.queryMapper(filterParam, SysOrganization.class);
        Page<SysOrganization> pages = new Page<>(page, size);
        Page<SysOrganization> results = this.sysOrganizationMapper.selectPage(pages, queryWrapper);

        return results;
    }

    @Override
    public SysOrganization findOneById(String id, String filterParam) throws Exception {
        SysOrganization org = this.sysOrganizationMapper.selectById(id);
        if (filterParam == null) {
            filterParam = "";
        }

        filterParam += ";pid_EQ_" + id;
        MyBatisQueryMapperUtils<SysOrganization> qmu = new MyBatisQueryMapperUtils<>();
        QueryWrapper queryWrapper = qmu.queryMapper(filterParam, SysOrganization.class);

        List<SysOrganization> orgs = this.sysOrganizationMapper.selectList(queryWrapper);
        if (org != null) {
            org.setChildren(orgs);
        }

        return org;
    }

    @Override
    public SysOrganization findOneByName(String name, String filterParam) throws Exception{
        LambdaQueryWrapper<SysOrganization> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(SysOrganization::getName, name);
        SysOrganization org = this.sysOrganizationMapper.selectOne(wrapper);
        if(org==null){
            return org;
        }
        if (filterParam == null) {
            filterParam = "";
        }

        filterParam += ";pid_EQ_" + org.getId();
        MyBatisQueryMapperUtils<SysOrganization> qmu = new MyBatisQueryMapperUtils<>();
        QueryWrapper queryWrapper = qmu.queryMapper(filterParam, SysOrganization.class);

        List<SysOrganization> orgs = this.sysOrganizationMapper.selectList(queryWrapper);
        if (org != null) {
            org.setChildren(orgs);
        }

        return org;
    }

    @Override
    public void batchDelete(String ids) throws Exception {
        if (StrUtil.isBlank(ids)) {
            return;
        }
        List<String> idList = Arrays.asList(ids.split(","));
        for (String id : idList) {
            this.deleteById(id);
        }
    }


    @Override
    public void bindUsers(String orgId, String userIds) {
        sysUserOrganizationService.removeByOrgUserId(orgId,null);
        if (StrUtil.isBlank(userIds)) {
            return;
        }
        String[] userIdArr = userIds.split(",");
        List<SysUserOrganization> userOrgs = new ArrayList<>();
        for (String userId : userIdArr) {
            SysUserOrganization userOrg = new SysUserOrganization();
            userOrg.setUserid(userId);
            userOrg.setOrgid(orgId);
            userOrgs.add(userOrg);
        }
        sysUserOrganizationService.saveBatch(userOrgs);
    }

    @Override
    public void bindOneUser(String orgId, String userId) {


        sysUserOrganizationService.removeByOrgUserId(orgId,userId);
        SysUserOrganization userOrg = new SysUserOrganization();
        userOrg.setUserid(userId);
        userOrg.setOrgid(orgId);

        sysUserOrganizationService.saveOne(userOrg);

    }

    @Override
    public void removeBindUser(String orgId, String userIds) {
        String[] userIdArr = userIds.split(",");
        for (String userId : userIdArr) {
            sysUserOrganizationService.removeByOrgUserId(orgId,userId);
        }
    }

    @Override
    public List<SysUser> queryRelUsers(String orgId) {
        List<SysUserOrganization> organizations=sysUserOrganizationService.queryByOrgId(orgId);
        return sysUserService.query(organizations.stream().map(f->f.getUserid()).collect(Collectors.toList()));
    }

    @Override
    public List<SysUser> queryNonOrgUsers() {
        try {
            List<SysUser> allUsers = sysUserService.queryList("");
            List<SysUserOrganization> organizations=sysUserOrganizationService.queryList("");
            List<String> orgUserIds=organizations.stream().map(f->f.getUserid()).collect(Collectors.toList());
            return allUsers.stream().filter(f->!orgUserIds.contains(f.getId())).collect(Collectors.toList());
        }catch (Exception exception){
            throw new RuntimeException(exception);
        }
    }

    @Override
    public int queryCount(String filterParam) throws Exception {
        int count = 0;
        if (filterParam == null) {
            filterParam = "";
        }

        MyBatisQueryMapperUtils<SysOrganization> qmu = new MyBatisQueryMapperUtils<>();
        QueryWrapper queryWrapper = qmu.queryMapper(filterParam, SysOrganization.class);
        count = this.sysOrganizationMapper.selectCount(queryWrapper);
        return count;
    }

    @Override
    public List<SysOrganization> queryUserOrgs(String userId) {
        return this.sysOrganizationMapper.queryUserOrgs(userId);
    }

    @Override
    public void bindUsersToOrg(List<SysOrganization> list) {
        List<SysUserOrganization> userOrgs = sysUserOrganizationService.queryList("");

        List<String> userIds=userOrgs.stream().map(f->f.getUserid()).collect(Collectors.toList());
        List<SysUser> users = sysUserService.query(userIds);

        if (users.isEmpty() || userOrgs.isEmpty()) {
            return;
        }
        Map<String,SysUser> sysUserMap=new HashMap<>();
        users.forEach(f->sysUserMap.put(f.getId(),f));
        Map<String, List<SysUser>> mapUser = new HashMap<>();

        Map<String,  List<SysUserOrganization>> groupedUserOrgs = userOrgs.stream().filter(f->StrUtil.isNotBlank(f.getOrgid()))
                .collect(Collectors.groupingBy(SysUserOrganization::getOrgid));
        for (Map.Entry<String,  List<SysUserOrganization>> kv: groupedUserOrgs.entrySet()) {
            String orgId = kv.getKey();
            List<SysUser> subUsers = new ArrayList<>();
            mapUser.put(orgId,subUsers);
            kv.getValue().forEach(f->{
                if(sysUserMap.containsKey(f.getUserid())) {
                    subUsers.add(sysUserMap.get(f.getUserid()));
                }
            });
        }

        for (SysOrganization org : list) {
            String orgId = org.getId();
            if (mapUser.containsKey(orgId)) {
                List<SysUser> subUsers = mapUser.get(orgId);
                org.setBindUsers(subUsers);
            }
        }
    }

    private void deleteById(String id) throws Exception {
        int iCount = sysUserOrganizationService.queryByOrgId(id).size();
        if (iCount > 0) {
            throw new Exception("该组织下已关联用户，不能删除!");
        }
        this.sysOrganizationMapper.deleteById(id);
    }

    /**
     * 递归查询树
     *
     * @param list
     * @return
     */
    @Override
    public List<SysOrganization> constructTree(List<SysOrganization> list) {
      int minLevel = 99;
      Map<String, List<SysOrganization>> mapParam = new HashMap<>();
       for (SysOrganization org : list) {

          String key = org.getPid();
          if (!StringUtils.isEmpty(key) && !key.equals("-1")) {
                if (mapParam.containsKey(key)) {
                   mapParam.get(key).add(org);
              } else {
                List<SysOrganization> childList = new ArrayList<>();
                   childList.add(org);
                 mapParam.put(key, childList);
             }
           }

           if (org.getLevel() < minLevel) {
               minLevel = org.getLevel();
           }
       }
        List<SysOrganization> results = new ArrayList<>();
        for (SysOrganization org : list) {
           String key = org.getId();
           if (StringUtils.isNotBlank(key) && mapParam.containsKey(key)) {
              org.setChildren(mapParam.get(key));
               org.setChildCount(doCount(org));
            }
            if (org.getLevel() == minLevel) {
               results.add(org);
          }
       }

        return results;
    }

    private static Integer doCount(SysOrganization org) {
        List<SysOrganization> orgList = org.getChildren();
        Integer sum = 0;
        if (orgList == null || orgList.size() == 0) {
            if (org.getChildCount() > 0) {
                return org.getChildCount();
            } else {
                return sum;
            }
        }
        if (orgList != null && orgList.size() > 0) {
            for (SysOrganization child : orgList) {
                //递归
                Integer numCount = orgList.size() + org.getChildCount();
                //求和计算
                sum += numCount;
                doCount(child);
            }
        }
        return sum;
    }

    private void buildChildALlName(SysOrganization org) {
        LambdaQueryWrapper<SysOrganization> wrapper = Wrappers.lambdaQuery();
        wrapper.like(SysOrganization::getParents, org.getId());

        List<SysOrganization> orgs = this.sysOrganizationMapper.selectList(wrapper);
        for (SysOrganization subOrg : orgs) {
            this.buildAllName(subOrg);
        }

    }

    private void buildAllName(SysOrganization org) {
        String parents = org.getParents();
        if (StrUtil.isBlank(parents)) {
            org.setAllname(org.getName());
        } else {
            List<String> pOrgIds = Arrays.asList(parents.split("/"));
            List<SysOrganization> pOrgs = this.sysOrganizationMapper.selectBatchIds(pOrgIds);
            String allName = "";
            for (SysOrganization pOrg : pOrgs) {
                allName += pOrg.getName() + "/";
            }
            allName += org.getName();
            org.setAllname(allName);
        }

    }
}




