package com.geoway.design.biz.service.dev.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.geoway.design.base.support.spring.SpringUtils;
import com.geoway.design.biz.dto.CaptchaTextDTO;
import com.geoway.design.biz.service.dev.ILoginService;
import com.geoway.design.biz.service.dev.IUnityLoginService;
import com.geoway.design.biz.service.login.ICaptchaService;
import com.geoway.sso.client.constant.Oauth2Constant;
import com.geoway.sso.client.rpc.Result;
import com.geoway.sso.client.rpc.SsoUser;
import com.geoway.sso.client.util.HttpServletUtil;
import com.geoway.sso.client.util.SessionUtils;
import com.google.code.kaptcha.impl.DefaultKaptcha;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.awt.image.BufferedImage;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * @author 连世忠
 * @ClassName UnityLoginServiceImpl
 * @Description 统一登入
 * @date 2024/10/17 10:28
 * @Version 1.0
 */
@Service
@Slf4j
public final class UnityLoginServiceImpl implements IUnityLoginService {
    @Autowired
    private ICaptchaService commonService;
    @Autowired
    private DefaultKaptcha defaultKaptcha;
    @Value("${sso.setting.login-type:1}")
    private String sysLoginTypes;

    @Override
    public CaptchaTextDTO createCaptchaText(HttpServletRequest req) {
        String capText = defaultKaptcha.createText();
        UUID uid = UUID.randomUUID();
        String uuid = uid.toString();
        commonService.storeCaptcha(uuid, capText);
        CaptchaTextDTO captchaDTO = new CaptchaTextDTO();
        captchaDTO.setText(capText);
        captchaDTO.setUid(uuid);
        return captchaDTO;
    }

    @Override
    public void createCaptchaPng(HttpServletRequest req, HttpServletResponse resp, String uid) {
        try {
            String w = req.getParameter("w");
            String h = req.getParameter("h");
            BufferedImage buffImg = commonService.createImageCaptcha(uid, w, h, req, resp);
            ServletOutputStream out = resp.getOutputStream();
            ImageIO.write(buffImg, "png", (OutputStream) out);
            out.close();
        } catch (Exception exception) {
            throw new RuntimeException(exception);
        }
    }

    @Override
    public String getUserSys() {
        return "Unity";
    }

    @Override
    public Result<SsoUser> login(String username, String password, String tel, String uuid, String code, HttpServletRequest req) {
        List<ILoginService> loginServices = getLoginServices();
        Result<SsoUser> result = null;
        for (ILoginService loginService : loginServices) {
            try {
                result = loginService.login(username, password, tel, uuid, code, req);
                if (result != null && result.getCode() == Oauth2Constant.RESETPASSWORD_CODE) {
                    return result;
                }
                if (result != null && result.isSuccess()) {
                    result.getData().setLoginTime(DateUtil.now());
                    result.getData().setMachineName("Administrator");
                    result.getData().setIp(HttpServletUtil.getRequestClientIp(HttpServletUtil.getRequest()));
                    return result;
                } else {
                    log.error(String.format("登录方式[%s]报错：", loginService.getUserSys()), result == null ? "" : result.getMessage());
                    // 当前若为最后一个实现类，抛出异常
                    if (loginServices.indexOf(loginService) == loginServices.size() - 1) {
                        throw new RuntimeException(result.getMessage());
                    }
                }
            } catch (Exception exception) {
                log.error(String.format("登录方式[%s]报错：", loginService.getUserSys()), exception);
                throw new RuntimeException("登录失败：" + exception.getMessage());
            }
        }
        return result;
    }

    @Override
    public Result<SsoUser> login(String username, String password, String uuid, String code, HttpServletRequest req) {
        return login(username, password, "", uuid, code, req);
    }

    @Override
    public Result<SsoUser> login(String username, String password) {

        Result<SsoUser> result = login(username, password, null, null, null, null);
        return result;
    }

    @Override
    public boolean loginOut(HttpServletRequest request, String access_token) {
        List<ILoginService> loginServices = getLoginServices();
        for (ILoginService loginService : loginServices) {
            loginService.loginOut(request, access_token);
        }
        SessionUtils.invalidate(request);
        return true;
    }

    private List<ILoginService> getLoginServices() {
        List<String> loginTypes = new ArrayList<>();
        if (StrUtil.isBlank(sysLoginTypes)) {
            loginTypes.add("1");
        } else {
            loginTypes = Arrays.stream(sysLoginTypes.split(",")).collect(Collectors.toList());
        }

        List<ILoginService> wholeBeans = SpringUtils.getBeansOfType(ILoginService.class);
        List<ILoginService> result = new ArrayList<>();
        loginTypes.forEach(f -> {
            List<ILoginService> collect = wholeBeans.stream().filter(g -> g.getUserSys().equals(f)).collect(Collectors.toList());
            if (collect.size() == 0) {
                return;
            }
            if (collect.size() > 1) {
                for (ILoginService service : collect) {
                    // 如果出现多个
                    Class<? extends ILoginService> aClass = service.getClass();
                    Primary annotation = aClass.getAnnotation(Primary.class);
                    if (annotation != null) {
                        result.add(service);
                        return;
                    }
                }
            }
            result.add(collect.get(0));
        });
        return result;
    }

    @Override
    public  boolean checkIsResetPassword(String userId){

        List<ILoginService> loginServices = getLoginServices();
        boolean isReset = false;
        for (ILoginService loginService : loginServices) {
            isReset =  loginService.checkIsResetPassword(userId);
            if(isReset){
                break;
            }
        }
        return  isReset;
    }

}
