package com.geoway.design.biz.service.dev.impl;

import cn.hutool.core.util.StrUtil;
import com.geoway.design.base.config.SsoConfigProp;
import com.geoway.design.base.support.ServletUtils;
import com.geoway.design.biz.service.dev.IUserService;
import com.geoway.design.biz.service.oauth2.IOauth2Service;
import com.geoway.sso.client.constant.SsoConstant;
import com.geoway.sso.client.rpc.SsoUser;
import com.geoway.sso.client.util.CommonLoginUserUtil;
import com.geoway.sso.client.util.CookieUtils;
import com.geoway.sso.client.util.HttpServletUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @description: 用户体系工厂类
 * @author: wujing
 * @date: 2024-11-08 15:31
 */
@Component
public class UserSysFactory {

    /**
     * 默认体系
     */
    private IUserService defaultService;

    @Resource
    private Map<String, IUserService> userServiceMap;

    @Autowired
    private SsoConfigProp ssoConfig;
    
    @Autowired
    private IOauth2Service oauth2Service;

    @PostConstruct
    private void setService(){
        this.defaultService = getDefaultService();
    }





    public IUserService getService(){
        String userSys=getUserSysFromRequest();
        if(StrUtil.isNotBlank(userSys)){
            return getByType(userSys);
        }
        SsoUser ssoUser = CommonLoginUserUtil.getUser();
        if(ssoUser!=null&&StrUtil.isNotBlank(ssoUser.getSource())){
            return getByType(ssoUser.getSource());
        }
        return defaultService;
    }
    /**
     * 获取当前的用户体系
     *
     * @return
     */

    private  String getUserSysFromRequest(){
        // 获取当前http请求
        HttpServletRequest request = HttpServletUtil.getRequest();
        // 1. 优先从param参数中获取
        String parameterUserSys = request.getParameter(SsoConstant.PARAM_USER_SYS);
        // 不为空则直接返回param
        if (StrUtil.isNotBlank(parameterUserSys)) {
            return parameterUserSys;
        }

        // 2. 从header中获取
        String headerUserSys = request.getHeader(SsoConstant.HEADER_USER_SYS);
        if (StrUtil.isNotBlank(headerUserSys)) {
            return headerUserSys;
        }

        // 3. 从cookie中获取
        String cookieUserSys = CookieUtils.getCookie(request, SsoConstant.COOKIE_USER_SYS);
        if (StrUtil.isNotBlank(cookieUserSys)) {
            return cookieUserSys;
        }
        return null;
    }


    /**
     * 获取默认
     * @return
     */
    private IUserService getDefaultService() {
        // 如果没有传类型，则使用系统第一个配置
        String type = ssoConfig.getLoginType();
        String[] split = type.split(",");
        type = split[0];
        return getByType(type);
    }

    /**
     * 根据类型获取用户体系实现类
     * @param type
     * @return
     */
    private IUserService getByType(String type){
        List<String> syncType = Arrays.asList(ssoConfig.getSyncType().split(","));
        for (Map.Entry<String, IUserService> entry : userServiceMap.entrySet()) {
            // 当前实现类用户体系
            String userSys = entry.getValue().getUserSys();
            // 当前实现类是信息同步方式
            Boolean isSync = entry.getValue().isSync();
            // 判断当前用户体系是否设置的同步方式
            boolean flag = syncType.stream().anyMatch(a -> a.equals(userSys));
            // 对应用户体系、对接方式
            if(type.equals(userSys) && (isSync == flag)) {
                return entry.getValue();
            }
        }
        return defaultService;
    }



}
