package com.geoway.design.biz.service.login.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.geoway.design.base.base.dto.ResponseDataBase;
import com.geoway.design.base.constants.CommonConstants;
import com.geoway.design.base.exception.ResetPasswordException;
import com.geoway.design.biz.config.ProjectConfig;
import com.geoway.design.biz.entity.SysUserRole;
import com.geoway.design.biz.mapper.SysUserSecurityMapper;
import com.geoway.design.biz.service.login.ISysLoginService;
import com.geoway.design.biz.service.oauth2.IOauth2Service;
import com.geoway.design.biz.service.sys.ISysUserRoleService;
import com.geoway.design.biz.util.PasswordUtil;
import com.geoway.sso.client.constant.Oauth2Constant;
import com.geoway.sso.client.constant.SsoConstant;
import com.geoway.sso.client.enums.GrantTypeEnum;
import com.geoway.sso.client.rpc.Result;
import com.geoway.sso.client.rpc.RpcAccessToken;
import com.geoway.sso.client.rpc.SsoUser;
import com.geoway.sso.client.session.redis.RedisSessionMappingStorage;
import com.geoway.sso.client.util.CommonLoginUserUtil;
import com.geoway.sso.client.util.Oauth2Utils;
import com.geoway.sso.client.util.SessionUtils;
import com.geoway.sso.client.util.SsoCheckLogin;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

/**
 * @author AA
 */
@Service
@Transactional
@Slf4j
public class SysLoginServiceImpl implements ISysLoginService {
    @Autowired
    private ProjectConfig projectConfig;

    @Autowired
    private RedisSessionMappingStorage sessionMappingStorage;

    @Autowired
    private ISysUserRoleService sysUserRoleService;
    @Autowired
    private IOauth2Service ssoOauth2Service;
    @Autowired
    private SysUserSecurityMapper sysUserSecurityMapper;
    @Autowired
    private PasswordUtil passwordUtil;

    @Value("${password.forceChangeInitPWD:true}")
    private Boolean forceChangeInitPWD;

    @Value("${sso.setting.singeUser:false}")
    private Boolean singeUser;


    @Override
    public ResponseDataBase checkLogin(String username, String password, String uuid, String code, HttpServletRequest request) {
        ResponseDataBase response = new ResponseDataBase();
        try {

            RpcAccessToken rpcAccessToken = ssoOauth2Service.getAccessToken(GrantTypeEnum.PASSWORD_CAPTCHA.getValue(), projectConfig.getSsoAppId(), projectConfig.getSsoAppSecret(), "", username, password, "", uuid, code);

            SessionUtils.setAccessToken(request, rpcAccessToken);
            SsoUser user = rpcAccessToken.getUser();
            response.put("token", rpcAccessToken.getAccessToken());
            response.put("userId", user.getId());
            response.put("username", user.getLoginName());
            response.put("alisname", user.getUserName());
            response.put("role", sysUserRoleService.listObjs(Wrappers.lambdaQuery(SysUserRole.class)
                    .eq(SysUserRole::getUserid, user.getId()).select(SysUserRole::getRoleid), a -> (String) a));
            response.put("status", "OK");
            response.put("validateType", "sso");
            response.put("passwordPolicy", user.getPasswordPolicy());
            response.put("userCatalog", user.getUserCatalog());

            response.put("regionCode", user.getRegionCode());
            response.put("regionName", user.getRegionName());

        } catch (Exception e) {
            if(e instanceof ResetPasswordException){
                return ResponseDataBase.resetPassword(e.getMessage());
            }
            return ResponseDataBase.error(e.getMessage());
        }

        return response;
    }

    @Override
    public ResponseDataBase checkLogin(String username, String password, HttpServletRequest request) {
        ResponseDataBase response = new ResponseDataBase();
        try {
            RpcAccessToken rpcAccessToken = ssoOauth2Service.getAccessToken(GrantTypeEnum.PASSWORD.getValue(), projectConfig.getSsoAppId(), projectConfig.getSsoAppSecret(), "", username, password, "", "", "");
            SessionUtils.setAccessToken(request, rpcAccessToken);
            SsoUser user = rpcAccessToken.getUser();
            response.put("token", rpcAccessToken.getAccessToken());
            response.put("refreshToken", rpcAccessToken.getRefreshToken());
            response.put("userId", user.getId());
            response.put("username", user.getLoginName());
            response.put("alisname", user.getUserName());
            response.put("status", "OK");
            response.put("validateType", "sso");
            response.put("passwordPolicy", user.getPasswordPolicy());

            response.put("regionCode", user.getRegionCode());
            response.put("regionName", user.getRegionName());
            response.put("userCatalog", user.getUserCatalog());

        }catch (Exception e) {
            if(e instanceof ResetPasswordException){
                return ResponseDataBase.resetPassword(e.getMessage());
            }
            return ResponseDataBase.error(e.getMessage());
        }
        return response;
    }

    @Override
    public String redirectToSsoLogin(String redirectUri) {
        String loginUrl = "";
        try {
            loginUrl = new StringBuilder().append(projectConfig.getSsoServerUrl()).append(SsoConstant.LOGIN_URL).append("?")
                    .append(Oauth2Constant.APP_ID).append("=").append(projectConfig.getSsoAppId()).append("&")
                    .append(SsoConstant.REDIRECT_URI).append("=")
                    .append(URLEncoder.encode(redirectUri, "utf-8")).toString();

        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return loginUrl;
    }

    @Override
    public String getSsoLogoutUrl(String redirectUri, HttpServletRequest request) {
        SsoCheckLogin ssoCheckLogin = new SsoCheckLogin(projectConfig.getSsoServerUrl());
        return ssoCheckLogin.userLogut(redirectUri, request);
    }


    @Override
    public Result<RpcAccessToken> getAccessTokenByCode(String code, HttpServletRequest request) {
        Result<RpcAccessToken> result = Oauth2Utils.getAccessToken(projectConfig.getSsoServerUrl()
                , projectConfig.getSsoAppId(), projectConfig.getSsoAppSecret(), code);
        setAccessTokenInSession(result.getData(), request);
        return result;
    }

    @Override
    public ResponseDataBase restLogout(HttpServletRequest request, String access_token) {
        SessionUtils.invalidate(request);
        Result<RpcAccessToken> result = Oauth2Utils.revokeAccessToken(projectConfig.getSsoServerUrl(), access_token);
        if (result.isSuccess()) {
            return ResponseDataBase.ok();
        }
        return ResponseDataBase.error("退出失败");
    }

    @Override
    public boolean checkIsResetPassword(String accessToken) {
//        String token = CommonLoginUserUtil.getToken();
        return this.ssoOauth2Service.checkIsResetPassword(accessToken);
    }

    @Override
    public ResponseDataBase validUserOnline(String username, String password) {
        ResponseDataBase response = new ResponseDataBase();
        boolean isLogin =  ssoOauth2Service.validUserOnline(username,password);
        response.put("singeUser",singeUser);
        response.put("isLogin",isLogin);
        return response;
    }

    private boolean setAccessTokenInSession(RpcAccessToken rpcAccessToken, HttpServletRequest request) {
        if (rpcAccessToken == null) {
            return false;
        }
        // 记录accessToken到本地session
        SessionUtils.setAccessToken(request, rpcAccessToken);
        // 记录本地session和accessToken映射
        recordSession(request, rpcAccessToken.getAccessToken(), rpcAccessToken.getUser().getLoginName());
        return true;
    }

    private void recordSession(final HttpServletRequest request, String accessToken, String userName) {
        final HttpSession session = request.getSession();
        sessionMappingStorage.removeBySessionById(session.getId());
        sessionMappingStorage.addSessionById(accessToken, session);
        sessionMappingStorage.removeTokenByUserName(userName);
        sessionMappingStorage.addTokenByUserName(accessToken, userName);
    }


}


