package com.geoway.design.biz.service.oauth2.impl;

import cn.hutool.core.util.StrUtil;
import com.geoway.design.base.exception.ResetPasswordException;
import com.geoway.design.biz.service.dev.IUnityLoginService;
import com.geoway.design.biz.service.dev.impl.LoginSysFactory;
import com.geoway.design.biz.service.login.ISsoAppService;
import com.geoway.design.biz.service.oauth2.IOauth2Service;
import com.geoway.design.biz.service.sys.SysUserLimitService;
import com.geoway.design.biz.util.GuestUtil;
import com.geoway.sso.client.constant.Oauth2Constant;
import com.geoway.sso.client.enums.GrantTypeEnum;
import com.geoway.sso.client.rpc.Result;
import com.geoway.sso.client.rpc.RpcAccessToken;
import com.geoway.sso.client.rpc.SsoUser;
import com.geoway.sso.client.util.CookieUtils;
import com.geoway.sso.client.util.SessionUtils;
import com.geoway.sso.server.common.AccessTokenContent;
import com.geoway.sso.server.common.CodeContent;
import com.geoway.sso.server.common.RefreshTokenContent;
import com.geoway.sso.server.constant.AppConstant;
import com.geoway.sso.server.session.AccessTokenManager;
import com.geoway.sso.server.session.CodeManager;
import com.geoway.sso.server.session.RefreshTokenManager;
import com.geoway.sso.server.session.TicketGrantingTicketManager;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author 连世忠
 * @ClassName SsoOauth2ServiceImpl
 * @Description TODO
 * @date 2024/3/12 17:24
 * @Version 1.0
 */
@Service
@Slf4j
public class Oauth2ServiceImpl implements IOauth2Service {
    @Autowired
    private ISsoAppService ssoAppService;

    @Autowired
    private IUnityLoginService userService;

    @Autowired
    private LoginSysFactory loginSysFactory;

    @Autowired
    private CodeManager codeManager;

    @Autowired
    private AccessTokenManager accessTokenManager;

    @Autowired
    private RefreshTokenManager refreshTokenManager;

    @Autowired
    private TicketGrantingTicketManager ticketGrantingTicketManager;

    @Autowired
    private SysUserLimitService sysUserLimitService;


    @Value("${sso.setting.singeUser:false}")
    private Boolean singeUser;


    @Override
    public RpcAccessToken getAccessToken(String grantType, String appId, String appSecret, String code, String username, String password,String tel,String sessionId,String catpcha) {
        Result<Void> result = validateParam(grantType, code, username, password,sessionId,catpcha);
        if (!result.isSuccess()) {
            throw new RuntimeException(result.getMessage());
        }

        // 校验应用
        Result<Void> appResult = ssoAppService.validate(appId, appSecret);
        if (!appResult.isSuccess()) {
            throw new RuntimeException(appResult.getMessage());
        }

        // 校验授权
        Result<AccessTokenContent> accessTokenResult = validateAuth(grantType, code, username, password,tel, appId,sessionId,catpcha);
//        if(accessTokenResult.getCode() == Oauth2Constant.RESETPASSWORD_CODE){
//            throw  new ResetPasswordException(accessTokenResult.getMessage());
//        }

        //如果需要重置密码
        if (accessTokenResult.getCode() == Oauth2Constant.RESETPASSWORD_CODE) {
            throw new ResetPasswordException(accessTokenResult.getMessage());
        }

        if (!accessTokenResult.isSuccess()) {
            throw new RuntimeException(accessTokenResult.getMessage());
        }

        //检验是否加入黑名单
        String userId = accessTokenResult.getData().getUser().getId();
        boolean isValid = sysUserLimitService.isValid(appId,userId);
        if (!isValid) {
            throw new RuntimeException("该用户被限制访问，请联系管理员");
        }

        if(singeUser){
            String oldToken = accessTokenManager.getAccessTokenByUserId(userId);
            if(StrUtil.isNotBlank(oldToken)){
                accessTokenManager.removeOnlineUser(userId);
                SessionUtils.invalidate(oldToken);
            }
        }

        // 生成RpcAccessToken返回
        return genereateRpcAccessToken(accessTokenResult.getData(), null);
    }

    @Override
    public RpcAccessToken refreshToken(String appId, String refreshToken) {
        if(GuestUtil.validateGuestToken(refreshToken)){
            return GuestUtil.getGuestRpcAccessToken(refreshToken);
        }
        if (!ssoAppService.exists(appId)) {
            throw new RuntimeException("非法应用");
        }
        RefreshTokenContent refreshTokenContent = refreshTokenManager.validate(refreshToken);
        if (refreshTokenContent == null) {
            throw new RuntimeException("refreshToken有误或已过期");
        }
        AccessTokenContent accessTokenContent = refreshTokenContent.getAccessTokenContent();
        if (!appId.equals(accessTokenContent.getAppId())) {
            throw new RuntimeException("非法应用");
        }
        SsoUser user = ticketGrantingTicketManager.getAndRefresh(accessTokenContent.getCodeContent().getTgt());
        if (user == null) {
            throw new RuntimeException("服务端session已过期");
        }
        return genereateRpcAccessToken(accessTokenContent, refreshTokenContent.getAccessToken());
    }

    @Override
    public RpcAccessToken queryAccessToken(String token) {
        if(GuestUtil.validateGuestToken(token)){
            return GuestUtil.getGuestRpcAccessToken(token);
        }
        AccessTokenContent tokenContent = accessTokenManager.get(token);
        if (tokenContent != null) {
            RpcAccessToken rpcAccessToken = new RpcAccessToken(token, accessTokenManager.getExpiresIn(), tokenContent.getRefreshToken(), tokenContent.getUser());
            return rpcAccessToken;
        } else {
            throw new RuntimeException("token有误或已过期");
        }
    }

    @Override
    public void revokeAccessToken(String token) {
        ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
        HttpServletResponse response = servletRequestAttributes.getResponse();
        AccessTokenContent tokenContent = accessTokenManager.get(token);
        if (tokenContent != null) {
            String tgt = tokenContent.getCodeContent().getTgt();
            // 删除登录凭证
            ticketGrantingTicketManager.remove(tgt);
            // 删除凭证Cookie
            CookieUtils.removeCookie(AppConstant.TGC, "/", response);
            // 删除所有tgt对应的调用凭证，并通知客户端登出注销本地session
            accessTokenManager.remove(tgt);
            accessTokenManager.removeToken(token);
            if(singeUser){
                accessTokenManager.removeOnlineUser(tokenContent.getUser().getId());
            }


        } else {
            throw new RuntimeException("token有误或已过期");
        }

    }

    @Override
    public String queryCurrentUserId(HttpServletRequest request) {

        SsoUser ssoUser= this.queryCurrentSsoUser(request);
        if(ssoUser != null){
            return ssoUser.getId();
        }
        return null;
    }

    @Override
    public SsoUser queryCurrentSsoUser(HttpServletRequest request) {
        String accessToken = request.getParameter("token");
        if(StrUtil.isBlank(accessToken)){
            accessToken  = request.getHeader("access_token");
        }
        if(StrUtil.isBlank(accessToken)){
            accessToken  = request.getHeader("access-token");
        }
        AccessTokenContent accessTokenContent = accessTokenManager.get(accessToken);
        if(accessTokenContent != null){
            return  accessTokenContent.getUser();
        }
        return null;
    }

    @Override
    public RpcAccessToken getAccessToken(String appId, String appSecret, String username, String password, String accessToken) {
        // 校验应用
        Result<Void> appResult = ssoAppService.validate(appId, appSecret);
        if (!appResult.isSuccess()) {
            throw new RuntimeException(appResult.getMessage());
        }

        // app通过此方式由客户端代理转发http请求到服务端获取accessToken
        Result<SsoUser> loginResult = userService.login(username,password);
        if (!loginResult.isSuccess()) {
            throw new RuntimeException(loginResult.getMessage());
        }
        SsoUser user = loginResult.getData();
        user.setToken(accessToken);
        String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
        CodeContent codeContent = new CodeContent(tgt, false, null);
        AccessTokenContent accessTokenContent = new AccessTokenContent(codeContent, user, appId,null);

        //检验是否加入黑名单
        String userId = accessTokenContent.getUser().getId();
        boolean isValid = sysUserLimitService.isValid(appId,userId);
        if (!isValid) {
            throw new RuntimeException("该用户被限制访问，请联系管理员");
        }
        if(singeUser){
            String oldToken = accessTokenManager.getAccessTokenByUserId(userId);
            if(StrUtil.isNotBlank(oldToken)){
                accessTokenManager.removeOnlineUser(userId);
                SessionUtils.invalidate(oldToken);
            }
        }

        // 生成RpcAccessToken返回
        return genereateRpcAccessToken2(accessTokenContent);
    }

    @Override
    public boolean checkIsResetPassword(String token) {
        RpcAccessToken rpcToken = this.queryAccessToken(token);
        if(rpcToken != null && rpcToken.getUser() != null){
            return  loginSysFactory.getService().checkIsResetPassword(rpcToken.getUser().getUserid());
        }
        return false;
    }

    @Override
    public boolean validUserOnline(String username, String password) {

        boolean flag = false;
        Result<SsoUser> loginResult = userService.login(username,password);
        if(loginResult==null){
            return false;
        }
        if (!loginResult.isSuccess()||loginResult.getData()==null) {
            return false;
        }
        String token =   accessTokenManager.getAccessTokenByUserId(loginResult.getData().getUserid());
        flag = StrUtil.isNotBlank(token);
        return flag;
    }

    private Result<Void> validateParam(String grantType, String code, String username, String password,String sessionId,String catpcha) {
        if (GrantTypeEnum.AUTHORIZATION_CODE.getValue().equals(grantType)) {
            if (StringUtils.isEmpty(code)) {
                return Result.createError("code不能为空");
            }
        } else if (GrantTypeEnum.PASSWORD.getValue().equals(grantType)) {
            if (StringUtils.isEmpty(username) || StringUtils.isEmpty(password)) {
                return Result.createError("username和password不能为空");
            }
        } else if(GrantTypeEnum.PASSWORD_CAPTCHA.getValue().equals(grantType)){
            if (StringUtils.isEmpty(username) || StringUtils.isEmpty(password)) {
                return Result.createError("username和password不能为空");
            }
            if (StringUtils.isEmpty(sessionId) || StringUtils.isEmpty(catpcha)) {
                return Result.createError("验证码不能为空");
            }
        }else {
            return Result.createError("授权方式不支持");
        }
        return Result.success();
    }

    private Result<AccessTokenContent> validateAuth(String grantType, String code, String username, String password,String tel,
                                                    String appId,String sessionId,String catpcha) {
        AccessTokenContent authDto = null;
        if (GrantTypeEnum.AUTHORIZATION_CODE.getValue().equals(grantType)) {
            CodeContent codeContent = codeManager.getAndRemove(code);
            if (codeContent == null) {
                return Result.createError("code有误或已过期");
            }
            SsoUser user = ticketGrantingTicketManager.getAndRefresh(codeContent.getTgt());
            if (user == null) {
                return Result.createError("服务端session已过期");
            }
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        } else if (GrantTypeEnum.PASSWORD.getValue().equals(grantType)) {
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
            HttpServletRequest request = servletRequestAttributes.getRequest();
            // app通过此方式由客户端代理转发http请求到服务端获取accessToken
            Result<SsoUser> loginResult = userService.login(username,password,tel,sessionId,catpcha,request);

            //如果需要重置密码
           if (loginResult.getCode() == Oauth2Constant.RESETPASSWORD_CODE) {
               return Result.createNeedResetPassword(loginResult.getMessage());
            }

            if (!loginResult.isSuccess()) {
                return Result.createError(loginResult.getMessage());
            }

            SsoUser user = loginResult.getData();
            String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
            CodeContent codeContent = new CodeContent(tgt, false, null);
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        }else if(GrantTypeEnum.PASSWORD_CAPTCHA.getValue().equals(grantType)){
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.currentRequestAttributes();
            HttpServletRequest request = servletRequestAttributes.getRequest();
            // app通过此方式由客户端代理转发http请求到服务端获取accessToken
            Result<SsoUser> loginResult = userService.login(username,password,tel,sessionId,catpcha,request);

            //如果需要重置密码
            if (loginResult.getCode() == Oauth2Constant.RESETPASSWORD_CODE) {
                return Result.createNeedResetPassword(loginResult.getMessage());
            }

            if (!loginResult.isSuccess()) {
                return Result.createError(loginResult.getMessage());
            }
            SsoUser user = loginResult.getData();
            String tgt = ticketGrantingTicketManager.generate(loginResult.getData());
            CodeContent codeContent = new CodeContent(tgt, false, null);
            authDto = new AccessTokenContent(codeContent, user, appId,null);
        }
        return Result.createSuccess(authDto);
    }


    private RpcAccessToken genereateRpcAccessToken(AccessTokenContent accessTokenContent, String accessToken) {
        String newAccessToken = accessToken;
        if (newAccessToken == null || !accessTokenManager.refresh(newAccessToken)) {
            newAccessToken = accessTokenManager.generate(accessTokenContent);
        }

        String refreshToken = refreshTokenManager.generate(accessTokenContent, newAccessToken);

        AccessTokenContent tokenContent = accessTokenManager.get(newAccessToken);
        tokenContent.setRefreshToken(refreshToken);
        accessTokenManager.create(newAccessToken,tokenContent);

        return new RpcAccessToken(newAccessToken, accessTokenManager.getExpiresIn(), refreshToken,
                accessTokenContent.getUser());
    }

    private RpcAccessToken genereateRpcAccessToken2(AccessTokenContent accessTokenContent) {
        String newAccessToken = accessTokenManager.generate(accessTokenContent);
        String refreshToken = refreshTokenManager.generate(accessTokenContent, newAccessToken);

        AccessTokenContent tokenContent = accessTokenManager.get(newAccessToken);
        tokenContent.setRefreshToken(refreshToken);
        accessTokenManager.create(newAccessToken,tokenContent);

        return new RpcAccessToken(newAccessToken, accessTokenManager.getExpiresIn(), refreshToken,
                accessTokenContent.getUser());
    }
}
