package com.geoway.jckj.base.support;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.enmus.ExcelType;
import cn.afterturn.easypoi.excel.imports.ExcelImportService;
import com.geoway.jckj.base.enums.ExcelTypeEnum;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.List;
import java.util.NoSuchElementException;

/**
 * @author ALMJ
 * @desc 基于EasyPOI 的 Excel操作工具类封装
 * @date 2022/10/24 12:34
 */
public class ExcelUtil {
    /**
     * 导出Excel
     *
     * @param list           需要导出的数据列表
     * @param title          标题
     * @param sheetName      sheet名称
     * @param pojoClass      实体类
     * @param fileName       导出文件名称
     * @param isCreateHeader 是否创建表头
     * @param response
     * @throws IOException
     */
    public static <T> void exportExcel(List<T> list, String title, String sheetName, Class<T> pojoClass, String fileName, boolean isCreateHeader, HttpServletResponse response) throws IOException {
        ExportParams exportParams = new ExportParams(title, sheetName, ExcelType.XSSF);
        exportParams.setCreateHeadRows(isCreateHeader);
        defaultExport(list, pojoClass, fileName, response, exportParams);
    }

    /**
     * 导出Excel
     *
     * @param list      需要导出的数据列表
     * @param title     标题
     * @param sheetName sheet名称
     * @param pojoClass 实体类
     * @param fileName  导出文件名称
     * @param response
     * @throws IOException
     */
    public static <T> void exportExcel(List<T> list, String title, String sheetName, Class<T> pojoClass, String fileName, HttpServletResponse response) throws IOException {
        defaultExport(list, pojoClass, fileName, response, new ExportParams(title, sheetName, ExcelType.XSSF));
    }

    /**
     * 导出Excel
     *
     * @param list         需要导出的数据列表
     * @param pojoClass    实体类
     * @param fileName     导出文件名称
     * @param exportParams 导出参数
     * @param response     返回请求
     * @throws IOException
     */
    public static <T> void exportExcel(List<T> list, Class<T> pojoClass, String fileName, ExportParams exportParams, HttpServletResponse response) throws IOException {
        defaultExport(list, pojoClass, fileName, response, exportParams);
    }

    /**
     * 默认导出方法
     *
     * @param list         需要导出的数据列表
     * @param pojoClass    实体类
     * @param fileName     导出文件名称
     * @param response     返回请求
     * @param exportParams 导出参数
     * @throws IOException
     */
    private static <T> void defaultExport(List<T> list, Class<T> pojoClass, String fileName, HttpServletResponse response, ExportParams exportParams) throws IOException {
        Workbook workbook = ExcelExportUtil.exportExcel(exportParams, pojoClass, list);
        downLoadExcel(fileName, response, workbook);
    }

    /**
     * 下载Excel文件
     *
     * @param fileName 文件名
     * @param response
     * @param workbook Excel数据
     * @throws IOException
     */
    private static void downLoadExcel(String fileName, HttpServletResponse response, Workbook workbook) throws IOException {
        try {
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            response.setCharacterEncoding("UTF-8");
            response.setHeader("content-Type", "application/vnd.ms-excel");
            response.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(fileName + "." + ExcelTypeEnum.XLSX.getValue(), "UTF-8"));
            workbook.write(response.getOutputStream());
        } catch (Exception e) {
            throw new IOException(e.getMessage());
        }
    }

    /**
     * 导入Excel
     *
     * @param filePath   文件路径
     * @param titleRows  标题行数
     * @param headerRows 表头行数
     * @param pojoClass  实体类
     * @throws IOException
     */
    public static <T> List<T> importExcel(String filePath, Integer titleRows, Integer headerRows, Class<T> pojoClass) throws IOException {
        if (StringUtils.isBlank(filePath)) {
            return null;
        }
        ImportParams params = new ImportParams();
        params.setTitleRows(titleRows);
        params.setHeadRows(headerRows);
        params.setNeedSave(true);
        params.setSaveUrl("/excel/");
        try {
            return ExcelImportUtil.importExcel(new File(filePath), pojoClass, params);
        } catch (NoSuchElementException e) {
            throw new IOException("模板不能为空");
        } catch (Exception e) {
            throw new IOException(e.getMessage());
        }
    }

    /**
     * 导入Excel
     *
     * @param file      文件
     * @param pojoClass 实体类
     * @throws IOException
     */
    public static <T> List<T> importExcel(MultipartFile file, Class<T> pojoClass) throws IOException {
        return importExcel(file, 1, 1, pojoClass);
    }

    /**
     * 导入Excel
     *
     * @param file       文件
     * @param titleRows  标题行数
     * @param headerRows 表头行数
     * @param pojoClass  实体类
     * @throws IOException
     */
    public static <T> List<T> importExcel(MultipartFile file, Integer titleRows, Integer headerRows, Class<T> pojoClass) throws IOException {
        return importExcel(file, titleRows, headerRows, false, pojoClass);
    }

    /**
     * 导入Excel
     *
     * @param file       文件
     * @param titleRows  标题行数
     * @param headerRows 表头行数
     * @param needVerfiy 是否需要校验
     * @param pojoClass  实体类
     * @throws IOException
     */
    public static <T> List<T> importExcel(MultipartFile file, Integer titleRows, Integer headerRows, boolean needVerfiy, Class<T> pojoClass) throws IOException {
        if (file == null) {
            return null;
        }
        try {
            return importExcel(file.getInputStream(), titleRows, headerRows, needVerfiy, pojoClass);
        } catch (Exception e) {
            throw new IOException(e.getMessage());
        }
    }

    /**
     * 导入Excel
     *
     * @param inputStream 输入流
     * @param titleRows   标题行数
     * @param headerRows  表头行数
     * @param needVerfiy  是否需要校验
     * @param pojoClass   实体类
     * @throws IOException
     */
    public static <T> List<T> importExcel(InputStream inputStream, Integer titleRows, Integer headerRows, boolean needVerfiy, Class<T> pojoClass) throws IOException {
        if (inputStream == null) {
            return null;
        }
        ImportParams params = new ImportParams();
        params.setTitleRows(titleRows);
        params.setHeadRows(headerRows);
        params.setSaveUrl("/excel/");
        params.setNeedSave(true);
        params.setNeedVerify(needVerfiy);
        try {
//            return ExcelImportUtil.importExcel(inputStream, pojoClass, params);
            return new ExcelImportService().importExcelByIs(inputStream, pojoClass, params, false).getList();
        } catch (NoSuchElementException e) {
            throw new IOException("excel文件不能为空");
        } catch (Exception e) {
            throw new IOException(e.getMessage());
        }
    }




}
