package com.geoway.sso.client.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.geoway.sso.client.constant.Oauth2Constant;
import com.geoway.sso.client.enums.GrantTypeEnum;
import com.geoway.sso.client.rpc.Result;
import com.geoway.sso.client.rpc.RpcAccessToken;
import com.geoway.sso.client.service.IOauth2Utils;
import com.geoway.sso.client.util.HttpServletUtil;
import com.geoway.sso.client.util.HttpUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.Map;

/**
 * @author 连世忠
 * @ClassName IOauth2UtilsImpl
 * @Description TODO
 * @date 2025/5/20 15:31
 * @Version 1.0
 */
@Component
@ConditionalOnExpression("#{!T(com.geoway.sso.client.constant.SystemConstant).IS_SERVER}")
//@ConditionalOnExpression("#{'${sso.server.url:}'.isEmpty()}")
public class ClientOauth2UtilsImpl implements IOauth2Utils {
    private static final Logger logger = LoggerFactory.getLogger(ClientOauth2UtilsImpl.class);

    /**
     * 获取accessToken（密码模式，app通过此方式由客户端代理转发http请求到服务端获取accessToken）
     *
     * @param serverUrl
     * @param appId
     * @param appSecret
     * @param username
     * @param password
     * @return
     */
    @Override
    public  Result<RpcAccessToken> getAccessToken(String serverUrl, String appId, String appSecret, String username,
                                                        String password) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.GRANT_TYPE, GrantTypeEnum.PASSWORD.getValue());
        paramMap.put(Oauth2Constant.APP_ID, appId);
        paramMap.put(Oauth2Constant.APP_SECRET, appSecret);
        paramMap.put(Oauth2Constant.USERNAME, username);
        paramMap.put(Oauth2Constant.PASSWORD, password);
        return getHttpAccessToken(serverUrl + Oauth2Constant.ACCESS_TOKEN_URL, paramMap);
    }

    @Override
    public Result<RpcAccessToken> getAccessToken(String serverUrl, String appId, String appSecret, String username, String password, String uuid, String code) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.GRANT_TYPE, GrantTypeEnum.PASSWORD_CAPTCHA.getValue());
        paramMap.put(Oauth2Constant.APP_ID, appId);
        paramMap.put(Oauth2Constant.APP_SECRET, appSecret);
        paramMap.put(Oauth2Constant.USERNAME, username);
        paramMap.put(Oauth2Constant.PASSWORD, password);
        paramMap.put(Oauth2Constant.SESSIONID, uuid);
        paramMap.put(Oauth2Constant.CATPCHA, code);
        return getHttpAccessToken(serverUrl + Oauth2Constant.ACCESS_TOKEN_URL, paramMap);
    }

    @Override
    public Result<RpcAccessToken> getAccessTokenBySms(String serverUrl, String appId, String appSecret, String tel, String uuid, String code) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.GRANT_TYPE, GrantTypeEnum.TEL_CAPTCHA.getValue());
        paramMap.put(Oauth2Constant.APP_ID, appId);
        paramMap.put(Oauth2Constant.APP_SECRET, appSecret);
        paramMap.put(Oauth2Constant.TEL, tel);
        paramMap.put(Oauth2Constant.SESSIONID, uuid);
        paramMap.put(Oauth2Constant.CATPCHA, code);
        HttpServletRequest request = HttpServletUtil.getRequest();
        if(request != null){
            paramMap.put(Oauth2Constant.SOURCE, request.getParameter(Oauth2Constant.SOURCE));
        }
        return getHttpAccessToken(serverUrl + Oauth2Constant.ACCESS_TOKEN_URL, paramMap);
    }

    @Override
    public Result<RpcAccessToken> getAccessTokenByOne(String serverUrl, String appId, String appSecret, String processId, String token, String authCode) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.GRANT_TYPE, GrantTypeEnum.ONE.getValue());
        paramMap.put(Oauth2Constant.APP_ID, appId);
        paramMap.put(Oauth2Constant.APP_SECRET, appSecret);
        paramMap.put(Oauth2Constant.ONEAUTHCODE,authCode );
        paramMap.put(Oauth2Constant.ONEPROCESSID, processId);
        paramMap.put(Oauth2Constant.ONETOKEN, token);
        return getHttpAccessToken(serverUrl + Oauth2Constant.ACCESS_TOKEN_URL, paramMap);
    }

    @Override
    public Result<RpcAccessToken> getAccessTokenByCa(String serverUrl, String appId, String appSecret, String caUserCert, String caOauthName) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.GRANT_TYPE, GrantTypeEnum.CA.getValue());
        paramMap.put(Oauth2Constant.APP_ID, appId);
        paramMap.put(Oauth2Constant.APP_SECRET, appSecret);
        paramMap.put(Oauth2Constant.CAUSERCERT, caUserCert);
        paramMap.put(Oauth2Constant.CAOAUTHNAME, caOauthName);
        return getHttpAccessToken(serverUrl + Oauth2Constant.ACCESS_TOKEN_URL, paramMap);
    }

    /**
     * 获取accessToken（授权码模式）
     *
     * @param serverUrl
     * @param appId
     * @param appSecret
     * @param code
     * @return
     */
    @Override
    public  Result<RpcAccessToken> getAccessTokenByCode(String serverUrl, String appId, String appSecret, String code) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.GRANT_TYPE, GrantTypeEnum.AUTHORIZATION_CODE.getValue());
        paramMap.put(Oauth2Constant.APP_ID, appId);
        paramMap.put(Oauth2Constant.APP_SECRET, appSecret);
        paramMap.put(Oauth2Constant.AUTH_CODE, code);
        return getHttpAccessToken(serverUrl + Oauth2Constant.ACCESS_TOKEN_URL, paramMap);
    }

    /**
     * 刷新accessToken
     *
     * @param serverUrl    单点登录服务地址
     * @param appId        应用授权id
     * @param refreshToken 刷新新令牌
     * @return
     */
    @Override
    public  Result<RpcAccessToken> refreshToken(String serverUrl, String appId, String refreshToken) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.APP_ID, appId);
        paramMap.put(Oauth2Constant.REFRESH_TOKEN, refreshToken);
        return getHttpAccessToken(serverUrl + Oauth2Constant.REFRESH_TOKEN_URL, paramMap);
    }

    /**
     * 查询令牌对应的用户登录信息
     *
     * @param serverUrl   单点登录服务地址
     * @param accessToken 当前用户访问令牌
     * @return
     */
    @Override
    public  Result<RpcAccessToken> queryAccessToken(String serverUrl, String accessToken) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.TOKEN, accessToken);
        return getHttpAccessToken(serverUrl + Oauth2Constant.QUERY_TOKEN_CONTENT, paramMap);
    }

    /**
     * 撤销访问凭证(单点退出)
     *
     * @param serverUrl   单点登录服务地址
     * @param accessToken 当前用户令牌
     * @return
     */
    @Override
    public  Result<RpcAccessToken> revokeAccessToken(String serverUrl, String accessToken) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.TOKEN, accessToken);
        return getHttpAccessToken(serverUrl + Oauth2Constant.REVOKE_ACCESS_TOKEN, paramMap);
    }

    @Override
    public Result<String> getAccessCode(String serverUrl, String appId, String appSecret, String accessToken,Integer allowedUseCount) {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put(Oauth2Constant.TOKEN, accessToken);
        paramMap.put(Oauth2Constant.APP_ID, appId);
        paramMap.put(Oauth2Constant.APP_SECRET, appSecret);
        if(allowedUseCount!=null) {
            paramMap.put("allowedUseCount", allowedUseCount.toString());
        }
        return HttpUtils.getHttp(serverUrl + Oauth2Constant.QUERY_ACCESS_CODE,paramMap,null,String.class);
    }

    @Override
    public Result<String> getAccessCode(String serverUrl, String appId, String appSecret, String accessToken) {
     return getAccessCode(serverUrl,appId,appSecret,accessToken,1);
    }

    public static Result<RpcAccessToken> getHttpAccessToken(String url, Map<String, String> paramMap) {
        String jsonStr = HttpUtils.get(url, paramMap);
        if (jsonStr == null || jsonStr.isEmpty()) {
            logger.error("getHttpAccessToken exception, return null. url:{}", url);
            return null;
        }
        return JSONObject.parseObject(jsonStr, new TypeReference<Result<RpcAccessToken>>() {
        });
    }
}
