package com.geoway.sso.client.util;

import cn.hutool.cache.CacheUtil;
import cn.hutool.cache.impl.FIFOCache;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.geoway.sso.client.constant.SsoConstant;
import com.geoway.sso.client.session.SessionAccessToken;
import com.geoway.sso.client.rpc.RpcAccessToken;
import com.geoway.sso.client.rpc.SsoUser;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;
import org.springframework.web.context.support.ServletRequestHandledEvent;

import javax.servlet.http.HttpServletRequest;

/**
 * @author ALMJ
 * @desc Session工具
 */
@Component
public class SessionUtils implements ApplicationListener<ServletRequestHandledEvent> {

    private static  FIFOCache<String,String> fifoCache;
    private static  ThreadLocal<SessionAccessToken> localAccessSession=new ThreadLocal();
    @Autowired
    public SessionUtils(){
        fifoCache=CacheUtil.newFIFOCache(10000000,4*3600*1000);
    }

    public static SessionAccessToken getAccessToken(HttpServletRequest request, String access_token) {
        if(localAccessSession.get()!=null){
            return localAccessSession.get();
        }
        String atcStr=fifoCache.get(SsoConstant.SESSION_ACCESS_TOKEN+ "_" +access_token);
        if (StringUtils.isEmpty(atcStr)) {
            return null;
        }
        SessionAccessToken sessionAccessToken= JSONObject.parseObject(atcStr, SessionAccessToken.class);
        localAccessSession.set(sessionAccessToken);
        return sessionAccessToken;
    }

    public static SessionAccessToken getAccessTokenByUserId(HttpServletRequest request, String userId) {
        if(localAccessSession.get()!=null){
            return localAccessSession.get();
        }
        //从redis中取
        String atcStr = fifoCache.get(SsoConstant.SESSION_ACCESS_USERID + "_" + userId);
        if (StringUtils.isEmpty(atcStr)) {
            return null;
        }
        SessionAccessToken sessionAccessToken= JSONObject.parseObject(atcStr, SessionAccessToken.class);
        localAccessSession.set(sessionAccessToken);
        return sessionAccessToken;
    }

    public static SsoUser getUser(HttpServletRequest request, String access_token) {
        SessionAccessToken sessionAccessToken = getAccessToken(request, access_token);
        if (ObjectUtil.isNotNull(sessionAccessToken)) {
            return sessionAccessToken.getUser();
        } else {
            return null;
        }
    }

    public static void setAccessToken(HttpServletRequest request, RpcAccessToken rpcAccessToken) {
        SessionAccessToken sessionAccessToken = null;
        if (rpcAccessToken != null) {
            sessionAccessToken = createSessionAccessToken(rpcAccessToken);
        }

        localAccessSession.set(sessionAccessToken);
        if (sessionAccessToken != null) {
            //redis里面也存一份
            fifoCache.put(SsoConstant.SESSION_ACCESS_TOKEN + "_" + sessionAccessToken.getAccessToken(), JSONObject.toJSONString(sessionAccessToken),rpcAccessToken.getExpiresIn()*1000*2);
            fifoCache.put(SsoConstant.SESSION_ACCESS_USERID + "_" + sessionAccessToken.getUser().getId(), JSONObject.toJSONString(sessionAccessToken),rpcAccessToken.getExpiresIn()*1000*2);
        }
    }

    private static SessionAccessToken createSessionAccessToken(RpcAccessToken accessToken) {
        long expirationTime = System.currentTimeMillis() + accessToken.getExpiresIn()/2 * 1000;
        return new SessionAccessToken(accessToken.getAccessToken(), accessToken.getExpiresIn(),
                accessToken.getRefreshToken(), accessToken.getUser(), expirationTime);
    }

    public static void invalidate(HttpServletRequest request,String access_token) {
        fifoCache.remove(SsoConstant.SESSION_ACCESS_TOKEN + "_" + access_token);
        request.getSession().invalidate();
    }

    public static void invalidate(HttpServletRequest request) {
        request.getSession().invalidate();
    }

    @Override
    public void onApplicationEvent(ServletRequestHandledEvent event) {
        localAccessSession.set(null);
    }
}


