package com.geoway.sso.client.session.redis;

import com.geoway.sso.client.session.SessionMappingStorage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.session.SessionRepository;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpSession;
import java.util.concurrent.TimeUnit;

/**
 * @author ALMJ
 * @title 基于redis借鉴CAS
 * @desc 将单点登录后台集群部署的时候seesion共享到redis来统一管理
 */
@Component
@ConditionalOnProperty(value = "sso.session.manager", havingValue = "redis")
public final class RedisSessionMappingStorage implements SessionMappingStorage {
    private static final String SESSION_TOKEN_KEY = "ns-design-session_token_key_";
    private static final String TOKEN_SESSION_KEY = "ns-design-token_session_key_";
    private static final String TOKEN_USERNAME_KEY = "ns-design-token-user-name-key_";

    @Autowired
    private SessionRepository<?> sessionRepository;

    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    @Value("${sso.timeout}")
    private Long timeout;

    @Override
    public synchronized void addSessionById(final String accessToken, final HttpSession session) {
        redisTemplate.opsForValue().set(SESSION_TOKEN_KEY + session.getId(), accessToken, timeout, TimeUnit.SECONDS);
        redisTemplate.opsForValue().set(TOKEN_SESSION_KEY + accessToken, session.getId(), timeout,TimeUnit.SECONDS);
    }

    @Override
    public void addTokenByUserName(String accessToken, String userName) {
        redisTemplate.opsForValue().set(TOKEN_USERNAME_KEY + userName, accessToken, timeout,TimeUnit.SECONDS);
    }

    @Override
    public void removeTokenByUserName(String userName) {
        final String accessToken = redisTemplate.opsForValue().get(TOKEN_USERNAME_KEY + userName);
        if (accessToken != null) {
            redisTemplate.delete(TOKEN_USERNAME_KEY + userName);
        }
    }

    @Override
    public String getTokenByUserName(String userName) {
        return redisTemplate.opsForValue().get(TOKEN_USERNAME_KEY + userName);
    }

    @Override
    public synchronized void removeBySessionById(final String sessionId) {
        final String accessToken = redisTemplate.opsForValue().get(SESSION_TOKEN_KEY + sessionId);
        if (accessToken != null) {
            redisTemplate.delete(TOKEN_SESSION_KEY + accessToken);
            redisTemplate.delete(SESSION_TOKEN_KEY + sessionId);
            sessionRepository.deleteById(sessionId);
        }
    }

    @Override
    public synchronized HttpSession removeSessionByMappingId(final String accessToken) {
        final String sessionId = redisTemplate.opsForValue().get(TOKEN_SESSION_KEY + accessToken);
        if (sessionId != null) {
            removeBySessionById(sessionId);
        }
        return null;
    }

//    @Override
//    public void addUserInfo(String loginName, String token, String userid) {
//        redisTemplate.opsForHash().put("NSS-"+loginName, AppConstant.TOKEN,token);
//        redisTemplate.opsForHash().put("NSS-"+loginName,AppConstant.USERID,userid);
//        redisTemplate.expire("NSS-"+loginName,timeout, TimeUnit.SECONDS);
//    }
//
//    @Override
//    public Map<String, String> getUserInfo(String loginName) {
//        Map map = redisTemplate.boundHashOps("NSS-"+loginName).entries();
//        if( map.size() == 0 ){
//            return null;
//        }
//        return map;
//    }

}
