package com.geoway.sso.server.session.redis;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.geoway.sso.client.rpc.SsoUser;
import com.geoway.sso.server.common.AccessTokenContent;
import com.geoway.sso.server.config.PropertyConfig;
import com.geoway.sso.server.constant.AppConstant;
import com.geoway.sso.server.session.AccessTokenManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author ALMJ
 * @desc 分布式调用凭证管理
 */
@Component
@ConditionalOnProperty(name = "sso.session.manager", havingValue = "redis")
public class RedisAccessTokenManager implements AccessTokenManager {


    @Autowired
    private PropertyConfig propertyConfig;
    @Value("${sso.userstat: false}")
    private boolean userstat;

//    @Autowired
//    private StringRedisTemplate redisTemplate;

    @Autowired
    private RedisTemplate<String, String> redisTemplate;



    @Override
    public void create(String accessToken, AccessTokenContent accessTokenContent) {

        accessTokenContent.setToken(accessToken);

        //存储Token信息
        String tokenKey=getTokenKey(accessToken);
        redisTemplate.opsForValue().set(tokenKey, JSON.toJSONString(accessTokenContent), getExpiresIn(),
                TimeUnit.SECONDS);


        //存储tgt信息，暂时不清楚要干嘛
        String tgtKey = getTgtKey(accessTokenContent.getCodeContent().getTgt());
        redisTemplate.opsForSet().add(tgtKey, accessToken);
        redisTemplate.expire(tgtKey,getExpiresIn(),TimeUnit.SECONDS);

       //存储在线用户信息
        String userKey=getUserIdKey(accessTokenContent);
        redisTemplate.opsForValue().set(userKey, JSON.toJSONString(accessTokenContent),getExpiresIn(),TimeUnit.SECONDS);

        String userSession=getUserSessionKey(accessTokenContent);
        redisTemplate.opsForSet().add(userSession, accessToken);
        redisTemplate.expire(userSession,getExpiresIn(),TimeUnit.SECONDS);
    }

    @Override
    public AccessTokenContent get(String accessToken) {
        String atcStr = redisTemplate.opsForValue().get(getTokenKey(accessToken));
        if (StringUtils.isEmpty(atcStr)) {
            return null;
        }
        return JSONObject.parseObject(atcStr, AccessTokenContent.class);
    }

    @Override
    public boolean refresh(String accessToken) {
        String atcStr=redisTemplate.opsForValue().get(getTokenKey(accessToken));
        if (StrUtil.isBlank(atcStr)) {
            return false;
        }
        redisTemplate.expire(getTokenKey(accessToken),getExpiresIn(), TimeUnit.SECONDS);
        //刷新在线用户信息
        AccessTokenContent accessTokenContent=JSONObject.parseObject(atcStr, AccessTokenContent.class);
        String userIdKey=getUserIdKey(accessTokenContent);
        String tgtKey=getTgtKey(accessTokenContent.getCodeContent().getTgt());
        String userSessionKey=getUserSessionKey(accessTokenContent);
        if(redisTemplate.hasKey(userIdKey)){
            redisTemplate.expire(userIdKey, getExpiresIn(), TimeUnit.SECONDS);
        }
        if(redisTemplate.hasKey(tgtKey)){
            redisTemplate.expire(tgtKey, getExpiresIn(), TimeUnit.SECONDS);
        }
        if(redisTemplate.hasKey(userSessionKey)){
            redisTemplate.expire(userSessionKey, getExpiresIn(), TimeUnit.SECONDS);
        }
        return true;
    }

    @Override
    public void remove(String tgt) {
        Set<String> accessTokenSet = redisTemplate.opsForSet().members(getTgtKey(tgt));
        if (CollectionUtils.isEmpty(accessTokenSet)) {
            return;
        }
        redisTemplate.delete(getTgtKey(tgt));
        String userKey="";
        String userSessionKey="";
        for(String accessToken:accessTokenSet){
            String atcStr = redisTemplate.opsForValue().get(AppConstant.REDIS_TOKEN_PREFIX + accessToken);
            redisTemplate.delete(AppConstant.REDIS_TOKEN_PREFIX + accessToken);
            if (StringUtils.isEmpty(atcStr)) {
                continue;
            }
            AccessTokenContent accessTokenContent = JSONObject.parseObject(atcStr, AccessTokenContent.class);
            userSessionKey=getUserSessionKey(accessTokenContent);
            redisTemplate.opsForSet().remove(userSessionKey,accessToken);
            userKey=getUserIdKey(accessTokenContent);
            if(!accessTokenContent.getCodeContent().isSendLogoutRequest()) {
                continue;
            }
            sendLogoutRequest(accessTokenContent.getCodeContent().getRedirectUri(), accessToken);
        }
        //如果已经没有存活的会话信息了，那么这个用户就是已经离线了
        if(StrUtil.isNotBlank(userKey)&&StrUtil.isNotBlank(redisTemplate.opsForValue().get(userKey))
                &&redisTemplate.opsForSet().size(userSessionKey)==0){
            //移除在线用户信息
            redisTemplate.delete(userKey);
        }
    }

    @Override
    public List<SsoUser> getOnlineUsers() {
        List<SsoUser> result = new ArrayList<>();
        Set<String> onlineUserKeys=redisTemplate.keys(AppConstant.REDIS_TOKEN_USERONLINE + "*");
        if(onlineUserKeys==null||onlineUserKeys.size()==0){
            return result;
        }
        List<String>  userStrs=redisTemplate.opsForValue().multiGet(onlineUserKeys);
        for(String userStr:userStrs){
            if(StrUtil.isBlank(userStr)){
                continue;
            }
            AccessTokenContent accessTokenContent = JSONObject.parseObject(userStr, AccessTokenContent.class);
            result.add(accessTokenContent.getUser());
        }
        return result;
    }

    @Override
    public String getAccessTokenByUserId(String userId) {
        AccessTokenContent tokenContent =getTokenContentByUserId(userId);
        return tokenContent==null?null:tokenContent.getToken();
    }

    @Override
    public void removeOnlineUser(String userId) {
        AccessTokenContent accessTokenContent = this.getTokenContentByUserId(userId);
        if(accessTokenContent==null){
            return;
        }
        String tgt = accessTokenContent.getCodeContent().getTgt();
        remove(tgt);
    }

    @Override
    public void removeToken(String token) {
        if(StrUtil.isNotBlank(token)){
            //移除Token信息
            String tokenKey=getTokenKey(token);
            redisTemplate.delete(tokenKey);
        }

    }

    private AccessTokenContent getTokenContentByUserId(String userId){
        AccessTokenContent accessTokenContent = null;
        String userKey= AppConstant.REDIS_TOKEN_USERONLINE + userId;
        if( redisTemplate.hasKey(userKey)) {
            String findJson = redisTemplate.opsForValue().get(userKey);
            accessTokenContent = JSONObject.parseObject(findJson,AccessTokenContent.class);
        }
        return accessTokenContent;
    }

    private String getTgtKey(String tgt) {
        return AppConstant.REDIS_TOKEN_PREFIX+tgt + "_access_token";
    }
    private String getTokenKey(String accessToken) {
        return AppConstant.REDIS_TOKEN_PREFIX+accessToken;
    }

    private String getUserIdKey(AccessTokenContent accessTokenContent){
        return AppConstant.REDIS_TOKEN_USERONLINE+accessTokenContent.getUser().getId();
    }



    private String getUserSessionKey(AccessTokenContent accessTokenContent){
        return AppConstant.REDIS_TOKEN_USERSESSION+accessTokenContent.getUser().getId();
    }
    /**
     * accessToken时效为登录session时效的1/2
     */
    @Override
    public int getExpiresIn() {
        return propertyConfig.getSsoTimeout() ;
    }
}
