package com.geoway.sso.server.session;

import com.geoway.sso.client.rpc.SsoUser;
import com.geoway.sso.client.util.CookieUtils;
import com.geoway.sso.server.constant.AppConstant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author ALMJ
 * @desc 服务端凭证管理
 */
@Component
public class SessionManager {

    @Autowired
    private AccessTokenManager accessTokenManager;

    @Autowired
    private TicketGrantingTicketManager ticketGrantingTicketManager;

    public String setUser(SsoUser user, HttpServletRequest request, HttpServletResponse response) {
        String tgt = getCookieTgt(request);
        // cookie中没有
        if (StringUtils.isEmpty(tgt)) {
            tgt = ticketGrantingTicketManager.generate(user);
            // TGT存cookie，和Cas登录保存cookie中名称一致为：TGC
            CookieUtils.addCookie(AppConstant.TGC, tgt, "/", request, response);
        } else if (ticketGrantingTicketManager.getAndRefresh(tgt) == null) {
            ticketGrantingTicketManager.create(tgt, user);
        } else {
            ticketGrantingTicketManager.set(tgt, user);
        }
        return tgt;
    }

    public String setUser(SsoUser user, String path, HttpServletRequest request, HttpServletResponse response) {
        String tgt = getCookieTgt(request);
        // cookie中没有
        if (ObjectUtils.isEmpty(tgt)) {
            tgt = ticketGrantingTicketManager.generate(user);
            // TGT存cookie，和Cas登录保存cookie中名称一致为：TGC
            CookieUtils.addCookie(AppConstant.TGC, tgt, path, request, response);
            CookieUtils.addCookie(AppConstant.LOGINNAME, user.getLoginName(), path, request, response);
        } else if (ticketGrantingTicketManager.getAndRefresh(tgt) == null) {
            ticketGrantingTicketManager.create(tgt, user);
        } else {
            ticketGrantingTicketManager.set(tgt, user);
        }
        return tgt;
    }

    public SsoUser getUser(HttpServletRequest request) {
        String tgt = getCookieTgt(request);
        if (ObjectUtils.isEmpty(tgt)) {
            return null;
        }
        return ticketGrantingTicketManager.getAndRefresh(tgt);
    }

    public void invalidate(HttpServletRequest request, HttpServletResponse response) {
        String tgt = getCookieTgt(request);
        if (StringUtils.isEmpty(tgt)) {
            return;
        }
        // 删除登录凭证
        ticketGrantingTicketManager.remove(tgt);
        // 删除凭证Cookie
        CookieUtils.removeCookie(AppConstant.TGC, "/", response);
        // 删除所有tgt对应的调用凭证，并通知客户端登出注销本地session
        accessTokenManager.remove(tgt);
    }

    public String getTgt(HttpServletRequest request) {
        String tgt = getCookieTgt(request);
        if (StringUtils.isEmpty(tgt) || ticketGrantingTicketManager.getAndRefresh(tgt) == null) {
            return null;
        } else {
            return tgt;
        }
    }

    private String getCookieTgt(HttpServletRequest request) {
        return CookieUtils.getCookie(request, AppConstant.TGC);
    }
}