package com.geoway.sso.server.session.redis;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.geoway.sso.server.common.AccessTokenContent;
import com.geoway.sso.server.constant.AppConstant;
import com.geoway.sso.server.session.AccessTokenManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author ALMJ
 * @desc 分布式调用凭证管理
 */
@Component
@ConditionalOnProperty(name = "sso.session.manager", havingValue = "redis")
public class RedisAccessTokenManager implements AccessTokenManager {

    @Value("${sso.timeout}")
    private int timeout;

    @Value("${sso.userstat: false}")
    private boolean userstat;

//    @Autowired
//    private StringRedisTemplate redisTemplate;

    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    final String KEY_ONLINE_USER = "UserMonitor";

    @Override
    public void create(String accessToken, AccessTokenContent accessTokenContent) {
        redisTemplate.opsForValue().set(AppConstant.REDIS_TOKEN_PREFIX + accessToken, JSON.toJSONString(accessTokenContent), getExpiresIn(),
                TimeUnit.SECONDS);
        String addKey = AppConstant.REDIS_TOKEN_PREFIX + getKey(accessTokenContent.getCodeContent().getTgt());
        redisTemplate.opsForSet().add(addKey, accessToken);
        redisTemplate.expire(addKey,getExpiresIn(),TimeUnit.SECONDS);

        //存储在线用户token,但影响性能
        if(userstat){
            Number size = redisTemplate.opsForList().size(KEY_ONLINE_USER);
            if(size != null && size.longValue()>0){
                List<String> list = redisTemplate.opsForList().range(KEY_ONLINE_USER,0,-1);
                if(!list.contains(accessToken)){
                    redisTemplate.opsForList().leftPush(KEY_ONLINE_USER,accessToken);
                }

                //移除无效的token
                for(String existToken: list){
                    if(!redisTemplate.hasKey(AppConstant.REDIS_TOKEN_PREFIX + existToken)) {
                        redisTemplate.opsForList().remove(KEY_ONLINE_USER,0, existToken);
                    }
                }
            }else{
                List<String> value = new ArrayList<String>();
                value.add(accessToken);
                redisTemplate.opsForList().rightPushAll(KEY_ONLINE_USER,value);
            }
        }
    }

    @Override
    public AccessTokenContent get(String accessToken) {
        String atcStr = redisTemplate.opsForValue().get(AppConstant.REDIS_TOKEN_PREFIX + accessToken);
        if (StringUtils.isEmpty(atcStr)) {
            return null;
        }
        return JSONObject.parseObject(atcStr, AccessTokenContent.class);
    }

    @Override
    public boolean refresh(String accessToken) {
        if (redisTemplate.opsForValue().get(AppConstant.REDIS_TOKEN_PREFIX + accessToken) == null) {
            return false;
        }
        redisTemplate.expire(AppConstant.REDIS_TOKEN_PREFIX + accessToken, timeout, TimeUnit.SECONDS);
        return true;
    }

    @Override
    public void remove(String tgt) {
        Set<String> accessTokenSet = redisTemplate.opsForSet().members(AppConstant.REDIS_TOKEN_PREFIX + getKey(tgt));
        if (CollectionUtils.isEmpty(accessTokenSet)) {
            return;
        }
        redisTemplate.delete(AppConstant.REDIS_TOKEN_PREFIX + getKey(tgt));

        accessTokenSet.forEach(accessToken -> {
            String atcStr = redisTemplate.opsForValue().get(AppConstant.REDIS_TOKEN_PREFIX + accessToken);
            redisTemplate.delete(AppConstant.REDIS_TOKEN_PREFIX + accessToken);
            redisTemplate.opsForList().remove(KEY_ONLINE_USER,0,accessToken);
            if (StringUtils.isEmpty(atcStr)) {
                return;
            }
            AccessTokenContent accessTokenContent = JSONObject.parseObject(atcStr, AccessTokenContent.class);
            if (accessTokenContent == null || !accessTokenContent.getCodeContent().isSendLogoutRequest()) {
                return;
            }

            sendLogoutRequest(accessTokenContent.getCodeContent().getRedirectUri(), accessToken);
        });
    }

    private String getKey(String tgt) {
        return tgt + "_access_token";
    }

    /**
     * accessToken时效为登录session时效的1/2
     */
    @Override
    public int getExpiresIn() {
        return timeout / 2;
    }
}
