package com.geoway.nsapp.common.data.util;

import com.baomidou.mybatisplus.annotation.TableName;
import com.geoway.nsapp.common.core.util.StringUtils;
import com.geoway.nsapp.common.data.po.TableField;

import java.lang.reflect.Field;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * @author by wangqiang
 * @date 2023/12/13.
 */
public class DBTableUtil {

    public static final Map<String, String> FIELD_TYPE_DIC = new HashMap<>();

    static {
        FIELD_TYPE_DIC.put("1", "text");
        FIELD_TYPE_DIC.put("2", "varchar");
        FIELD_TYPE_DIC.put("3", "decimal");
        FIELD_TYPE_DIC.put("4", "date");
        FIELD_TYPE_DIC.put("5", "timestamp");
        FIELD_TYPE_DIC.put("6", "varchar");
        FIELD_TYPE_DIC.put("7", "decimal");
        FIELD_TYPE_DIC.put("8", "int4");
        FIELD_TYPE_DIC.put("9", "geometry");
        FIELD_TYPE_DIC.put("10", "text");
    }

    public static String getDBType(String fieldTypeDic) {
        return FIELD_TYPE_DIC.getOrDefault(fieldTypeDic, "varchar");
    }

    public static String getDBAppType(String fieldTypeDic) {
        if ("3".equalsIgnoreCase(fieldTypeDic)
                || "7".equalsIgnoreCase(fieldTypeDic)
                || "8".equalsIgnoreCase(fieldTypeDic)) {
            return "number";
        } else {
            return "text";
        }
    }

    public static String generateFieldSql(List<TableField> list) {
        StringBuilder fieldSql = new StringBuilder();
        for (TableField f : list) {
            fieldSql.append(f.getFieldName()).append(" ");
            String fieldType = f.getFieldType();
            String dbType = FIELD_TYPE_DIC.getOrDefault(fieldType, "varchar");

            if (f.getLength() == null) {
                fieldSql.append(dbType);
            } else {
                if ("decimal".equals(dbType)) {
                    fieldSql
                            .append(dbType)
                            .append("(")
                            .append(f.getLength())
                            .append(",")
                            .append(f.getPrecision() == null ? 0 : f.getPrecision())
                            .append(")");
                } else if ("varchar".equals(dbType)) {
                    fieldSql.append(dbType).append("(").append(f.getLength()).append(")");
                } else {
                    fieldSql.append(dbType);
                }
            }
//            fieldSql.append(" ");
//            if (0==f.getNullAble()) {
//                fieldSql.append("NOT NULL ");
//            }
            fieldSql.append(",");
        }
        if (list.size() > 1) {
            fieldSql.append("primary key(f_id));");
        } else {
            return fieldSql.substring(0, fieldSql.length() - 1);
        }
        return fieldSql.toString();
    }

    public static String generateTableIndex(String tableName) {
        StringBuilder creatTableSql = new StringBuilder();

        // 创建索引: 行政区划代码索引(B-Tree索引) shape字段索引(GiST类型) 用户id索引(B-Tree索引)
        creatTableSql.append(
                MessageFormat.format("CREATE INDEX {0}_xzqdm_idx ON {0} ({1});", tableName, "f_xzqdm"));

//        creatTableSql.append(MessageFormat.format("CREATE INDEX {0}_shape_idx ON {0} USING GIST ({1});", tableName, "f_shape"));
//        creatTableSql.append(MessageFormat.format("CREATE INDEX {0}_upload_userid_idx ON {0} ({1});", tableName, "f_upload_userid"));
//        creatTableSql.append(MessageFormat.format("CREATE INDEX {0}_committime_idx ON {0} ({1});", tableName, "f_committime"));
//        creatTableSql.append(MessageFormat.format("CREATE INDEX {0}_upload_useridcommittime_idx ON {0} ({1},{2});", tableName, "f_upload_userid", "f_committime"));


        return creatTableSql.toString();
    }

    /**
     * 将待入库值转为sql格式
     *
     * @param storeFieldValues
     * @param tableFields
     * @return
     */
    public static Map<String, Object> getSqlValueRow(Map<String, Object> storeFieldValues, List<TableField> tableFields) {
        Map<String, Object> result = new HashMap();
        for (String key : storeFieldValues.keySet()) {
            Object sqlValue = null;
            Object attributeValue = storeFieldValues.get(key);
            if (attributeValue != null && StringUtils.isNotBlank(attributeValue.toString()) && !"null".equals(attributeValue.toString())) {
                TableField tableField = tableFields.stream().filter(f -> f.getFieldName().equals(key)).findFirst().orElse(null);
                if (tableField == null) continue;
                String fieldType = getDBType(tableField.getFieldType());
                switch (fieldType) {
                    case "varchar":
                    case "text":
                        // 处理sql特殊字符
                        attributeValue = escapeSql(attributeValue.toString());
                        sqlValue = "'" + attributeValue + "'";
                        break;
                    case "int4":
                    case "decimal":
                        sqlValue = attributeValue;
                        break;
                    case "date":
                        sqlValue = "'" + attributeValue + "'";
                        break;
                    case "timestamp":
                        Timestamp value5 = null;
                        try {
                            DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                            Date date = df.parse(attributeValue.toString());
                            value5 = new Timestamp(date.getTime());
                        } catch (java.text.ParseException e) {
                            DateTimeFormatter df =
                                    DateTimeFormatter.ofPattern("EEE MMM dd HH:mm:ss zzz yyyy", Locale.US);
                            LocalDateTime localDateTime = LocalDateTime.parse(attributeValue.toString(), df);
                            Date date = Date.from(localDateTime.atZone(ZoneOffset.ofHours(8)).toInstant());
                            value5 = new Timestamp(date.getTime());
                        }
                        sqlValue = "'" + value5 + "'";
                        break;
                    case "geometry":
//                        sqlValue = "ST_AsText('" + attributeValue + "')";
                        sqlValue = "ST_GeomFromText('SRID=4490;" + attributeValue + "')";
                        break;
                    default:
                        sqlValue = attributeValue;
                        break;
                }
            }
            if (sqlValue != null) {
                result.put(key, sqlValue);
            }
        }
        return result;
    }

    /**
     * 处理sql特殊字符
     * @param sql
     * @return
     */
    public static String escapeSql(String sql) {
        if (sql == null) {
            return null;
        }
        return sql.replace("'", "''")
//                .replace("\"", "\"\"")
//                .replace("\\", "\\\\")
//                .replace("/", "\\/")
                ;
    }

    /**
     * 根据对象，获取其在数据库中的字段名和值
     *
     * @param o
     * @return
     * @throws IllegalAccessException
     */
    public static HashMap<String, Object> getDbFieldNameValue(Object o) throws IllegalAccessException {
        HashMap<String, Object> fieldMap = new HashMap<>();
        TableName annotation = o.getClass().getAnnotation(TableName.class);
        if (annotation != null) {
            //获取表名 @TableName(value = "t_doctor")
            String tableName = annotation.value();
        }
        Field[] declaredFields = o.getClass().getDeclaredFields();
        for (Field field : declaredFields) {
            field.setAccessible(true);
            com.baomidou.mybatisplus.annotation.TableField tableField = field.getAnnotation(com.baomidou.mybatisplus.annotation.TableField.class);
            if (tableField != null) {
                //获取@TableField("hospital_id") 里面的列名
                String columnName = tableField.value();
                if (StringUtils.isNotBlank(columnName)) {
                    fieldMap.put(columnName, field.get(o));
                }
            }
        }
        return fieldMap;
    }

}
