package com.geoway.nsapp.common.data.util;

import cn.hutool.core.map.MapUtil;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.sql.*;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * SqlLite工具类
 *
 * @author zhangguodong
 * @since 2023-12-27
 */
@SuppressWarnings("unused")
@Slf4j
public class SqlLiteHelper {
    private final String _dbPath;

    /**
     * 数据库连接
     */
    @Getter
    @Setter
    private Connection connection = null;

    public SqlLiteHelper(String dbPath) {
        this._dbPath = dbPath;

        try {
            Class.forName("org.sqlite.JDBC");
            connection = DriverManager.getConnection("jdbc:sqlite:" + dbPath);
        } catch (Exception exception) {
            log.error("数据库连接失败", exception);
        }
    }

    /**
     * 获取第一行第一列的值
     *
     * @param sql sql语句
     * @return 第一行第一列的值
     */
    public Object getFirstScalar(String sql) {
        List<Map<String, Object>> queryList = queryList(sql);

        if (CollectionUtils.isEmpty(queryList)) {
            return null;
        }

        return queryList.get(0).values().iterator().next();
    }

    /**
     * 执行sql查询语句，返回查询结果
     *
     * @param sql sql语句
     * @return 查询结果
     */
    public List<Map<String, Object>> queryList(String sql) {
        try (Statement statement = createStatement(connection)) {
            if (statement == null) {
                throw new RuntimeException("db数据库操作失败");
            }

            ResultSet resultSet = statement.executeQuery(sql);
            return convertResultSetToList(resultSet);
        } catch (Exception e) {
            throw new RuntimeException("db数据库操作失败");
        }
    }

    /**
     * 根据指定的表名和 where 条件查询
     *
     * @param tableName 表名
     * @param where     where条件
     * @return 查询结果
     */
    public List<Map<String, Object>> queryList(String tableName, String where) {
        String sql = "select * from " + tableName;

        if (StringUtils.isNotBlank(where)) {
            sql += " where " + where;
        }

        return queryList(sql);
    }

    /**
     * 执行sql语句
     *
     * @param sql sql语句
     * @return 影响行数
     */
    public int executeSql(String sql) {
        try (Statement statement = createStatement(connection)) {
            if (statement == null) {
                throw new RuntimeException("db数据库操作失败");
            }

            return statement.executeUpdate(sql);
        } catch (Exception e) {
            throw new RuntimeException("db数据库操作失败");
        }
    }

    /**
     * 批量执行sql语句
     *
     * @param sqlArray sql语句数组
     */
    public void batchExecuteSql(String[] sqlArray) {
        try (Statement statement = createStatement(connection)) {
            if (statement == null) {
                throw new RuntimeException("db数据库操作失败");
            }

            for (String sql : sqlArray) {
                if (StringUtils.isBlank(sql)) {
                    continue;
                }

                statement.executeUpdate(sql);
            }
        } catch (Exception e) {
            throw new RuntimeException("db数据库操作失败");
        }
    }

    /**
     * 判断表是否存在
     *
     * @param tableName 表名
     * @return 存在返回true，不存在返回false
     */
    public boolean isTableExist(String tableName) {
        String sql = String.format("select count(1) from sqlite_master where type = 'table' and name = '%s'",
                tableName);
        Object count = getFirstScalar(sql);
        return count != null && Integer.parseInt(count.toString()) > 0;
    }

    /**
     * 获取所有表名
     *
     * @return 所有表名
     */
    public List<String> getTables() {
        String sql = "select name from sqlite_master where type='table'";
        List<Map<String, Object>> queryList = queryList(sql);

        if (CollectionUtils.isEmpty(queryList)) {
            return new ArrayList<>();
        }

        return queryList.stream()
                .map(map -> map.get("name").toString())
                .sorted()
                .collect(Collectors.toList());
    }

    private Statement createStatement(Connection connection) {
        if (connection == null) {
            throw new RuntimeException("db数据库操作失败");
        }

        try {
            return connection.createStatement();
        } catch (Exception exception) {
            log.error("创建statement失败", exception);
            return null;
        }
    }

    private Connection createConnection() {
        try {
            if (connection == null) {
                connection = DriverManager.getConnection("jdbc:sqlite:" + _dbPath);
            }

            return connection;
        } catch (Exception exception) {
            log.error("创建db连接失败", exception);
            return null;
        }
    }

    /**
     * 获取表字段
     *
     * @param tableName 表名
     * @return 表字段
     */
    public List<Map<String, Object>> getColumns(String tableName) {
        try {
            DatabaseMetaData databaseMetaData = connection.getMetaData();
            ResultSet resultSet = databaseMetaData.getColumns(null, null, tableName, null);
            return convertResultSetToList(resultSet);
        } catch (Exception exception) {
            log.error("获取表字段失败", exception);
            throw new RuntimeException("获取表字段失败：" + exception.getMessage());
        }
    }

    /**
     * 插入数据
     *
     * @param tableName 表名
     * @param data      数据
     * @return 是否插入成功
     */
    public boolean insert(String tableName, Map<String, Object> data) {
        Assert.notEmpty(data, "插入数据不能为空");

        return insertBatch(tableName, Collections.singletonList(data)) > 0;
    }

    /**
     * 批量插入数据
     *
     * @param tableName 表名
     * @param list      数据
     * @return 影响行数
     */
    public int insertBatch(String tableName, List<Map<String, Object>> list) {
        if (CollectionUtils.isEmpty(list)) {
            return 0;
        }

        List<Map<String, Object>> columns = getColumns(tableName);

        if (CollectionUtils.isEmpty(columns)) {
            return 0;
        }

        String insertSql = buildInsertSql(tableName, columns);

        try (PreparedStatement preparedStatement = connection.prepareStatement(insertSql)) {
            for (Map<String, Object> record : list) {
                for (int i = 0; i < columns.size(); i++) {
                    String fieldName = MapUtil.getStr(columns.get(i), "COLUMN_NAME");
                    String typeName = MapUtil.getStr(columns.get(i), "TYPE_NAME", "VARCHAR");

                    if (StringUtils.isNotBlank(typeName)) {
                        typeName = typeName.toUpperCase();

                        if (typeName.contains("(")) {
                            typeName = typeName.substring(0, typeName.indexOf("("));
                        }
                    }

                    Object value = record.get(fieldName);

                    if (value == null) {
                        preparedStatement.setNull(i + 1, Types.NULL);
                        continue;
                    }

                    switch (typeName) {
                        case "INTEGER":
                            try {
                                preparedStatement.setInt(i + 1, Integer.parseInt(value.toString()));
                            } catch (Exception exception) {
                                log.error("类型转换失败[" + value + "]->Integer");
                            }
                            break;
                        case "NUMERIC":
                        case "REAL":
                            try {
                                preparedStatement.setDouble(i + 1, Double.parseDouble(value.toString()));
                            } catch (Exception exception) {
                                log.error("类型转换失败[" + value + "]->Double");
                            }
                            break;
                        case "BLOB":
                            preparedStatement.setBytes(i + 1, (byte[]) value);
                            break;
                        default:
                            preparedStatement.setString(i + 1, value.toString());
                            break;
                    }
                }

                preparedStatement.addBatch();
            }

            int result = preparedStatement.executeBatch().length;
            preparedStatement.clearBatch();
            preparedStatement.closeOnCompletion();

//            connection.close();

            return result;
        } catch (Exception exception) {
            throw new RuntimeException("db数据库操作失败：" + exception.getMessage(), exception);
        }
    }

    public void close() {
        try {
            if (connection != null) {
                connection.close();
            }
        } catch (SQLException exception) {
            log.error("关闭db连接失败", exception);
        }
    }

    /**
     * 批量执行sql
     *
     * @param dbPath   数据库路径
     * @param sqlArray 批量sql
     */
    public static void batchExecuteSql(String dbPath, String[] sqlArray) {
        SqlLiteHelper sqlLiteHelper = new SqlLiteHelper(dbPath);
        sqlLiteHelper.batchExecuteSql(sqlArray);
        sqlLiteHelper.close();
    }

    /**
     * 在db中执行操作
     *
     * @param dbPath   数据库路径
     * @param consumer 操作
     */
    public static void executeInDb(String dbPath, Consumer<SqlLiteHelper> consumer) {
        SqlLiteHelper sqlLiteHelper = new SqlLiteHelper(dbPath);

        try {
            consumer.accept(sqlLiteHelper);
        } catch (Exception exception) {
            log.error("db数据库操作失败：" + exception.getMessage(), exception);
        } finally {
            if (sqlLiteHelper.connection != null) {
                try {
                    sqlLiteHelper.connection.close();
                    sqlLiteHelper.close();
                } catch (SQLException exception) {
                    log.error("关闭db连接失败", exception);
                }
            }
        }
    }

    private List<Map<String, Object>> convertResultSetToList(ResultSet resultSet) throws SQLException {
        List<Map<String, Object>> list = new ArrayList<>();

        while (resultSet.next()) {
            Map<String, Object> map = new LinkedHashMap<>();

            for (int i = 1; i <= resultSet.getMetaData().getColumnCount(); i++) {
                map.put(resultSet.getMetaData().getColumnName(i), resultSet.getObject(i));
            }

            list.add(map);
        }

        return list;
    }


    private String buildInsertSql(String tableName, List<Map<String, Object>> columns) {
        List<String> fieldNames = new ArrayList<>();
        List<String> values = new ArrayList<>();

        columns.forEach(column -> {
            fieldNames.add(column.get("COLUMN_NAME").toString());
            values.add("?");
        });

        return String.format("INSERT INTO %s (%s) VALUES (%s)",
                tableName,
                String.join(",", fieldNames),
                String.join(",", values));
    }
}
