package com.geoway.nsapp.common.file.service.impl;

import com.amazonaws.*;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.retry.PredefinedRetryPolicies;
import com.amazonaws.retry.RetryPolicy;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.*;
import com.geoway.nsapp.common.core.util.FileUtil;
import com.geoway.nsapp.common.file.dto.FileDownloadMeta;
import com.geoway.nsapp.common.file.dto.FileStoreMeta;
import com.geoway.nsapp.common.file.entity.FileServer;
import com.geoway.nsapp.common.file.service.AWSOssFileService;
import io.minio.MinioClient;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.*;
import java.net.URL;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.*;

/**
 * @author by wangqiang
 * @date 2025/7/14.
 */
@Service
@Slf4j
public class AWSOssFileServiceImpl implements AWSOssFileService {
    @Autowired
    @Qualifier("redisTemplate")
    private RedisTemplate redisTemplate;

    // 请求超时时间，单位秒
    @Value("${fileStorage.upload.timeout:600}")
    private Integer timeout;

    private AmazonS3 _awsClient;

    private int _currentServerId = 0;

    private static long CHUNK_PART_SIZE = 10 * 1024 * 1024L;

    // 并发上传的分片数量
    private static final int CONCURRENT_PARTS = 10;

    @Override
    public AmazonS3 getOssClient(FileServer fileServer) {
        if (fileServer == null) {
            return null;
        }

        String preKey = "fileServer-";
        boolean hasExist = redisTemplate.hasKey(preKey + fileServer.getId());
        if (_awsClient == null || !hasExist || fileServer.getId() != _currentServerId) {
            _currentServerId = fileServer.getId();

            AWSCredentials credentials = null;
            credentials = new BasicAWSCredentials(fileServer.getAppkey(), fileServer.getAppsecret());

            ClientConfiguration config = new ClientConfiguration();
            if (fileServer.getEndpoint().startsWith("https"))
                config.setProtocol(Protocol.HTTPS);
            else config.setProtocol(Protocol.HTTP);

            config.setUseExpectContinue(true);

            config.setSocketTimeout(2 * 60 * 1000); // 分片上传时，单个分片的网络传输时间
            config.setConnectionTimeout(10 * 1000); // 10秒连接超时
            config.setRequestTimeout(timeout * 1000); // 总请求超时（包括重试）

            config.setRetryPolicy(new RetryPolicy(
                    PredefinedRetryPolicies.DEFAULT_RETRY_CONDITION,
                    PredefinedRetryPolicies.DEFAULT_BACKOFF_STRATEGY,
                    1,  // 最大重试次数
                    true  // 是否允许重试
            ));

            AwsClientBuilder.EndpointConfiguration end_point = null;
            end_point = new AwsClientBuilder.EndpointConfiguration(fileServer.getEndpoint(), "");
            _awsClient = AmazonS3ClientBuilder.standard()
                    .withCredentials(new AWSStaticCredentialsProvider(credentials))
                    .withClientConfiguration(config)
                    .withEndpointConfiguration(end_point)
                    .withPathStyleAccessEnabled(true)
                    .build();

            redisTemplate.opsForValue().set(preKey + _currentServerId, _currentServerId + "", 10, TimeUnit.HOURS);
        }

        return _awsClient;
    }

    @Override
    public FileStoreMeta sendFile(FileServer fileServer, MultipartFile file, String objectName) throws Exception {
        AmazonS3 s3Client = getOssClient(fileServer);
        String contentType = FileUtil.getContentType(file.getOriginalFilename());
        uploadMultipartFromStream(s3Client, fileServer.getBucket(), objectName, file.getInputStream(), contentType);

        FileStoreMeta fileStoreMeta = new FileStoreMeta();
        fileStoreMeta.setRelPath(objectName);
        fileStoreMeta.setName(objectName.substring(objectName.lastIndexOf('/') + 1));
        fileStoreMeta.setFileSize(file.getSize());
        fileStoreMeta.setFileType(file.getContentType());
        fileStoreMeta.setPreviewURL(buildPreviewURL(fileServer, objectName, 3600));

        return fileStoreMeta;
    }

    @Override
    public FileStoreMeta sendFile(FileServer fileServer, File file, String objectName) throws Exception {
        AmazonS3 s3Client = getOssClient(fileServer);
        String contentType = FileUtil.getContentType(file.getName());

        // 上传
        uploadMultipartFromStream(s3Client, fileServer.getBucket(), objectName, new FileInputStream(file), contentType);

        String fileName = file.getName();
        String fileType = fileName.substring(fileName.lastIndexOf(".") + 1);
        FileStoreMeta fileStoreMeta = new FileStoreMeta();
        fileStoreMeta.setRelPath(objectName);
        fileStoreMeta.setName(objectName.substring(objectName.lastIndexOf('/') + 1));
        fileStoreMeta.setFileSize(file.length());
        fileStoreMeta.setFileType(fileType);
        fileStoreMeta.setPreviewURL(buildPreviewURL(fileServer, objectName, 3600));

        return fileStoreMeta;
    }

    /**
     * 并发分片上传输入流到S3，控制每次并发10个分片
     */
    public static void uploadMultipartFromStream(AmazonS3 s3Client, String bucketName, String key,
                                                 InputStream inputStream, String contentType) throws IOException, InterruptedException, ExecutionException {

        // 1. 初始化分片上传
        InitiateMultipartUploadRequest initRequest = new InitiateMultipartUploadRequest(bucketName, key);
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentType(contentType);
        initRequest.setObjectMetadata(metadata);

        InitiateMultipartUploadResult initResponse;
        try {
            initResponse = s3Client.initiateMultipartUpload(initRequest);
        } catch (AmazonServiceException e) {
            throw new RuntimeException("初始化分片上传失败: " + e.getMessage(), e);
        }
        String uploadId = initResponse.getUploadId();
        List<PartETag> partETags = new ArrayList<>();

        // 2. 创建固定大小的线程池（核心线程和最大线程均为10）
        ThreadPoolExecutor executor = new ThreadPoolExecutor(
                CONCURRENT_PARTS,
                CONCURRENT_PARTS,
                0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(CONCURRENT_PARTS), // 队列容量为10，控制待处理任务数
                new ThreadPoolExecutor.CallerRunsPolicy() // 队列满时由提交线程执行，起到限流作用
        );

        try {
            int partNumber = 1;
            byte[] buffer = new byte[(int) CHUNK_PART_SIZE];
            int bytesRead;
            long bytesUploaded = 0;

            // 用于临时保存当前批次的Future任务
            List<Future<PartETag>> currentBatchFutures = new ArrayList<>(CONCURRENT_PARTS);

            // 3. 分块读取并控制并发上传
            while ((bytesRead = readFully(inputStream, buffer)) != -1) {
                // 复制当前分片数据（避免buffer被后续读取覆盖）
                final byte[] partData = new byte[bytesRead];
                System.arraycopy(buffer, 0, partData, 0, bytesRead);
                final int currentPartNumber = partNumber;

                // 提交分片上传任务
                Future<PartETag> future = executor.submit(() -> {
                    try (ByteArrayInputStream partStream = new ByteArrayInputStream(partData)) {
                        UploadPartRequest uploadRequest = new UploadPartRequest()
                                .withBucketName(bucketName)
                                .withKey(key)
                                .withUploadId(uploadId)
                                .withPartNumber(currentPartNumber)
                                .withInputStream(partStream)
                                .withPartSize(partData.length);

                        UploadPartResult result = s3Client.uploadPart(uploadRequest);
                        System.out.printf("已完成分片 %d，大小: %.2f MB%n",
                                currentPartNumber, partData.length / (1024.0 * 1024));
                        return result.getPartETag();
                    } catch (AmazonServiceException e) {
                        throw new RuntimeException("分片 " + currentPartNumber + " 上传失败: " + e.getMessage(), e);
                    }
                });

                currentBatchFutures.add(future);
                bytesUploaded += bytesRead;
                partNumber++;

                // 当累积到10个任务时，等待所有任务完成再继续（控制并发数）
                if (currentBatchFutures.size() >= CONCURRENT_PARTS) {
                    for (Future<PartETag> f : currentBatchFutures) {
                        partETags.add(f.get()); // 阻塞等待任务完成
                    }
                    currentBatchFutures.clear(); // 清空批次，准备下一批
                    System.out.printf("已完成一批分片上传，累计上传: %.2f MB%n",
                            bytesUploaded / (1024.0 * 1024));
                }
            }

            // 处理剩余不足10个的分片任务
            if (!currentBatchFutures.isEmpty()) {
                for (Future<PartETag> f : currentBatchFutures) {
                    partETags.add(f.get());
                }
            }

            // 4. 完成分片上传
            CompleteMultipartUploadRequest compRequest = new CompleteMultipartUploadRequest(
                    bucketName, key, uploadId, partETags);
            s3Client.completeMultipartUpload(compRequest);
            System.out.println("所有分片上传完成，文件组装成功");

        } catch (Exception e) {
            // 异常时中止上传
            s3Client.abortMultipartUpload(new AbortMultipartUploadRequest(bucketName, key, uploadId));
            System.err.println("上传失败，已中止分片上传: " + e.getMessage());
            throw e;
        } finally {
            executor.shutdown(); // 关闭线程池
            inputStream.close(); // 关闭输入流
        }
    }

    /**
     * 从输入流读取指定长度的数据（解决read方法可能读取不完整的问题）
     */
    private static int readFully(InputStream input, byte[] buffer) throws IOException {
        int bytesRead = 0;
        int totalRead = 0;
        while (totalRead < buffer.length &&
                (bytesRead = input.read(buffer, totalRead, buffer.length - totalRead)) != -1) {
            totalRead += bytesRead;
        }
        return totalRead == 0 ? -1 : totalRead;
    }

    @Override
    public FileStoreMeta buildThumbnailFile(FileServer fileServer, String objectName) throws Exception {
        // 缩略图生成逻辑可根据具体需求实现，这里简单返回一个包含对象名的 FileStoreMeta
        FileStoreMeta fileStoreMeta = new FileStoreMeta();
        fileStoreMeta.setName(objectName);
        return fileStoreMeta;
    }

    @Override
    public void deleteFile(FileServer fileServer, String bucket, String path) throws Exception {
        AmazonS3 s3Client = getOssClient(fileServer);
        try {
            s3Client.deleteObject(bucket, path);
        } catch (SdkClientException e) {
            log.error("Failed to delete file from S3: {}", e.getMessage(), e);
            throw new Exception("Failed to delete file from S3", e);
        }
    }

    @Override
    public String buildPreviewURL(FileServer fileServer, String objectName, Integer expires) throws Exception {
        AmazonS3 s3Client = getOssClient(fileServer);
        Date expiration = new Date();
        long expTimeMillis = expiration.getTime();
        expTimeMillis += expires * 1000;
        expiration.setTime(expTimeMillis);
        GeneratePresignedUrlRequest generatePresignedUrlRequest = new GeneratePresignedUrlRequest(fileServer.getBucket(), objectName)
                .withMethod(HttpMethod.GET)
                .withExpiration(expiration);
        URL url = s3Client.generatePresignedUrl(generatePresignedUrlRequest);
        return url.toString();
    }

    @Override
    public String getPreviewURL(FileServer fileServer, String url, Integer expires) throws Exception {
        // 截取objectName
        String objectName;
        try {
            URL urls = new URL(url);
            objectName = urls.getPath();
            String bucket = fileServer.getBucket();
            objectName = objectName.replace(bucket, "");

            while (objectName.startsWith("/")) {
                objectName = objectName.substring(1);
            }
        } catch (Exception e) {
            objectName = url;
        }
        return buildPreviewURL(fileServer, objectName, expires);
    }

    @Override
    public byte[] createThumbnail(FileServer fileServer, String objectName) throws Exception {
        // 缩略图生成逻辑可根据具体需求实现，这里简单返回空字节数组
        return new byte[0];
    }

    @Override
    public FileDownloadMeta downLoadFile(FileServer fileServer, String objectName) throws Exception {
        AmazonS3 s3Client = getOssClient(fileServer);
        try {
            S3Object s3Object = s3Client.getObject(fileServer.getBucket(), objectName);
            S3ObjectInputStream inputStream = s3Object.getObjectContent();
            FileDownloadMeta fileDownloadMeta = new FileDownloadMeta();
            fileDownloadMeta.setInputStream(inputStream);
            return fileDownloadMeta;
        } catch (SdkClientException e) {
            log.error("Failed to download file from S3: {}", e.getMessage(), e);
            throw new Exception("Failed to download file from S3", e);
        }
    }
}