package com.geoway.nsapp.common.file.service.impl;

import cn.hutool.core.util.StrUtil;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.model.GeneratePresignedUrlRequest;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.PutObjectRequest;
import com.geoway.nsapp.common.core.util.FileUtil;
import com.geoway.nsapp.common.file.dto.FileDownloadMeta;
import com.geoway.nsapp.common.file.dto.FileStoreMeta;
import com.geoway.nsapp.common.file.entity.FileServer;
import com.geoway.nsapp.common.file.service.AliOssFileService;
import com.geoway.nsapp.common.file.util.ObjectKeyUtil;
import io.minio.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * @author by wangqiang
 * @date 2025/7/11.
 */
@Service
@Slf4j
public class AliOssFileServiceImpl implements AliOssFileService {

    @Override
    public OSS getOssClient(FileServer fileServer) throws Exception {
        String endpoint = fileServer.getEndpoint();
        String accessKeyId = fileServer.getAppkey();
        String accessKeySecret = fileServer.getAppsecret();
        return new OSSClientBuilder().build(endpoint, accessKeyId, accessKeySecret);
    }

    @Override
    public FileStoreMeta sendFile(FileServer fileServer, File file, String objectName) throws Exception {
        if (StrUtil.isBlank(objectName)) {
            objectName = ObjectKeyUtil.formatObjectName(file.getName());
        }

        String contentType = FileUtil.getContentType(file.getName());
        return sendFileStream(fileServer,file.getName(),objectName,new FileInputStream(file),file.length(), contentType);
    }

    @Override
    public FileStoreMeta sendFile(FileServer fileServer, MultipartFile multipartFile, String objectName) throws Exception {
        if (StrUtil.isBlank(objectName)) {
            objectName = ObjectKeyUtil.formatObjectName(multipartFile.getOriginalFilename());
        }

        String contentType = FileUtil.getContentType(multipartFile.getOriginalFilename());

        return sendFileStream(fileServer,multipartFile.getOriginalFilename(),objectName,multipartFile.getInputStream(),multipartFile.getSize(),contentType);
    }

    private FileStoreMeta sendFileStream(FileServer fileServer, String fileName, String objectName, InputStream inputStream, Long fileSize, String contentType) throws Exception {
        try {
            FileStoreMeta fileStoreMeta = new FileStoreMeta();
            String fileType = fileName.substring(fileName.lastIndexOf(".") + 1);
            String bucket = fileServer.getBucket();

            if (StrUtil.isBlank(objectName)) {
                objectName = ObjectKeyUtil.formatObjectName(fileName);
            }

            OSS ossClient = getOssClient(fileServer);
            String bucketName = fileServer.getBucket();
            java.io.File localFile = new java.io.File(fileName);

            ossClient.putObject(bucketName, objectName, localFile);
            fileStoreMeta.setAbsolutePath("oss://" + bucketName + "/" + objectName);
            fileStoreMeta.setRelPath(objectName);
            fileStoreMeta.setName(objectName.substring(objectName.lastIndexOf('/') + 1));
            fileStoreMeta.setFileSize(localFile.length());
            // 这里简单根据文件扩展名判断文件类型，实际应用可能需要更复杂的逻辑
            fileStoreMeta.setFileType(fileType);
            fileStoreMeta.setPreviewURL(buildPreviewURL(fileServer, objectName, 3600));
            return fileStoreMeta;
        } catch (Exception ex) {
            throw new Exception(ex);
        } finally {
            inputStream.close();
        }
    }

    @Override
    public FileStoreMeta buildThumbnailFile(FileServer fileServer, String objectName) throws Exception {
        OSS ossClient = getOssClient(fileServer);
        String bucketName = fileServer.getBucket();
        String thumbnailObjectName = objectName + "_thumbnail";
        GetObjectRequest getObjectRequest = new GetObjectRequest(bucketName, objectName);
        OSSObject ossObject = ossClient.getObject(getObjectRequest);
        try (InputStream inputStream = ossObject.getObjectContent()) {
            // 这里应该添加缩略图处理逻辑，比如使用 ImageIO 等库
            // 示例中直接将原文件作为缩略图上传
            ossClient.putObject(bucketName, thumbnailObjectName, inputStream);
            FileStoreMeta fileStoreMeta = new FileStoreMeta();
            fileStoreMeta.setAbsolutePath("oss://" + bucketName + "/" + thumbnailObjectName);
            fileStoreMeta.setRelPath(thumbnailObjectName);
            fileStoreMeta.setName(thumbnailObjectName.substring(thumbnailObjectName.lastIndexOf('/') + 1));
            // 这里假设缩略图大小和原文件相同，实际需要根据处理逻辑调整
            fileStoreMeta.setFileSize(ossObject.getObjectMetadata().getContentLength());
            fileStoreMeta.setFileType(ossObject.getObjectMetadata().getContentType());
            fileStoreMeta.setPreviewURL(buildPreviewURL(fileServer, thumbnailObjectName, 3600));
            return fileStoreMeta;
        } finally {
            ossClient.shutdown();
        }
    }

    @Override
    public void deleteFile(FileServer fileServer, String bucket, String path) throws Exception {
        OSS ossClient = getOssClient(fileServer);
        try {
            ossClient.deleteObject(bucket, path);
        } finally {
            ossClient.shutdown();
        }
    }

    @Override
    public String buildPreviewURL(FileServer fileServer, String objectName, Integer expires) throws Exception {
        OSS ossClient = getOssClient(fileServer);
        String bucketName = fileServer.getBucket();
        try {
            Date expiration = new Date(System.currentTimeMillis() + expires * 1000);
            URL url = ossClient.generatePresignedUrl(bucketName, objectName, expiration);
            return url.toString();
        } finally {
            ossClient.shutdown();
        }
    }

    @Override
    public byte[] createThumbnail(FileServer fileServer, String objectName) throws Exception {
        OSS ossClient = getOssClient(fileServer);
        String bucketName = fileServer.getBucket();
        GetObjectRequest getObjectRequest = new GetObjectRequest(bucketName, objectName);
        OSSObject ossObject = ossClient.getObject(getObjectRequest);
        try (InputStream inputStream = ossObject.getObjectContent()) {
            // 这里应该添加缩略图处理逻辑，比如使用 ImageIO 等库
            // 示例中直接返回原文件内容
            byte[] buffer = new byte[inputStream.available()];
            inputStream.read(buffer);
            return buffer;
        } finally {
            ossClient.shutdown();
        }
    }

    @Override
    public FileDownloadMeta downLoadFile(FileServer fileServer, String objectName) throws Exception {
        OSS ossClient = getOssClient(fileServer);
        String bucketName = fileServer.getBucket();
        GetObjectRequest getObjectRequest = new GetObjectRequest(bucketName, objectName);
        OSSObject ossObject = ossClient.getObject(getObjectRequest);
        try (InputStream inputStream = ossObject.getObjectContent()) {
            FileDownloadMeta fileDownloadMeta = new FileDownloadMeta();
            fileDownloadMeta.setFileSize(ossObject.getObjectMetadata().getContentLength());
            fileDownloadMeta.setInputStream(inputStream);
            return fileDownloadMeta;
        } finally {
            ossClient.shutdown();
        }
    }

    @Override
    public String getPreviewURL(FileServer fileServer, String url, Integer expires) throws Exception {
        // 截取objectName
        String objectName;
        try {
            URL urls = new URL(url);
            objectName = urls.getPath();
            String bucket = fileServer.getBucket();
            objectName = objectName.replace(bucket, "");

            while (objectName.startsWith("/")) {
                objectName = objectName.substring(1);
            }
        } catch (Exception e) {
            objectName = url;
        }
        return buildPreviewURL(fileServer, objectName, expires);
    }
}