package com.geoway.nsapp.common.file.service.impl;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.geoway.atlas.license.authorize.LicenseCheck;
import com.geoway.nsapp.common.file.dto.FileDownloadMeta;
import com.geoway.nsapp.common.file.dto.FileServerResponse;
import com.geoway.nsapp.common.file.dto.FileStoreMeta;
import com.geoway.nsapp.common.file.entity.FileServer;
import com.geoway.nsapp.common.file.mapper.FileServerMapper;
import com.geoway.nsapp.common.file.service.*;
import com.geoway.nsapp.common.file.util.ObjectKeyUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.InputStream;

/**
 * @author daidongdong
 * @description 针对表【tbsys_fileserver】的数据库操作Service实现
 * @createDate 2022-06-29 15:24:19
 */
@Service
public class FileServerServiceImpl extends ServiceImpl<FileServerMapper, FileServer>
        implements FileServerService {

    @Autowired
    ShareFileService shareFileService;

    @Autowired
    MinIOFileService minIOFileService;

    @Autowired
    HuWeiObsFileService huWeiObsFileService;

    @Autowired
    AliOssFileService aliOssFileService;

    @Autowired
    AWSOssFileService awsOssFileService;

    @Autowired
    FileServerMapper fileServerMapper;

    static {
        synchronized (LicenseCheck.class) {
            LicenseCheck.isValid(65035);
        }
    }

    @Override
    public FileServerResponse sendFileToServer(MultipartFile file) throws Exception {
        return sendFileToServer(file, false);
    }

    @Override
    public FileServerResponse sendFileToServer(MultipartFile file, boolean buildThumbnail) throws Exception {
        return sendFileToServer(file, null, buildThumbnail);
    }

    @Override
    public FileServerResponse sendFileToServer(MultipartFile file, String objectName, boolean buildThumbnail) throws Exception {
        FileServer fileServer = this.findDefault();
        if (fileServer == null) {
            throw new Exception("未配置文件服务");
        }

        return sendFileToServer(fileServer.getId().toString(), file, objectName, buildThumbnail);
    }

    @Override
    public FileServerResponse sendFileToServer(String serverId, MultipartFile file, String objectName, boolean buildThumbnail) throws Exception {
        FileServer fileServer = this.findById(Integer.parseInt(serverId));
        if (fileServer == null) {
            throw new Exception("未配置文件服务");
        }
        return saveFile(fileServer, file, objectName, buildThumbnail);

    }

    @Override
    public FileServerResponse sendFileToServer(File file, String objectName, boolean buildThumbnail) throws Exception {
        FileServer fileServer = this.findDefault();
        if (fileServer == null) {
            throw new Exception("未配置文件服务");
        }
        return sendFileToServer(fileServer.getId().toString(), file, objectName, buildThumbnail);
    }

    @Override
    public FileServerResponse sendFileToServer(String serverId, File file, String objectName, boolean buildThumbnail) throws Exception {
        FileServer fileServer = this.findById(Integer.parseInt(serverId));

        return saveFile(fileServer, file, objectName, buildThumbnail);

    }

    @Override
    public FileServerResponse sendFileToServerOnSpeFolder(MultipartFile file, String foldName) throws Exception {
        return sendFileToServerOnSpeFolder(file, foldName, false);
    }

    @Override
    public FileServerResponse sendFileToServerOnSpeFolder(MultipartFile file, String foldName, boolean buildThumbnail) throws Exception {
        FileServer fileServer = this.findDefault();
        if (fileServer == null) {
            throw new Exception("未配置文件服务");
        }

        String fileName = file.getOriginalFilename();
        String objectName = ObjectKeyUtil.formatObjectName(fileName);
        if (StrUtil.isNotBlank(foldName)) {
            objectName = foldName + "/" + objectName;
        }

        return sendFileToServer(file, objectName, buildThumbnail);
    }


    @Override
    public FileServerResponse sendFileToServerByStream(InputStream stream, String objectName, String contentType) throws Exception {
        return null;
    }

    private FileServerResponse saveFile(FileServer fileServer, MultipartFile file, String objectName, Boolean buildThumbnail) throws Exception {
        String type = fileServer.getType();
        FileStoreMeta fileStoreMeta = null;
        FileServerResponse fileServerResponse = new FileServerResponse();
        if ("share".equalsIgnoreCase(type)) {

        } else if ("minio".equalsIgnoreCase(type)) {
            fileStoreMeta = minIOFileService.sendFile(fileServer, file, objectName);
            if (buildThumbnail) {
//                fileStoreMeta = minIOFileService.buildThumbnailFile(fileServer, fileStoreMeta.getRelPath());
            }
        } else if ("obs".equalsIgnoreCase(type)) {
            fileStoreMeta = huWeiObsFileService.sendFile(fileServer, file, objectName);
            if (buildThumbnail) {
//                fileStoreMeta = huWeiObsFileService.buildThumbnailFile(fileServer, fileStoreMeta.getRelPath());
            }
        } else if ("oss".equalsIgnoreCase(type)) {
            fileStoreMeta = aliOssFileService.sendFile(fileServer, file, objectName);
            if (buildThumbnail) {
//                fileStoreMeta = aliOssFileService.buildThumbnailFile(fileServer, fileStoreMeta.getRelPath());
            }
        } else if ("aws".equalsIgnoreCase(type)) {
            fileStoreMeta = awsOssFileService.sendFile(fileServer, file, objectName);
            if (buildThumbnail) {
//                fileStoreMeta = awsOssFileService.buildThumbnailFile(fileServer, fileStoreMeta.getRelPath());
            }
        } else {
            throw new RuntimeException("不支持的类型：" + type);
        }
        fileServerResponse.setServerURL(fileServer.getUrl());
        fileServerResponse.setBucket(fileServer.getBucket());
        if (fileStoreMeta != null) {
            fileServerResponse.setPath(fileStoreMeta.getRelPath());
            fileServerResponse.setFileSize(fileStoreMeta.getFileSize());
            fileServerResponse.setFileType(fileStoreMeta.getFileType());
            fileServerResponse.setPreviewURL(replacePreviewURLIp(fileStoreMeta.getPreviewURL(), fileServer));
            fileServerResponse.setViewUrl(fileServer.getUrl() + "/" + fileServer.getBucket() + "/" + fileStoreMeta.getRelPath());
        }
        return fileServerResponse;
    }

    private FileServerResponse saveFile(FileServer fileServer, File file, String objectName, Boolean buildThumbnail) throws Exception {
        String type = fileServer.getType();
        FileStoreMeta fileStoreMeta = null;
        FileServerResponse fileServerResponse = new FileServerResponse();
        if ("share".equalsIgnoreCase(type)) {

        } else if ("minio".equalsIgnoreCase(type)) {
            fileStoreMeta = minIOFileService.sendFile(fileServer, file, objectName);
            if (buildThumbnail) {
//                fileStoreMeta = minIOFileService.buildThumbnailFile(fileServer, fileStoreMeta.getRelPath());
            }
        } else if ("obs".equalsIgnoreCase(type)) {
            fileStoreMeta = huWeiObsFileService.sendFile(fileServer, file, objectName);
            if (buildThumbnail) {
//                fileStoreMeta = huWeiObsFileService.buildThumbnailFile(fileServer, fileStoreMeta.getRelPath());
            }
        } else if ("oss".equalsIgnoreCase(type)) {
            fileStoreMeta = aliOssFileService.sendFile(fileServer, file, objectName);
            if (buildThumbnail) {
//                fileStoreMeta = aliOssFileService.buildThumbnailFile(fileServer, fileStoreMeta.getRelPath());
            }
        } else if ("aws".equalsIgnoreCase(type)) {
            fileStoreMeta = awsOssFileService.sendFile(fileServer, file, objectName);
            if (buildThumbnail) {
//                fileStoreMeta = awsOssFileService.buildThumbnailFile(fileServer, fileStoreMeta.getRelPath());
            }
        } else {
            throw new RuntimeException("不支持的类型：" + type);
        }
        fileServerResponse.setServerURL(fileServer.getUrl());
        fileServerResponse.setBucket(fileServer.getBucket());
        if (fileStoreMeta != null) {
            fileServerResponse.setPath(fileStoreMeta.getRelPath());
            fileServerResponse.setFileSize(fileStoreMeta.getFileSize());
            fileServerResponse.setFileType(fileStoreMeta.getFileType());
            fileServerResponse.setPreviewURL(replacePreviewURLIp(fileStoreMeta.getPreviewURL(), fileServer));
            fileServerResponse.setViewUrl(fileServer.getUrl() + "/" + fileServer.getBucket() + "/" + fileStoreMeta.getRelPath());
        }
        return fileServerResponse;
    }

    @Override
    public void deleteFile(String objectName) throws Exception {
        FileServer fileServer = this.findDefault();
        deleteFileByFileServer(objectName, fileServer);
    }

    @Override
    public void deleteFile(Integer serverId, String objectName) throws Exception {
        FileServer fileServer = this.findById(serverId);
        deleteFileByFileServer(objectName, fileServer);
    }

    private void deleteFileByFileServer(String objectName, FileServer fileServer) throws Exception {
        String type = fileServer.getType().toLowerCase();

        switch (type) {
            case "share":
                shareFileService.deleteFile(fileServer, objectName);
                break;
            case "minio":
                minIOFileService.deleteFile(fileServer, fileServer.getBucket(), objectName);
                break;
            case "obs":
                huWeiObsFileService.deleteFile(fileServer, fileServer.getBucket(), objectName);
                break;
            case "oss":
                aliOssFileService.deleteFile(fileServer, fileServer.getBucket(), objectName);
                break;
            default:
                break;

        }
    }

    @Override
    @Cacheable(value = "FileServerCache#10", key = "#root.methodName")
    public FileServer findDefault() {
        LambdaQueryWrapper<FileServer> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(FileServer::getIsdefault, 1);
        wrapper.last("LIMIT 1");

        FileServer fileServer = this.getOne(wrapper);

        return fileServer;
    }

    @Override
    @Cacheable(value = "FileServerCache#10", key = "#root.methodName+ '-' + #id")
    public FileServer findById(Integer id) {
        return this.getById(id);
    }

    @Override
    public String buildPreviewURL(Integer serverId, String objectName, Integer expires) throws Exception {

        String previewURL = "";
        FileServer fileServer = this.getById(serverId);
        if (fileServer == null) {
            return previewURL;
        }
        String type = fileServer.getType().toLowerCase();

        switch (type) {
            case "share":
                previewURL = shareFileService.buildPreviewURL(fileServer, objectName, expires);
                break;
            case "minio":
                previewURL = minIOFileService.buildPreviewURL(fileServer, objectName, expires);
                break;
            case "obs":
                previewURL = huWeiObsFileService.buildPreviewURL(fileServer, objectName, expires);
                break;
            case "oss":
                previewURL = aliOssFileService.buildPreviewURL(fileServer, objectName, expires);
                break;
            case "aws":
                previewURL = awsOssFileService.buildPreviewURL(fileServer, objectName, expires);
                break;
            default:
                break;

        }
        String fileEndPoint = fileServer.getEndpoint();
        String fileUrl = fileServer.getUrl();
        if (previewURL.contains(fileEndPoint)) {
            previewURL = previewURL.replace(fileEndPoint, fileUrl);
        }
        return previewURL;
    }

    @Override
    public String getPreviewURL(Integer serverId, String url, Integer expires) throws Exception {
        String previewURL = "";
        FileServer fileServer = this.getById(serverId);
        if (fileServer == null) {
            return previewURL;
        }
        String type = fileServer.getType().toLowerCase();
        expires = expires == null ? 3600 : expires;
        switch (type) {
            case "share":
                break;
            case "minio":
                previewURL = minIOFileService.getPreviewURL(fileServer, url, expires);
                break;
            case "obs":
                previewURL = huWeiObsFileService.getPreviewURL(fileServer, url, expires);
                break;
            case "oss":
                previewURL = aliOssFileService.getPreviewURL(fileServer, url, expires);
                break;
            case "aws":
                previewURL = awsOssFileService.getPreviewURL(fileServer, url, expires);
                break;
            default:
                break;
        }

        return replacePreviewURLIp(previewURL, fileServer);
    }

    private String replacePreviewURLIp(String previewURL, FileServer fileServer) {
        if (StrUtil.isBlank(previewURL)) return previewURL;
        String fileEndPoint = fileServer.getEndpoint();
        String fileUrl = fileServer.getUrl();
        if (previewURL.contains(fileEndPoint)) {
            previewURL = previewURL.replace(fileEndPoint, fileUrl);
        }
        return previewURL;
    }

    @Override
    public byte[] createThumail(Integer serverId, String objectName) throws Exception {

        FileServer fileServer = this.getById(serverId);
        if (fileServer == null) {
            return new byte[0];
        }

        byte[] thumnailBytes = new byte[0];
        String type = fileServer.getType().toLowerCase();
        switch (type) {
            case "share":
                thumnailBytes = shareFileService.createThumbnail(fileServer, objectName);
                break;
            case "minio":
                thumnailBytes = minIOFileService.createThumbnail(fileServer, objectName);
                break;
            case "obs":
                thumnailBytes = huWeiObsFileService.createThumbnail(fileServer, objectName);
                break;
            case "oss":
                thumnailBytes = aliOssFileService.createThumbnail(fileServer, objectName);
                break;
            case "aws":
                thumnailBytes = awsOssFileService.createThumbnail(fileServer, objectName);
                break;
            default:
                break;

        }

        return thumnailBytes;
    }

    @Override
    public FileDownloadMeta queryDownLoadMeta(Integer serverId, String objectName) throws Exception {

        FileServer fileServer = this.getById(serverId);
        if (fileServer == null) {
            return null;
        }

        return getFileDownloadMeta(objectName, fileServer);
    }


    @Override
    public FileDownloadMeta queryDownLoadMeta(String objectName) throws Exception {
        FileServer fileServer = this.findDefault();
        if (fileServer == null) {
            return null;
        }
        return getFileDownloadMeta(objectName, fileServer);
    }

    private FileDownloadMeta getFileDownloadMeta(String objectName, FileServer fileServer) throws Exception {
        FileDownloadMeta fileDownloadMeta = null;
        String type = fileServer.getType().toLowerCase();
        switch (type) {
            case "share":
                fileDownloadMeta = shareFileService.downLoadFile(fileServer, objectName);
                break;
            case "minio":
                fileDownloadMeta = minIOFileService.downLoadFile(fileServer, objectName);
                break;
            case "obs":
                fileDownloadMeta = huWeiObsFileService.downLoadFile(fileServer, objectName);
                break;
            default:
                break;

        }

        return fileDownloadMeta;
    }

    @Override
    @CacheEvict(value = "FileServerCache#10", allEntries = true)
    public void deleteCache() {
    }

    @Override
    public FileServer saveOne(FileServer fileServer) {
        if (fileServer.getId() == null || fileServer.getId() == 0) {
            int maxId = fileServerMapper.findMaxId();
            fileServer.setId(maxId + 1);
        }
        int hasDefault = fileServerMapper.hasDefault();
        if (hasDefault == 0) {
            fileServer.setIsdefault(1);
        }
        this.saveOrUpdate(fileServer);

        return fileServer;
    }

    @Override
    public void setDefault(Integer serverId) {
        LambdaUpdateWrapper<FileServer> wrapper = Wrappers.lambdaUpdate();
        wrapper.set(FileServer::getIsdefault, 0);

        this.update(wrapper);

        wrapper.clear();
        wrapper.eq(FileServer::getId, serverId);
        wrapper.set(FileServer::getIsdefault, 1);
        this.update(wrapper);
    }
}




