package com.geoway.nsapp.common.file.service.impl;

import cn.hutool.core.util.StrUtil;
import com.geoway.nsapp.common.file.dto.FileDownloadMeta;
import com.geoway.nsapp.common.file.dto.FileStoreMeta;
import com.geoway.nsapp.common.file.entity.FileServer;
import com.geoway.nsapp.common.file.service.HuWeiObsFileService;
import com.geoway.nsapp.common.file.util.ObjectKeyUtil;
import com.obs.services.ObsClient;
import com.obs.services.model.*;
import lombok.extern.slf4j.Slf4j;
import net.coobird.thumbnailator.Thumbnails;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class HuweiObsFileServiceImpl implements HuWeiObsFileService {

    @Value("${project.uploadDir}")
    protected String workDir;

    @Autowired
    @Qualifier("redisTemplate")
    private RedisTemplate redisTemplate;

    private ObsClient _obsClient;

    private int _currentServerId = 0;

    @Override
    public ObsClient getOssClient(FileServer fileServer) throws Exception {
        if (fileServer == null) {
            return null;
        }
        String preKey = "fileServer-";
        boolean hasExist = redisTemplate.hasKey(preKey + fileServer.getId());
        if (_obsClient == null || !hasExist || fileServer.getId() != _currentServerId) {
            _currentServerId = fileServer.getId();
            //10分钟有效期
            redisTemplate.opsForValue().set(preKey + _currentServerId, _currentServerId + "", 10, TimeUnit.MINUTES);
            _obsClient = new ObsClient(fileServer.getEndpoint(), fileServer.getAppkey(), fileServer.getAppsecret());
        }

        return _obsClient;
    }

    @Override
    public FileStoreMeta sendFile(FileServer fileServer, MultipartFile file, String objectName) throws Exception {
        FileStoreMeta fileStoreMeta = new FileStoreMeta();
        // 获取文件全称
        String fileName = file.getOriginalFilename();
        // 上传文件类型
        String fileType = fileName.substring(fileName.lastIndexOf(".") + 1);


        String bucket = fileServer.getBucket();
        if (StrUtil.isBlank(objectName)) {
            objectName = ObjectKeyUtil.formatObjectName(fileName);
        }


        ObsClient obsClient = getOssClient(fileServer);
        boolean isExists = obsClient.headBucket(bucket);
        if (!isExists) {
            this.createBucket(obsClient, bucket, null);
        }
        obsClient.putObject(bucket, objectName, file.getInputStream());
        String url = fileServer.getEndpoint() + "/" + bucket + "/" + objectName;
        ObjectMetadata objMeta = obsClient.getObjectMetadata(bucket, objectName);

        fileStoreMeta.setAbsolutePath(url);
        fileStoreMeta.setRelPath(objectName);
        fileStoreMeta.setName(fileName);
        fileStoreMeta.setFileType(fileType);
        fileStoreMeta.setFileSize(objMeta.getContentLength());
        obsClient.close();

        return fileStoreMeta;
    }


    @Override
    public FileStoreMeta sendFile(FileServer fileServer, File file, String objectName) throws Exception {
        FileStoreMeta fileStoreMeta = new FileStoreMeta();
        String shortName = file.getName();
        String fileType = shortName.substring(shortName.lastIndexOf(".") + 1);

        String bucket = fileServer.getBucket();
        if (StrUtil.isBlank(objectName)) {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
            String foldName = simpleDateFormat.format(new Date());
            objectName = foldName + "/" + System.currentTimeMillis() + "/" + shortName;
        }

        ObsClient obsClient = getOssClient(fileServer);
        boolean isExists = obsClient.headBucket(bucket);
        if (!isExists) {
            this.createBucket(obsClient, bucket, null);
        }

        obsClient.putObject(bucket, objectName, file);
        obsClient.close();

        String url = fileServer.getEndpoint() + "/" + bucket + "/" + objectName;


        fileStoreMeta.setAbsolutePath(url);
        fileStoreMeta.setRelPath(objectName);
        fileStoreMeta.setName(shortName);
        fileStoreMeta.setFileType(fileType);
        fileStoreMeta.setFileSize(file.length());

        return fileStoreMeta;
    }

    @Override
    public FileStoreMeta buildThumbnailFile(FileServer fileServer, String objectName) throws Exception {
        FileStoreMeta fileStoreMeta = new FileStoreMeta();
        // 上传文件类型
        String fileType = objectName.substring(objectName.lastIndexOf(".") + 1).toLowerCase();
        String minObjectName = objectName.replace("." + fileType, "-min." + fileType);
        String fileName = minObjectName.substring(minObjectName.lastIndexOf("/") + 1);
        String thuamailFilePath = workDir + "/" + minObjectName;
        File thuamailFile = new File(thuamailFilePath);
        File pFile = thuamailFile.getParentFile();
        if (!pFile.exists()) {
            pFile.mkdirs();
        }
        String bucket = fileServer.getBucket();

        ObsClient obsClient = getOssClient(fileServer);
        ObsObject obsObject = obsClient.getObject(bucket, objectName);

        InputStream inputStream = obsObject.getObjectContent();
        inputStream.mark(0);

        String[] photoTypes = {"png", "jpg", "jpeg", "bmp"};
        boolean photoflag = Arrays.stream(photoTypes).anyMatch(type -> type.equalsIgnoreCase(fileType));
        if (photoflag) {
            BufferedImage bufferedImage = ImageIO.read(inputStream);
            int width = bufferedImage.getWidth();
            int height = bufferedImage.getHeight();
            int max = width > height ? width : height;
            int w = 0;
            int h = 0;
            if (max <= 100) {
                w = width;
                h = height;
            } else {
                double scale = 100.0 / max;
                w = (int) (width * scale);
                h = (int) (height * scale);
            }
            inputStream.reset();

            Thumbnails.of(inputStream).size(w, h).toFile(thuamailFile);
            obsObject.getObjectContent().close();
            obsClient.putObject(bucket, minObjectName, thuamailFile);

            inputStream.close();

            String url = fileServer.getEndpoint() + "/" + bucket + "/" + minObjectName;

            fileStoreMeta.setAbsolutePath(url);
            fileStoreMeta.setRelPath(objectName);
            fileStoreMeta.setName(thuamailFile.getName());
            fileStoreMeta.setFileSize(thuamailFile.length());
            fileStoreMeta.setFileType(fileType);
        }

        //处理PDF缩略图
        boolean pdfFlag = "pdf".equalsIgnoreCase(fileType);
        if (pdfFlag) {

        }
        //处理office缩略图
        String[] officeTypes = {"doc", "docx", "xls", "xlsx", "ppt", "pptx"};
        boolean officeFlag = Arrays.stream(officeTypes).anyMatch(type -> type.equalsIgnoreCase(fileType));
        if (officeFlag) {

        }
        //处理视频缩略图
        String[] videoTypes = {"mp4", "avi"};
        boolean videoFlag = Arrays.stream(videoTypes).anyMatch(type -> type.equalsIgnoreCase(fileType));
        if (videoFlag) {

        }


        return fileStoreMeta;
    }


    @Override
    public void deleteFile(FileServer fileServer, String bucket, String path) throws Exception {

        ObsClient obsClient = getOssClient(fileServer);
        obsClient.deleteObject(bucket, path);
    }

    @Override
    public String buildPreviewURL(FileServer fileServer, String objectName, Integer expires) throws Exception {
        ObsClient obsClient = getOssClient(fileServer);
        TemporarySignatureRequest request = new TemporarySignatureRequest();
        request.setBucketName(fileServer.getBucket());
        request.setObjectKey(objectName);
        request.setRequestDate(new Date());
        request.setExpires(expires);

        TemporarySignatureResponse signature = obsClient.createTemporarySignature(request);
        String url = signature.getSignedUrl();

        return url;
    }

    @Override
    public String getPreviewURL(FileServer fileServer, String url, Integer expires) throws Exception {
        // 截取objectName
        String objectName;
        try {
            URL urls = new URL(url);
            objectName = urls.getPath();
            String bucket = fileServer.getBucket();
            objectName = objectName.replace(bucket, "");

            while (objectName.startsWith("/")) {
                objectName = objectName.substring(1);
            }
        }catch (Exception e){
            objectName = url;
        }
        return buildPreviewURL(fileServer,objectName,expires);
    }

    @Override
    public byte[] createThumbnail(FileServer fileServer, String objectName) throws Exception {

        String bucket = fileServer.getBucket();
        ObsClient obsClient = getOssClient(fileServer);
        ObsObject obsObject = obsClient.getObject(bucket, objectName);

        InputStream inputStream = obsObject.getObjectContent();
        inputStream.mark(0);
        BufferedImage bufferedImage = ImageIO.read(inputStream);
        int width = bufferedImage.getWidth();
        int height = bufferedImage.getHeight();
        int max = width > height ? width : height;
        int w = 0;
        int h = 0;
        if (max <= 100) {
            w = width;
            h = height;
        } else {
            double scale = 100.0 / max;
            w = (int) (width * scale);
            h = (int) (height * scale);
        }
        inputStream.reset();

        ByteArrayOutputStream out = new ByteArrayOutputStream();
        Thumbnails.of(inputStream).size(w, h).toOutputStream(out);
        inputStream.close();

        byte[] thumnailBytes = out.toByteArray();
        out.close();

        return thumnailBytes;
    }

    @Override
    public FileDownloadMeta downLoadFile(FileServer fileServer, String objectName) throws Exception {
        FileDownloadMeta fileDownloadMeta = new FileDownloadMeta();

        ObsClient obsClient = getOssClient(fileServer);
        ObjectMetadata objectMetadata = obsClient.getObjectMetadata(fileServer.getBucket(), objectName);

        fileDownloadMeta.setFileSize(objectMetadata.getContentLength());
        ObsObject obsObject = obsClient.getObject(fileServer.getBucket(), objectName);
        fileDownloadMeta.setInputStream(obsObject.getObjectContent());

        return fileDownloadMeta;
    }

    private void createBucket(ObsClient obsClient, String bucketName, String location) {
        // 创建桶，并设置桶策略
        ObsBucket obsBucket = new ObsBucket();
        obsBucket.setBucketName(bucketName);
        // 设置桶访问权限为公共读，默认是私有读写
        obsBucket.setAcl(AccessControlList.REST_CANNED_PUBLIC_READ);
        // 设置桶的存储类型为标准存储
        obsBucket.setBucketStorageClass(StorageClassEnum.STANDARD);
        if (StrUtil.isNotBlank(location)) {
            obsBucket.setLocation(location);
        }
        // 创建桶
        obsClient.createBucket(obsBucket);
        // 设置桶策略
        String json =
                "{" + "\"Statement\":[" + "{" + "\"Sid\":\"为授权用户创建OBS使用的桶策略\"," + "\"Principal\":{\"ID\" : \"*\"},"
                        + "\"Effect\":\"Allow\"," + "\"Action\":[\"GetObject\",\"GetObjectVersion\"]," + "\"Resource\": [\""
                        + bucketName + "/*\"]" + "}" + "]}";
        obsClient.setBucketPolicy(bucketName, json);
    }
}
