package com.geoway.nsapp.common.file.service.impl;

import cn.hutool.core.util.StrUtil;
import com.geoway.nsapp.common.core.util.FileUtil;
import com.geoway.nsapp.common.file.dto.FileDownloadMeta;
import com.geoway.nsapp.common.file.dto.FileStoreMeta;
import com.geoway.nsapp.common.file.entity.FileServer;
import com.geoway.nsapp.common.file.service.MinIOFileService;
import com.geoway.nsapp.common.file.util.ObjectKeyUtil;
import io.minio.MinioClient;
import io.minio.ObjectStat;
import io.minio.policy.PolicyType;
import lombok.extern.slf4j.Slf4j;
import net.coobird.thumbnailator.Thumbnails;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.URL;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class MinIOFileServiceImpl implements MinIOFileService {
    @Autowired
    @Qualifier("redisTemplate")
    private RedisTemplate redisTemplate;

    private MinioClient _minioClient;

    private int _currentServerId = 0;

    @Override
    public MinioClient getOssClient(FileServer fileServer) {
        if (fileServer == null) {
            return null;
        }

        String preKey = "fileServer-";
        boolean hasExist = redisTemplate.hasKey(preKey + fileServer.getId());
        if (_minioClient == null || !hasExist || fileServer.getId() != _currentServerId) {
            _currentServerId = fileServer.getId();

            try{
                _minioClient = new MinioClient(fileServer.getEndpoint(), fileServer.getAppkey(), fileServer.getAppsecret());
            }catch (Exception ex){
                log.error("创建MinioClient失败");
            }

            redisTemplate.opsForValue().set(preKey + _currentServerId, _currentServerId + "", 10, TimeUnit.HOURS);
        }

        return _minioClient;
    }

    @Override
    public FileStoreMeta sendFile(FileServer fileServer, File file, String objectName) throws Exception {
        if (StrUtil.isBlank(objectName)) {
            objectName = ObjectKeyUtil.formatObjectName(file.getName());
        }

        String contentType = FileUtil.getContentType(file.getName());

        return sendFileStream(fileServer,file.getName(),objectName,new FileInputStream(file),file.length(),contentType);
    }

    @Override
    public FileStoreMeta sendFile(FileServer fileServer, MultipartFile multipartFile, String objectName) throws Exception {
        if (StrUtil.isBlank(objectName)) {
            objectName = ObjectKeyUtil.formatObjectName(multipartFile.getOriginalFilename());
        }

        String contentType = FileUtil.getContentType(multipartFile.getOriginalFilename());

        return sendFileStream(fileServer,multipartFile.getOriginalFilename(),objectName,multipartFile.getInputStream(),multipartFile.getSize(),contentType);
    }

    private FileStoreMeta sendFileStream(FileServer fileServer, String fileName, String objectName, InputStream inputStream, Long fileSize, String contentType) throws Exception {
        try {
            FileStoreMeta fileStoreMeta = new FileStoreMeta();
            String bucket = fileServer.getBucket();
            MinioClient minioClient = getOssClient(fileServer);
            boolean isExists = minioClient.bucketExists(bucket);
            if (!isExists) {
                minioClient.makeBucket(bucket);
                minioClient.setBucketPolicy(bucket, "*", PolicyType.READ_WRITE);
            }

            // 上传
            minioClient.putObject(bucket,objectName,inputStream,contentType);
            //
            String url = minioClient.getObjectUrl(bucket, objectName);
            fileStoreMeta.setAbsolutePath(url);
            fileStoreMeta.setRelPath(objectName);
            fileStoreMeta.setPreviewURL(buildPreviewURL(fileServer, objectName, 3600));
            return fileStoreMeta;
        } catch (Exception ex) {
            throw new Exception(ex);
        } finally {
            inputStream.close();
        }
    }

    @Override
    public FileStoreMeta buildThumbnailFile(FileServer fileServer, String objectName) throws Exception {
        FileStoreMeta fileStoreMeta = new FileStoreMeta();
        // 上传文件类型
        String fileType = objectName.substring(objectName.lastIndexOf(".") + 1).toLowerCase();
        String minObjectName = objectName.replace("." + fileType, "-min." + fileType);
        String fileName = minObjectName.substring(minObjectName.lastIndexOf("/") + 1);

        String bucket = fileServer.getBucket();

        MinioClient minioClient = getOssClient(fileServer);
        InputStream inputStream = minioClient.getObject(bucket, objectName);


        String[] photoTypes = {"png", "jpg", "jpeg", "bmp"};
        boolean photoflag = Arrays.stream(photoTypes).anyMatch(type -> type.equalsIgnoreCase(fileType));
        if (photoflag) {
            BufferedImage bufferedImage = ImageIO.read(inputStream);
            int width = bufferedImage.getWidth();
            int height = bufferedImage.getHeight();
            int max = width > height ? width : height;
            int w = 0;
            int h = 0;
            if (max <= 360) {
                w = width;
                h = height;
            } else {
                double scale = 360.0 / max;
                w = (int) (width * scale);
                h = (int) (height * scale);
            }
            inputStream.close();
            inputStream = minioClient.getObject(bucket, objectName);

            ByteArrayOutputStream out = new ByteArrayOutputStream();

            Thumbnails.of(inputStream).size(w, h).toOutputStream(out);
            inputStream.close();
            byte[] thumbBytes = out.toByteArray();
            long fileSize = thumbBytes.length;

            InputStream byteStream = new ByteArrayInputStream(thumbBytes);
            minioClient.putObject(bucket, minObjectName, byteStream, null);
            byteStream.close();
            out.close();

            String url = minioClient.getObjectUrl(bucket, objectName);
            fileStoreMeta.setAbsolutePath(url);
            fileStoreMeta.setRelPath(minObjectName);
            fileStoreMeta.setName(fileName);
            fileStoreMeta.setFileType(fileType);
        }

        //处理PDF缩略图
        boolean pdfFlag = "pdf".equalsIgnoreCase(fileType);
        if (pdfFlag) {

        }
        //处理office缩略图
        String[] officeTypes = {"doc", "docx", "xls", "xlsx", "ppt", "pptx"};
        boolean officeFlag = Arrays.stream(officeTypes).anyMatch(type -> type.equalsIgnoreCase(fileType));
        if (officeFlag) {

        }
        //处理视频缩略图
        String[] videoTypes = {"mp4", "avi"};
        boolean videoFlag = Arrays.stream(videoTypes).anyMatch(type -> type.equalsIgnoreCase(fileType));
        if (videoFlag) {

        }


        return fileStoreMeta;
    }

    @Override
    public void deleteFile(FileServer fileServer, String bucket, String path) throws Exception {

        MinioClient minioClient = getOssClient(fileServer);
        minioClient.removeObject(bucket, path);

    }

    @Override
    public String buildPreviewURL(FileServer fileServer, String objectName, Integer expires) throws Exception {
//        MinioClient minioClient = this.getOssClient(fileServer);
//        ObjectStat objectStat = minioClient.statObject(fileServer.getBucket(), objectName);
//        String url = minioClient.presignedGetObject(fileServer.getBucket(), objectName, expires);
        // 返回地址处理
        MinioClient minioClient = getOssClient(fileServer);
        return minioClient.presignedGetObject(fileServer.getBucket(), objectName, expires);
//        if (url.contains(fileServer.getEndpoint())) {
//            url = url.replace(fileServer.getEndpoint(), fileServer.getPreviewUrl());
//        }
//        return url;
    }

    @Override
    public String getPreviewURL(FileServer fileServer,String url,Integer expires) throws Exception {
        // 截取objectName
        String objectName;
        try {
            URL urls = new URL(url);
            objectName = urls.getPath();
            String bucket = fileServer.getBucket();
            objectName = objectName.replace(bucket, "");
            while (objectName.startsWith("/")) {
                objectName = objectName.substring(1);
            }
        }catch (Exception e){
            objectName = url;
        }
        return buildPreviewURL(fileServer,objectName,expires);
    }

    @Override
    public byte[] createThumbnail(FileServer fileServer, String objectName) throws Exception {


        String bucket = fileServer.getBucket();
        MinioClient minioClient = getOssClient(fileServer);
        InputStream inputStream = minioClient.getObject(bucket, objectName);
        BufferedImage bufferedImage = ImageIO.read(inputStream);
        int width = bufferedImage.getWidth();
        int height = bufferedImage.getHeight();
        int max = width > height ? width : height;
        int w = 0;
        int h = 0;
        if (max <= 300) {
            w = width;
            h = height;
        } else {
            double scale = 300.0 / max;
            w = (int) (width * scale);
            h = (int) (height * scale);
        }
        inputStream.close();
        inputStream = minioClient.getObject(bucket, objectName);

        ByteArrayOutputStream out = new ByteArrayOutputStream();

        Thumbnails.of(inputStream).size(w, h).toOutputStream(out);
        inputStream.close();


        byte[] thumnailBytes = out.toByteArray();
        out.close();

        return thumnailBytes;
    }

    @Override
    public FileDownloadMeta downLoadFile(FileServer fileServer, String objectName) throws Exception {
        FileDownloadMeta fileDownloadMeta = new FileDownloadMeta();

        MinioClient minioClient = getOssClient(fileServer);
        ObjectStat objStat = minioClient.statObject(fileServer.getBucket(), objectName);

        fileDownloadMeta.setFileSize(objStat.length());
        InputStream inputStream = minioClient.getObject(fileServer.getBucket(), objectName);
        fileDownloadMeta.setInputStream(inputStream);

        return fileDownloadMeta;
    }
}