package com.northpool.commons.filechannel;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.CompletionHandler;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

import com.northpool.exception.UException;

public class AsyncFileReadChannelPool {
	
	final Path filePath;
	
	final int readFileChannelNum;
	
	BlockingQueue<DataRequest> queue;
	
	final int queueSize;
	
	FileChannelHolder[] pool;
	
	Executor executor;
	
	boolean isClose = false;
	
	final ReentrantLock lock = new ReentrantLock();
	
	final Condition allFileChannelInUse = lock.newCondition();
	
    boolean useDirect = true;
    
	
	public AsyncFileReadChannelPool(Path filePath,int readFileChannelNum,int queueSize){
		this.filePath = filePath;
		this.readFileChannelNum = readFileChannelNum;
		this.queueSize = queueSize;
	
	}
	
	public void init() throws IOException{
		this.createFileChannelPool();
		this.queue = new ArrayBlockingQueue<DataRequest>(queueSize);
		this.executor = Executors.newSingleThreadExecutor();
		this.executor.execute(() -> {
			while(!isClose){
				DataRequest dataRequest = null;
				try {
					dataRequest = this.queue.take();
				} catch (InterruptedException e) {
					// TODO Auto-generated catch block
					UException.printStackTrace(e);
					//不用需要截获异常
				}
				lock.lock();
				FileChannelHolder holder = this.getUnusedFileChannelPool();
				this.doRead(holder, dataRequest);
				lock.unlock();
			}
		});
	}
	
	private void doRead(FileChannelHolder holder,DataRequest dataRequest){
		holder.using = true;
		ByteBuffer buffer;
		if(useDirect){
			buffer = ByteBuffer.allocateDirect(dataRequest.lenght);
		}else{
			buffer = ByteBuffer.allocate(dataRequest.lenght);
		}
		holder.channel.read(buffer, dataRequest.offset,buffer,new CompletionHandler<Integer, ByteBuffer>() {
		    @Override
		    public void completed(Integer result, ByteBuffer attachment) {
		    	lock.lock();
		    	dataRequest.buffer = attachment;
		    	dataRequest.future.complete(dataRequest);
		    	holder.using = false;
		    	allFileChannelInUse.signalAll();
		    	lock.unlock();
		    }

		    @Override
		    public void failed(Throwable exc, ByteBuffer attachment) {
		    	lock.lock();
		    	dataRequest.future.completeExceptionally(exc);
		    	holder.using = false;
		    	allFileChannelInUse.signalAll();
		    	lock.unlock();
		    }
		});
	}
	
	
	
	private FileChannelHolder getUnusedFileChannelPool() {
		FileChannelHolder holderUnused = null;
		while (holderUnused == null) {
			for (FileChannelHolder holder : this.pool) {
				if (!holder.using) {
					holderUnused = holder;
					return holder;
				}
			}
			try {
				allFileChannelInUse.await();
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				//不用需要截获异常
			}
		}
		return holderUnused;
	}
	

	void createFileChannelPool() throws IOException{
		this.pool = new FileChannelHolder[this.readFileChannelNum];
		for(int i = 0 ; i < this.readFileChannelNum ; i ++){
			AsynchronousFileChannel channel = AsynchronousFileChannel.open(filePath, new StandardOpenOption[]{StandardOpenOption.READ});
			this.pool[i] = FileChannelHolder.create(channel);
		}
	}
	
	
	public void close(){
		for(int i = 0 ; i < this.readFileChannelNum ; i ++){
			try {
				this.pool[i].channel.close();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				UException.printStackTrace(e);
				//不需要截获
			}
		}
		this.isClose = true;
	}
	
	
	public CompletableFuture<DataRequest> read(String key,long offset,int lenght) throws InterruptedException{
		DataRequest dataRequest = DataRequest.build(key,offset, lenght);
		
		queue.put(dataRequest);
		
		return dataRequest.future;
	}
	
	
	public static class DataRequest{
		long offset;
		int lenght;
		ByteBuffer buffer;
		String key;
		CompletableFuture<DataRequest> future;
		static DataRequest build(String key,long offset,int lenght){
			DataRequest dataRequest = new DataRequest();
			dataRequest.offset = offset;
			dataRequest.lenght = lenght;
			
			CompletableFuture<DataRequest> future = new CompletableFuture<>();
			dataRequest.future = future;
			//CompletableFuture<ByteBuffer> future = new CompletableFuture<>();
			//dataRequest.future = future;
			dataRequest.key = key;
			return dataRequest;
		}
		public long getOffset() {
			return offset;
		}
		public int getLenght() {
			return lenght;
		}
		public ByteBuffer getBuffer() {
			return buffer;
		}
		public String getKey() {
			return key;
		}
		
		
		
	}
	
	static class FileChannelHolder{
		
		volatile boolean using = false;
		AsynchronousFileChannel channel;	
		static FileChannelHolder create(AsynchronousFileChannel channel){
			FileChannelHolder holder = new FileChannelHolder();
			holder.channel = channel;
			return holder;
		}
		
		
	}
	
}
