package com.northpool.spatial.grid.impl;

import com.northpool.spatial.grid.Constants.GRID_UNIT;
import com.northpool.spatial.grid.Grid;
import com.northpool.spatial.grid.extent.GridExtent;
import com.northpool.spatial.grid.extent.impl.GridExtentImpl;
import com.northpool.spatial.grid.impl.quadtreegrid.TdtDegrees512;
import org.locationtech.proj4j.CRSFactory;
import org.locationtech.proj4j.CoordinateReferenceSystem;

import java.util.ArrayList;
import java.util.List;

public abstract class AbstractGrid implements Grid {

    /**
     * 最高支持切片等级，等级越高，单个瓦片内包含要素越少，切片消耗存储资源越高
     */
    protected int MAX_TILE_LEVEL = 30;
    
	protected GRID_UNIT unit;
	
	protected String name;
	@Override
	public GRID_UNIT getUnit() {
		return unit;
	}

	protected int base;

	protected double resolutionLevelBegin;

	protected double minX;

	protected double minY;

	protected double maxX;

	protected double maxY;

	protected int beginLevel;
	
	protected int maxLevel;
	
	@Deprecated
	public int getBase() {
		return this.base;
	}
	static final CRSFactory csFactory = new CRSFactory();
	static final String cgcs2000DegreeProj = "+proj=longlat +ellps=GRS80 +no_defs";
	static final String webMercProj = "+proj=merc +a=6378137 +b=6378137 +lat_ts=0.0 +lon_0=0.0 +x_0=0.0 +y_0=0 +k=1.0 +units=m +nadgrids=@null +wktext +no_defs";
	static final CoordinateReferenceSystem cs_cgcs2000Degree = csFactory.createFromParameters("cgcs2000Degree", cgcs2000DegreeProj);
	static final CoordinateReferenceSystem cs_webMerc = csFactory.createFromParameters("webMerc", webMercProj);
	
	
	@Deprecated
	public GridExtent getExtent(int level, int x, int y) {
		double resolution = this.getResolution(level);// 获得瓦片分辨率
		double standard = this.base * resolution;// 计算一张瓦片包含的经纬度
		// 计算瓦片空间范围
		double left = this.minX + (x * standard);
		double right = this.minX + ((x + 1) * standard);
		double top = this.maxY - (y * standard);
		double bottom = this.maxY - ((y + 1) * standard);
		// 返回一个计算好的瓦片对象
		return new GridExtentImpl(left, bottom, right, top, level, this, x, y);
	}
	
	
	@Override
	public GridExtent getGridExtent(int level, int x, int y) {
		Double resolution = this.getResolution(level);// 获得瓦片分辨率
		
		if(resolution == null){
            throw new RuntimeException("网格没有层级" + level);
        }
		
		double standard = this.base * resolution;// 计算一张瓦片包含的经纬度
		// 计算瓦片空间范围
		double left = this.minX + (x * standard);
		double right = this.minX + ((x + 1) * standard);
		double top;
		double bottom;
		
		if(this.unit == GRID_UNIT.pixel){
            bottom = this.minY + ((y + 1) * standard);
            top = this.minY + ((y) * standard);
        }else {
            top = this.maxY - (y * standard);
            bottom = this.maxY - ((y + 1) * standard);
        }

		
		// 返回一个计算好的瓦片对象
		return new GridExtentImpl(left, bottom, right, top, level, this, x, y);
	}
	
	@Override
	public int getBeginLevel() {
		return this.beginLevel;
	}

	
	public GridExtent getExtentByXY(int level,double x,double y){
		
		x = x < this.minX ? this.minX : x;
		y = y > this.maxY ? this.maxY : y;

		x = x > this.maxX ? this.maxX : x;
		y = y < this.minY ? this.minY : y;

		double resolution = this.getResolution(level);
		int extentX = (int) this.calculateX(resolution, x);
		int extentY = (int) this.calculateY(resolution, y);
		return this.getGridExtent(level, extentX, extentY);
	}
	
	
	public GridExtent getGridExtentByLonLat(int level,double  lon,double  lat){
		return this.getExtentByXY(level, lon, lat);
	}
	

	public String getName() {
		return name;
	}

	public double getBeginResolution() {
		return this.resolutionLevelBegin;
	}
	
	
	@Override
	public int getBaseTileSize(){
        return this.base;
    }
	
	
	
	@Override
	public double getMinX() {
		return minX;
	}
	@Override
	public double getMinY() {
		return minY;
	}
	@Override
	public double getMaxX() {
		return maxX;
	}
	@Override
	public double getMaxY() {
		return maxY;
	}
	@Override
	public double calculateX(double resolution, double x) {
		return (x - this.getMinX()) / resolution / this.base;
	}
	@Override
	public double calculateY(double resolution, double y) {
		 if(this.unit == GRID_UNIT.pixel){
            return (y - this.minY) / resolution / this.base;
        }
		return (this.getMaxY() - y) / resolution / this.getBase();
	}
	@Override
	public String getBBox(){
		return new StringBuilder().append(this.minX).append(',').append(this.minY).append(',').append(this.maxX).append(',').append(this.maxY).toString();
	}
	
	public int getFitableBeginLevel(double[] bbox) {
		double dx_bbox = bbox[2] - bbox[0];
		double dy_bbox = bbox[3] - bbox[1];
		if (dx_bbox < 0 || dy_bbox < 0) {
		    return -1;
		}
		double mark_bbox = Math.max(dy_bbox, dx_bbox);
		double resolution = mark_bbox / this.base;
		int level = this.getBeginLevel();
		while (true) {
			double resolutionForLevel = this.getResolution(level);
			if (resolutionForLevel < resolution) {
				break;
			}
			level++;
			if(level > MAX_TILE_LEVEL) {
			    return -2;
			}
		}
		return level - 1;
	}

	/*public GridExtent getGridExtentByLonLat(int level,  double lon,  double lat){
        lon = lon < this.minX ? this.minX : lon;
        lat = lat > this.maxY ? this.maxY : lat;
        
        lon = lon > this.maxX ? this.maxX : lon;
        lat = lat < this.minY ? this.minY : lat;
        
        double resolution = this.getResolution(level);
        
        double standard = this.base * resolution;
        int x = (int)((lon - this.minX) / standard);
        int y;
        if(this.unit == GRID_UNIT.pixel) {
            y = (int)((lat - this.minY) / standard);
        }else{
            y = (int)((this.maxY - lat) / standard);
        }
        return this.getExtentByXY(level, x, y)
    }*/
	
	
	@Override
    public void setMinX(int minX){
        this.minX = minX;
    }
	@Override
    public void setMaxY(int maxY){
        this.maxY = maxY;
    }
	
	
	@Override
	public String getPointToTileString(double x,double y,int level){
		double resolution = this.getResolution(level);
		Double _x = (Double)(x - this.minX) / (resolution * this.base) ;
		Double _y = (Double)(this.maxY - y) / (resolution * this.base);
		return _x.intValue() + "_" + _y.intValue() + "_" + level;
	}
	
	@Override
	public GridExtent getPointToTile(double x,double y,int level){
		double resolution = this.getResolution(level);
		Double _x = (Double)(x - this.minX) / (resolution * this.base) ;
		Double _y = (Double)(this.maxY - y) / (resolution * this.base);
		return this.getGridExtent(level, _x.intValue(), _y.intValue());
	}
	
	
	
	
	
    public static void main(String[] aaa){
//        TdtMeter256 tdtDegree256 = new TdtMeter256();
//        
//        double[] bbox = new double[]{8414187.9296875,4285365.48046875,8433755.80859375,4304933.359375};
//        System.out.println(tdtDegree256.getFitableBeginLevel(bbox));
        
        TdtDegrees512 tdtDegree512 = new TdtDegrees512();
        GridExtent gridExtent = tdtDegree512.getExtentByXY(9, 118.21702680400006, 27.83354398000027);
        System.out.println(gridExtent.getX() + " " + gridExtent.getY());
//        
//        
//        System.out.println(tdtDegree256.getExtentByXY(11, 8432451.427589528, 4300621.372043789).getBBOX());


//        TdtDegree512 tdtDegree512 = new TdtDegree512();
//        System.out.println(tdtDegree512.getExtentByXY(4, 109.17450717229161, 19.421138733812654));
        
     //  System.out.print(tdtDegree256.getExtentByXY(17, 75.75, 36));
    }
	
    
    public List<GridExtent> getGridExtentsByBBox(int level,double minX,double minY,double maxX,double maxY){
    	List<GridExtent> extents = new ArrayList<>();

    	GridExtent leftBottomExtent;
    	GridExtent rightTopExtent;

        if(this.unit == GRID_UNIT.pixel){
            leftBottomExtent = this.getGridExtentByLonLat(level,minX,maxY);
            rightTopExtent = this.getGridExtentByLonLat(level,maxX,minY);
        }else{
            leftBottomExtent = this.getGridExtentByLonLat(level,minX,minY);
            rightTopExtent = this.getGridExtentByLonLat(level,maxX,maxY);
        }

        int minExtentX;
        int maxExtentY;
        int maxExtentX;
        int minExtentY;

        minExtentX = leftBottomExtent.getX();
        maxExtentY = leftBottomExtent.getY();
        maxExtentX = rightTopExtent.getX();
        minExtentY = rightTopExtent.getY();

        int currentX;
        int currentY;
        for(int extentY = minExtentY ;extentY <= maxExtentY ; extentY ++){
            for(int extentX = minExtentX ; extentX <= maxExtentX ; extentX ++){
                currentX = extentX;
                currentY = extentY;
                GridExtent currentExtent = this.getGridExtent(level,currentX,currentY);
                extents.add(currentExtent);
            }
        }
        return extents;
    }

    public CoordinateReferenceSystem getProj(){
    	if(this.unit == GRID_UNIT.pixel){
    		throw new RuntimeException("pixel没有proj");
    	}
    	if(this.unit == GRID_UNIT.degree){
    		return cs_cgcs2000Degree;
    	}
    	if(this.unit == GRID_UNIT.meter){
    		return cs_webMerc;
    	}
    	throw new RuntimeException("不支持 " + this.unit + " 获得proj");
    }

    
    

}
