/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2002-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package com.northpool.spatial.shape;


/**
 * Not much but a type safe enumeration of file types as ints and names. The descriptions can easily
 * be tied to a ResourceBundle if someone wants to do that.
 *
 * @author Ian Schneider
 */
public final class ShapeType {

    /** Represents a Null shape (id = 0). */
    public static final ShapeType NULL = new ShapeType(0, "Null",false);
    /** Represents a Point shape (id = 1). */
    public static final ShapeType POINT = new ShapeType(1, "Point",false);
    /** Represents a PointZ shape (id = 11). */
    public static final ShapeType POINTZ = new ShapeType(11, "PointZ",true);
    /** Represents a PointM shape (id = 21). */
    public static final ShapeType POINTM = new ShapeType(21, "PointM",false);
    /** Represents an Arc shape (id = 3). */
    public static final ShapeType POLYLINE= new ShapeType(3, "Polyline",false);
    /** Represents an ArcZ shape (id = 13). */
    public static final ShapeType POLYLINEZ = new ShapeType(13, "PolylineZ",true);
    /** Represents an ArcM shape (id = 23). */
    public static final ShapeType POLYLINEM = new ShapeType(23, "PolylineM",false);
    /** Represents a Polygon shape (id = 5). */
    public static final ShapeType POLYGON = new ShapeType(5, "Polygon",false);
    /** Represents a PolygonZ shape (id = 15). */
    public static final ShapeType POLYGONZ = new ShapeType(15, "PolygonZ",true);
    /** Represents a PolygonM shape (id = 25). */
    public static final ShapeType POLYGONM = new ShapeType(25, "PolygonM",false);
    /** Represents a MultiPoint shape (id = 8). */
    public static final ShapeType MULTIPOINT = new ShapeType(8, "MultiPoint",false);
    /** Represents a MultiPointZ shape (id = 18). */
    public static final ShapeType MULTIPOINTZ = new ShapeType(18, "MultiPointZ",true);
    /** Represents a MultiPointM shape (id = 28). */
    public static final ShapeType MULTIPOINTM = new ShapeType(28, "MultiPointM",false);

    /** Represents an Undefined shape (id = -1). */
    public static final ShapeType UNDEFINED = new ShapeType(-1, "Undefined",false);

    /** The integer id of this ShapeType. */
    public final int id;
    /**
     * The human-readable name for this ShapeType.<br>
     * Could easily use ResourceBundle for internationialization.
     */
    public final String name;
    
    private boolean hasZ = false;
    
    public boolean isHasZ() {
		return hasZ;
	}

	private boolean hasM = false;
    
    /**
     * Creates a new instance of ShapeType. Hidden on purpose.
     *
     * @param id The id.
     * @param name The name.
     */
    protected ShapeType(int id, String name,boolean hasZ) {
        this.id = id;
        this.name = name;
        this.hasZ = hasZ;
    }

    /**
     * Get the name of this ShapeType.
     *
     * @return The name.
     */
    public String toString() {
        return name;
    }

    /**
     * Is this a multipoint shape? Hint- all shapes are multipoint except NULL, UNDEFINED, and the
     * POINTs.
     *
     * @return true if multipoint, false otherwise.
     */
    public boolean isMultiPoint() {
        boolean mp = true;
        if (this == UNDEFINED) {
            mp = false;
        } else if (this == NULL) {
            mp = false;
        } else if (this == POINT || this == POINTM || this == POINTZ) {
            mp = false;
        }
        return mp;
    }

    public boolean isPointType() {
        return id % 10 == 1;
    }

    public boolean isLineType() {
        return id % 10 == 3;
    }

    public boolean isPolygonType() {
        return id % 10 == 5;
    }

    public boolean isMultiPointType() {
        return id % 10 == 8;
    }

    /**
     * Determine the ShapeType for the id.
     *
     * @param id The id to search for.
     * @return The ShapeType for the id.
     */
    public static ShapeType forID(int id) {
        ShapeType t;
        switch (id) {
            case 0:
                t = NULL;
                break;
            case 1:
                t = POINT;
                break;
            case 11:
                t = POINTZ;
                break;
            case 21:
                t = POINTM;
                break;
            case 3:
                t = POLYLINE;
                break;
            case 13:
                t = POLYLINEZ;
                break;
            case 23:
                t = POLYLINEM;
                break;
            case 5:
                t = POLYGON;
                break;
            case 15:
                t = POLYGONZ;
                break;
            case 25:
                t = POLYGONM;
                break;
            case 8:
                t = MULTIPOINT;
                break;
            case 18:
                t = MULTIPOINTZ;
                break;
            case 28:
                t = MULTIPOINTM;
                break;
            default:
                t = UNDEFINED;
                break;
        }
        return t;
    }

   
}
