
package com.northpool.spatial;

import java.io.ByteArrayInputStream;
import java.io.IOException;

import org.locationtech.jts.io.ByteOrderValues;




/**
 * Allows reading a stream of Java primitive datatypes from an underlying
 * {@link InStream}, with the representation being in either common byte
 * ordering.
 */
public class ByteOrderDataInStream {
	private int byteOrder = ByteOrderValues.BIG_ENDIAN;
	private ByteArrayInputStream stream;
	boolean hex = false;
	// buffers to hold primitive datatypes
	private byte[] buf1 ;
	private byte[] buf4 ;
	private byte[] buf8 ;

	public ByteOrderDataInStream() {	    
		this(null,false);
	}

	public ByteOrderDataInStream(ByteArrayInputStream stream) {
		this(stream,false);
	}
	
	public ByteOrderDataInStream(ByteArrayInputStream stream,boolean hex) {
		this.stream = stream;
		this.hex = hex;
		this.buf1 = new byte[1 * (hex == true ? 2 : 1)];
		this.buf4 = new byte[4 * (hex == true ? 2 : 1)];
		this.buf8 = new byte[8 * (hex == true ? 2 : 1)];
		
		
		
	}

	/**
	 * Allows a single ByteOrderDataInStream to be reused on multiple InStreams.
	 *
	 * @param stream
	 */
	public void setInStream(ByteArrayInputStream stream) {
		this.stream = stream;
	}

	public void setOrder(int byteOrder) {
	  
		this.byteOrder = byteOrder;
	}

	/**
	 * Reads a byte value
	 *
	 * @return the byte read
	 */
	public byte readByte() throws IOException {
		stream.read(buf1);
		if(this.hex){
			ByteOrderDataInStream.decodeHex(buf1, 1);
		}
		return buf1[0];
	}
	
	

	
	
	public byte[] readByte4() throws IOException{
		stream.read(buf4);
		if(this.hex){
			ByteOrderDataInStream.decodeHex(buf4, 1);
		}
		return buf4;
	}

	public int readInt() throws IOException {
		stream.read(buf4);
		if(this.hex){
			ByteOrderDataInStream.decodeHex(buf4, 4);
		}
		return ByteOrderValues.getInt(buf4, byteOrder);
	}

	public long readLong() throws IOException {
		stream.read(buf8);
		if(this.hex){
			ByteOrderDataInStream.decodeHex(buf8, 8);
		}
		return ByteOrderValues.getLong(buf8, byteOrder);
	}

	public double readDouble() throws IOException {
		stream.read(buf8);
		if(this.hex){
			ByteOrderDataInStream.decodeHex(buf8, 8);
		}
		return ByteOrderValues.getDouble(buf8, byteOrder);
	}
	
	
	
	public static void decodeHex(byte[] buffer, int nBytes) {
		if (buffer.length < nBytes * 2){
			throw new IllegalArgumentException("byte array length invalid.");
		}
		for (int i = 0; i < nBytes; i++) {
			buffer[i] = (byte) (((unhex(buffer[2 * i]) & 0xFF) << 4)
					+ (unhex(buffer[2 * i + 1]) & 0xFF) - 256);
		}
	}

	// char to byte
	public static byte unhex(byte hexC) {
		char c = (char)hexC;
		if (c >= '0' && c <= '9') {
			return (byte) (c - '0');
		} else if (c >= 'A' && c <= 'F') {
			return (byte) (c - 'A' + 10);
		} else if (c >= 'a' && c <= 'f') {
			return (byte) (c - 'a' + 10);
		} else {
			throw new IllegalArgumentException("Not valid Hex char " + c);
		}
	}
	
	

}
