package com.northpool.spatial.geofeature;

import java.util.ArrayList;

import org.locationtech.jts.algorithm.RobustDeterminant;
import org.locationtech.jts.geom.Geometry;

import com.northpool.commons.util.BuilderCreator;
import com.northpool.commons.util.DoubleBuilder;
import com.northpool.spatial.Constants.GEO_TYPE;
import com.northpool.spatial.geofeature.GeoPart.RING_TYPE;
import com.northpool.spatial.wkt.WktEncoder;

public class GeoBuffer {
	GEO_TYPE geoType;
	private ArrayList<GeoPart> partList = new ArrayList<GeoPart>();                                                                                                                                         ;
    int srid = 0;
    int dimension = 2;
	
	private double x = Double.NaN;
	private double y = Double.NaN;
	private double z = Double.NaN;
	
	private double minX = Double.NaN;
	private double minY = Double.NaN;
	private double maxX = Double.NaN;
	private double maxY = Double.NaN;
	
	
	public void addPoint(double x,double y){
		if(this.dimension == 2){
			this.addPoint(x, y, Double.NaN);
		}else{
			this.addPoint(x, y, 0);
		}
		
	}
	
	public void addPoint(double x,double y,double z){
/*	    if(this.geoType == GEO_TYPE.POINT){
	        this.x = x;
	        this.y = y;
	        this.z = z;
	    }*/
		if(this.geoType != GEO_TYPE.POINT && this.geoType != GEO_TYPE.MULTIPOINT){
			return;
		}
	    if (this.partList == null || this.partList.isEmpty()){
	    	DoubleBuilder doubleBuilder = BuilderCreator.createDouble(dimension);
	    	doubleBuilder.append(x);
	    	doubleBuilder.append(y);
	    	if (this.dimension == 3){
	    		doubleBuilder.append(z);
			}
			GeoPart geoPart = new GeoPart(doubleBuilder,this);
			this.partList.add(geoPart);
		}else {
			GeoPart geoPart = this.partList.get(0);
			DoubleBuilder doubleBuilder = geoPart.getDoubleBuilder();
			doubleBuilder.append(x);
			doubleBuilder.append(y);
			if (this.dimension == 3){
				doubleBuilder.append(z);
			}
		}
	}


	public GeoBuffer dimensionTo2(){
	    if(this.dimension == 2){
	        return this;
	    }
	    GeoBuffer newBuffer = new GeoBuffer(this.geoType,this.srid ,2);
	    newBuffer.maxX = this.maxX;
	    newBuffer.minX = this.minX;
	    newBuffer.maxY = this.maxY;
	    newBuffer.minY = this.minY;
	    newBuffer.x = this.x;
	    newBuffer.y = this.y;
	    newBuffer.z = Double.NaN;
	    ArrayList<GeoPart> partList = new ArrayList<GeoPart>(this.partList.size());
	    for(GeoPart part : this.partList){
	        int size = part.doubleBuilder.size() / 3;
	        DoubleBuilder doubleBuilder = BuilderCreator.createDouble(size * 2);
	        for(int i = 0 ; i < size ; i ++){
	            doubleBuilder.append(part.doubleBuilder.get(i * 3));
	            doubleBuilder.append(part.doubleBuilder.get(i * 3 + 1));
	        }
	        GeoPart newPart = new GeoPart(doubleBuilder,part.minX,part.minY,part.maxX,part.maxY,newBuffer,part.ringType);
	        partList.add(newPart);
	    }
	    newBuffer.partList = partList;
	    return newBuffer;
	    
	}
	
	public GeoBuffer dimensionTo3(){
	    if(this.dimension == 3){
            return this;
        }
        GeoBuffer newBuffer = new GeoBuffer(this.geoType,this.srid ,3);
        newBuffer.maxX = this.maxX;
        newBuffer.minX = this.minX;
        newBuffer.maxY = this.maxY;
        newBuffer.minY = this.minY;
        newBuffer.x = this.x;
        newBuffer.y = this.y;
        newBuffer.z = 0.0;
        ArrayList<GeoPart> partList = new ArrayList<GeoPart>(this.partList.size());
        for(GeoPart part : partList){
            int size = part.doubleBuilder.size() / 2;
            DoubleBuilder doubleBuilder = BuilderCreator.createDouble(size * 3);
            for(int i = 0 ; i < size ; i ++){
                doubleBuilder.append(part.doubleBuilder.get(i * 3));
                doubleBuilder.append(part.doubleBuilder.get(i * 3 + 1));
                doubleBuilder.append(0.0);
            }
            GeoPart newPart = new GeoPart(doubleBuilder,part.minX,part.minY,part.maxX,part.maxY,newBuffer,part.ringType);
            partList.add(newPart);
        }
        return newBuffer;
    }
	
	public void setBBOX(double minX,double minY,double maxX,double maxY){
	    this.minX = minX;
	    this.minY = minY;
	    this.maxX = maxX;
	    this.maxY = maxY;
	}
	
	
	public double[] zInterval(){
	    double minZ = Double.NaN;
        double maxZ = Double.NaN;
        if(geoType == GEO_TYPE.POINT){
            minZ = maxZ = this.getZ();
        }else{
            int i = 0 ;
            if(partList.size() == 1){
                return partList.get(0).zInterval();
            }
            for(GeoPart part : partList){
                double[] subZInterval = part.zInterval();
                if(i == 0){
                    minZ = subZInterval[0];
                    maxZ = subZInterval[1];
                    
                }else{
                    if(subZInterval[0] < minZ){
                        minZ = subZInterval[0];
                    }
                    if(subZInterval[1] > maxZ){
                        maxZ = subZInterval[1];
                    }
                }
                i++;
            }
        }
        return new double[]{minZ,maxZ}; 
	}
	
	public double getX() {
	    if(this.geoType != GEO_TYPE.POINT || this.partList == null || this.partList.isEmpty()){
	        return Double.NaN;
	    }
        return this.partList.get(0).getDoubleBuilder().get(0);
    }

    public double getY() {
		if(this.geoType != GEO_TYPE.POINT || this.partList == null || this.partList.isEmpty()){
			return Double.NaN;
		}
		return this.partList.get(0).getDoubleBuilder().get(1);
    }

    public double getZ() {
		if(this.geoType != GEO_TYPE.POINT || this.dimension == 2 || this.partList == null || this.partList.isEmpty()){
			return Double.NaN;
		}
		return this.partList.get(0).getDoubleBuilder().get(2);
    }
	
	public int getCoordinateCount(){
	    int count = 0;
	    if(this.geoType == GEO_TYPE.POINT){
	        return 1;
	    }else{
	        for(int i = 0 ; i < this.getPartList().size() ; i ++){
	            GeoPart part = this.getPartList().get(i);
	            count = count + part.getCoordinateCount();
	        }
	    }
	    return count;
	}
	
	
	
	public double[] getBBOX(){
	   // throw new RuntimeException("有问题");
	    
	    //double minX = Double.NaN;
	    //double minY = Double.NaN;
	    //double maxX = Double.NaN;
	    //double maxY = Double.NaN;
	    if(Double.isNaN(this.minX)){
    	    if(geoType == GEO_TYPE.POINT){
    	        minX = maxX = this.getX();
    	        minY = maxY = this.getY();
    	    }else{
    	        int i = 0 ;
    	        if(partList.size() == 1){
    	            return partList.get(0).getBBOX();
    	        }
    	        for(GeoPart part : partList){
    	            if(part.getRingType() == RING_TYPE.inside){
    	                continue;
    	            }
    	            double[] subBBOX = part.getBBOX();
    	            if(i == 0){
    	                minX = subBBOX[0];
    	                minY = subBBOX[1];
    	                maxX = subBBOX[2];
    	                maxY = subBBOX[3];
    	            }else{
    	                if(subBBOX[0] < minX){
    	                    minX = subBBOX[0];
    	                }
    	                if(subBBOX[1] < minY){
    	                    minY = subBBOX[1];
    	                }
    	                if(subBBOX[2] > maxX){
    	                    maxX = subBBOX[2];
                        }
    	                if(subBBOX[3] > maxY){
    	                    maxY = subBBOX[3];
                        }
    	            }
    	            i++;
    	        }
    	    }
	    }
		return new double[]{this.minX,this.minY,this.maxX,this.maxY}; 
	}
	
	public Integer getPartSize(){
		return this.partList.size();
	}
	
	public double[] getPoint(){
		if (this.partList == null || this.partList.isEmpty()){
			return null;
		}
		GeoPart part = this.partList.get(0);
		DoubleBuilder doubleBuilder = part.getDoubleBuilder();
		if (dimension == 3){
			return new double[]{doubleBuilder.get(0),doubleBuilder.get(1),doubleBuilder.get(2)};
		}else {
			return new double[]{doubleBuilder.get(0),doubleBuilder.get(1)};
		}
	}

	public double[] getPoint(Integer index){
		if (this.partList == null || this.partList.isEmpty()){
			return null;
		}
		GeoPart part = this.partList.get(0);
		DoubleBuilder doubleBuilder = part.getDoubleBuilder();
		if (dimension == 3){
			return new double[]{doubleBuilder.get(index * 3),doubleBuilder.get(index * 3 + 1),doubleBuilder.get(index * 3 + 2)};
		}else {
			return new double[]{doubleBuilder.get(index * 2),doubleBuilder.get(index * 2 + 1)};
		}
	}
	
	public String toWkt(){
	    return WktEncoder.ENCODER.fromGeoBuffer(this);
	}
	
	public String toString(){
	    return this.toWkt();
	}
	
	public DoubleBuilder getData(Integer index){
		GeoPart part = this.partList.get(index);
		if(part == null){
			return null;
		}else{
			return part.getDoubleBuilder();
		}
	}
	
	public GeoPart getPart(Integer index){
		return this.partList.get(index);
	}
	
	public void tryMultiToSingle(){
		int partSize = this.partList.size();
		if(this.geoType == GEO_TYPE.MULTILINESTRING){
			if(partSize == 1){
				this.geoType = GEO_TYPE.LINESTRING;
			}
		}
		if(this.geoType == GEO_TYPE.MULTIPOLYGON){
			/*if(partSize == 1){
				this.geoType = GEO_TYPE.POLYGON;
			}*/
		    int out_side_occur_times = 0;
		    for(GeoPart part : partList){
		        if(part.getRingType() == RING_TYPE.outside){
		            if(out_side_occur_times == 0){
		                out_side_occur_times = 1;
		            }else{
		                out_side_occur_times ++;
		                break;
		            }
		        }
		    }
		    if(out_side_occur_times == 1){
		        this.geoType = GEO_TYPE.POLYGON;
		    }
		}
	}

	public Integer getDimension() {
		return dimension;
	}

	public Integer getSRID() {
		return srid;
	}

	public void setSRID(Integer srid) {
		this.srid = srid;
	}

	public GeoBuffer(GEO_TYPE geoType){
		this.geoType = geoType;
		this.srid = 0;
	}
	
	
	public GeoBuffer(GEO_TYPE geoType,int srid){
		this.geoType = geoType;
	
		this.srid = srid;
	}
	
	public GeoBuffer(GEO_TYPE geoType,int srid ,int dimension){
		this.geoType = geoType;
	
		this.srid = srid;
		if(dimension != 2 && dimension != 3){
			throw new RuntimeException(String.format("目前不支持维度为%s的Coordinate对象",dimension));
		}
		this.dimension = dimension;
	}
	
	
	
	
	public void trimToSize(){
		for(GeoPart part : this.partList){
			part.doubleBuilder.trimToSize();
		}
	}
	

	
	public GEO_TYPE getGeoType() {
		return geoType;
	}
	
	
	
	public ArrayList<GeoPart> getPartList() {
		return partList;
	}

	public void setPartList(ArrayList<GeoPart> partList) {
		this.partList = partList;
	}

	public void add(GeoPart part){
		this.partList.add(part);
	}
	
	public void destroy(){
		partList.forEach(part -> {
			part.doubleBuilder.destroy();
		});
	}
	
	public static GeoBuffer fromJTSGeometry(org.locationtech.jts.geom.Geometry geometry){
	    return GeoBufferJTSConverter.GEO_BUFFER_JTS_CONVERTER.fromGeometry(geometry);
	    
	}
	
	public Geometry toJTSGeometry(){
		
		return GeoBufferJTSConverter.GEO_BUFFER_JTS_CONVERTER.toGeometry(this);
	}
	
	/*public void addPointPart(DoubleBuilder doubleBuilder){
		GeoPart geoPart = new GeoPart(doubleBuilder,this);
		this.partList.add(geoPart);
	}*/
	
	public void addLinePart(DoubleBuilder doubleBuilder, double minX, double minY, double maxX, double maxY){
		GeoPart geoPart = new GeoPart(doubleBuilder, minX, minY, maxX, maxY, this);
		this.partList.add(geoPart);
	}
	
	public void addPolygonPart(DoubleBuilder doubleBuilder,double minX,double minY,double maxX,double maxY,RING_TYPE ringType){
		GeoPart geoPart = new GeoPart(doubleBuilder, minX, minY, maxX, maxY, this,ringType);
		this.partList.add(geoPart);
	}
	
	static boolean equals2D(DoubleBuilder doubleBuilder,int i,int j,int dimension){
		 return doubleBuilder.get(i * dimension) == doubleBuilder.get(j * dimension)
	                && doubleBuilder.get(i * dimension + 1) == doubleBuilder.get(j * dimension + 1);
	}
	
	public static boolean isCCW(DoubleBuilder doubleBuilder,int dimension) {
        // # of points without closing endpoint
        int nPts = (doubleBuilder.size() / dimension)  - 1;
        //Arrays.
        // find highest point
        double hiy = doubleBuilder.get(1);
        int hiIndex = 0;
        for (int i = 1; i <= nPts; i++) {
        	double y = doubleBuilder.get(i * dimension + 1);
            if (y > hiy) {
                hiy = y;
                hiIndex = i;
            }
        }

        // find distinct point before highest point
        int iPrev = hiIndex;
        do {
            iPrev = iPrev - 1;
            if (iPrev < 0) {iPrev = nPts;};
        } while (equals2D(doubleBuilder, iPrev, hiIndex,dimension) && iPrev != hiIndex);

        // find distinct point after highest point
        int iNext = hiIndex;
        do {
            iNext = (iNext + 1) % nPts;
        } while (equals2D(doubleBuilder, iNext, hiIndex,dimension) && iNext != hiIndex);

        /**
         * This check catches cases where the ring contains an A-B-A configuration of points. This
         * can happen if the ring does not contain 3 distinct points (including the case where the
         * input array has fewer than 4 elements), or it contains coincident line segments.
         */
        if (equals2D(doubleBuilder, iPrev, hiIndex,dimension)
                || equals2D(doubleBuilder, iNext, hiIndex,dimension)
                || equals2D(doubleBuilder, iPrev, iNext,dimension)) {return false;};

        int disc = computeOrientation(doubleBuilder, dimension,iPrev, hiIndex, iNext);

        /**
         * If disc is exactly 0, lines are collinear. There are two possible cases: (1) the lines
         * lie along the x axis in opposite directions (2) the lines lie on top of one another
         *
         * <p>(1) is handled by checking if next is left of prev ==> CCW (2) will never happen if
         * the ring is valid, so don't check for it (Might want to assert this)
         */
        boolean isCCW = false;
        if (disc == 0) {
            // poly is CCW if prev x is right of next x
            isCCW = (doubleBuilder.get(iPrev * dimension) > doubleBuilder.get(iPrev * dimension));
        } else {
            // if area is positive, points are ordered CCW
            isCCW = (disc > 0);
        }
        return isCCW;
    }
	
	static int computeOrientation(DoubleBuilder doubleBuilder,int dimension, int p1, int p2, int q) {

        double p1x = doubleBuilder.get(p1 * dimension);
        double p1y = doubleBuilder.get(p1 * dimension + 1);
        double p2x = doubleBuilder.get(p2 * dimension);
        double p2y = doubleBuilder.get(p2 * dimension + 1);
        double qx = doubleBuilder.get(q * dimension);
        double qy = doubleBuilder.get(q * dimension + 1);
        double dx1 = p2x - p1x;
        double dy1 = p2y - p1y;
        double dx2 = qx - p2x;
        double dy2 = qy - p2y;
        return RobustDeterminant.signOfDet2x2(dx1, dy1, dx2, dy2);
    }
	
	public static DoubleBuilder reverseCoordinates(DoubleBuilder doubleBuilder, int dimension){
	    int coordinatesSize = doubleBuilder.size() / dimension;
	    DoubleBuilder doubleBuilderReverse = BuilderCreator.createDouble(doubleBuilder.size());
	    int last = coordinatesSize - 1;
	    for(int i = last; i >= 0; i--) {
	        for(int j = 0; j < dimension; j ++) {
                double value = doubleBuilder.get(i * dimension + j);
                doubleBuilderReverse.append(value);
            }
	    }
	    return doubleBuilderReverse;
	}
	
	public boolean isMultiPart() {
	    int partSize = this.partList.size();
        
	    if(this.geoType == GEO_TYPE.MULTIPOINT || this.geoType == GEO_TYPE.MULTILINESTRING) {
            return partSize > 1;
        }

        if(this.geoType == GEO_TYPE.MULTIPOLYGON) {
            int outSideOccurTimes = 0;
            for(GeoPart part : partList){
                if(part.getRingType() == RING_TYPE.outside) {
                    if(outSideOccurTimes == 0) {
                        outSideOccurTimes = 1;
                    } else {
                        outSideOccurTimes ++;
                        break;
                    }
                }
            }
            return outSideOccurTimes > 1;
        }
	    
	    return false;
	}
	
	
}
