package com.northpool.spatial.geofeature;

import com.northpool.commons.util.BuilderCreator;
import com.northpool.commons.util.DoubleBuilder;
import com.northpool.spatial.Constants.GEO_TYPE;
import com.northpool.spatial.geofeature.GeoPart.RING_TYPE;
import org.locationtech.jts.geom.*;

import java.util.ArrayList;
import java.util.List;

public class GeoBufferJTSConverter {
    
    public static final GeoBufferJTSConverter GEO_BUFFER_JTS_CONVERTER = new GeoBufferJTSConverter();
    
   
    
	

	/**
	 * 
	 * @param geoBuffer
	 * @return
	 */
	@SuppressWarnings("incomplete-switch")
	public Geometry toGeometry(GeoBuffer geoBuffer){
		
		GEO_TYPE type = geoBuffer.getGeoType();
		GeometryFactory geometryFactory = new GeometryFactory(new PrecisionModel(),geoBuffer.getSRID());
		
		switch(type){
			case POINT:{
				return this.createPoint(geometryFactory,geoBuffer);
			}
			case LINESTRING:{
				return this.createLineString(geometryFactory, geoBuffer);
			}
			case POLYGON:{
				return this.createPolygon(geometryFactory, geoBuffer);
			}
			case MULTIPOINT:{
				return this.createMultiPoint(geometryFactory, geoBuffer);
			}
			case MULTILINESTRING:{
				return this.createMultiLineString(geometryFactory, geoBuffer);
			}
			case MULTIPOLYGON:{
				return this.createMultiPolygon(geometryFactory, geoBuffer);
			}
			case COLLECTION:{
				throw new RuntimeException("不支持COLLECTION");
			}
		}
		throw new RuntimeException(String.format("不支持类型%s",type.name()));
	}
	
	
	
	
	private Coordinate createCoordinate(DoubleBuilder doubleBuilder,int dimension,int index){
		Coordinate coordinate = new Coordinate();
		if(dimension == 2){
			double x = doubleBuilder.get(index);
			double y = doubleBuilder.get(index + 1);
			coordinate.setX(x);
			coordinate.setY(y);
			return coordinate;
			
		}
		if(dimension == 3){
			double x = doubleBuilder.get(index);
			double y = doubleBuilder.get(index + 1);
			double z = doubleBuilder.get(index + 2);
			coordinate.setX(x);
			coordinate.setY(y);
			coordinate.setZ(z);
			return coordinate;
		}
		throw new RuntimeException(String.format("目前不支持维度为%s的Coordinate对象",dimension));
	}
	
	
	private Point createPoint(GeometryFactory geometryFactory,GeoBuffer geoBuffer){
		Integer srid = geoBuffer.getSRID();
		int dimension = geoBuffer.getDimension();
		DoubleBuilder doubleBuilder = geoBuffer.getData(0);
		Coordinate coordinate = this.createCoordinate(doubleBuilder, dimension, 0);
		Point point = geometryFactory.createPoint(coordinate);
		point.setSRID(srid);
		return point;
	}
	
	private LineString createLineString(GeometryFactory geometryFactory,GeoBuffer geoBuffer){
		Integer srid = geoBuffer.getSRID();
		int dimension = geoBuffer.getDimension();
		GeoPart part = geoBuffer.getPart(0);
		LineString lineString = this._createLineString(geometryFactory, part, dimension);
		lineString.setSRID(srid);
		return lineString;
	}
	
	private LineString _createLineString(GeometryFactory geometryFactory,GeoPart part,int dimension){
		DoubleBuilder doubleBuilder = part.getDoubleBuilder();
		int size = doubleBuilder.size() / dimension;
		Coordinate[] coordinates = new Coordinate[size];
		for(int index = 0; index < size ; index ++ ){
			coordinates[index] = this.createCoordinate(doubleBuilder, dimension, index * dimension);
		}
		LineString lineString = geometryFactory.createLineString(coordinates);
		return lineString;
	}
	
	private MultiPoint createMultiPoint(GeometryFactory geometryFactory,GeoBuffer geoBuffer){
		throw new RuntimeException("暂时不支持");
	}
	
	private LinearRing createLinearRing(GeometryFactory geometryFactory,GeoPart part,int dimension){
		DoubleBuilder doubleBuilder = part.getDoubleBuilder();
		int size = doubleBuilder.size() / dimension;
		Coordinate[] coordinates = new Coordinate[size];
		for(int index = 0; index < size ; index ++ ){
			coordinates[index] = this.createCoordinate(doubleBuilder, dimension, index * dimension);
		}
		LinearRing linearRing = geometryFactory.createLinearRing(coordinates);
		return linearRing;
	}
	
	
	
	private Polygon createPolygon(GeometryFactory geometryFactory,GeoBuffer geoBuffer){
		Integer srid = geoBuffer.getSRID();
		int dimension = geoBuffer.getDimension();
		List<GeoPart> list = geoBuffer.getPartList();
		Polygon polygon = this._createPolygon(geometryFactory, list, dimension);
		polygon.setSRID(srid);
		return polygon;
	}
	
	
	
	private Polygon _createPolygon(GeometryFactory geometryFactory,List<GeoPart> geoPart,int dimension){
		GeoPart exteriorRingPart = geoPart.get(0);
		LinearRing exteriorRing = this.createLinearRing(geometryFactory,exteriorRingPart,dimension);
		if(geoPart.size() > 1){
			LinearRing[] rings = new LinearRing[geoPart.size() - 1];
			for(int index = 1 ; index < geoPart.size() ; index ++ ){
				rings[index - 1] = this.createLinearRing(geometryFactory,geoPart.get(index),dimension);
			}
			return geometryFactory.createPolygon(exteriorRing, rings);
		}else{
			return geometryFactory.createPolygon(exteriorRing);
		}
	}
	
	private MultiLineString createMultiLineString(GeometryFactory geometryFactory,GeoBuffer geoBuffer){
		Integer srid = geoBuffer.getSRID();
		int dimension = geoBuffer.getDimension();
		
		LineString[] arr = new LineString[geoBuffer.getPartList().size()];
		int index = 0;
		for(GeoPart part : geoBuffer.getPartList()){
			LineString lineString = this._createLineString(geometryFactory, part, dimension);
			arr[index] = lineString;
			index ++;
		}
		MultiLineString multiLineString = geometryFactory.createMultiLineString(arr);
		multiLineString.setSRID(srid);
		return multiLineString;
	} 
	
	private MultiPolygon createMultiPolygon(GeometryFactory geometryFactory,GeoBuffer geoBuffer){
		Integer srid = geoBuffer.getSRID();
		int dimension = geoBuffer.getDimension();
		ArrayList<Polygon> polygonList = new ArrayList<Polygon>();
		List<GeoPart> arr = geoBuffer.getPartList();
		ArrayList<GeoPart> polygonPart = new ArrayList<GeoPart>();
		for(GeoPart part : arr){
			RING_TYPE ringType = part.getRingType();
			//如果是外环
			if(ringType == RING_TYPE.outside){
				if(polygonPart.size() != 0){
					Polygon polygon = this._createPolygon(geometryFactory, polygonPart, dimension);
					polygonList.add(polygon);
					polygonPart = new ArrayList<GeoPart>();
				}
			}
			polygonPart.add(part);
		}
		Polygon polygon = this._createPolygon(geometryFactory, polygonPart, dimension);
		polygonList.add(polygon);
		MultiPolygon multiPolygon = geometryFactory.createMultiPolygon(polygonList.toArray(new Polygon[polygonList.size()]));
		multiPolygon.setSRID(srid);
		return multiPolygon;
		
	}


	void putIntoGeoBuffer(GeoBuffer geoBuffer,Geometry geometry,int dimension){
	    if(geometry instanceof Point){
            Point point = (Point)geometry;
            geoBuffer.addPoint(point.getX(), point.getY(), point.getCoordinate().getZ());
            return;
        }
        if(geometry instanceof LineString){
            LineString lineString = (LineString)geometry;
            Coordinate[] coordinates = lineString.getCoordinates();
            DoubleBuilder doubleBuilder = BuilderCreator.createDouble(dimension * coordinates.length);
            for(Coordinate coordinate : coordinates){
                doubleBuilder.append(coordinate.getX());
                doubleBuilder.append(coordinate.getY());
                if(dimension == 3){
                    doubleBuilder.append(coordinate.getZ());
                }
            }
            Envelope envelope = lineString.getEnvelopeInternal();
            geoBuffer.addLinePart(doubleBuilder, envelope.getMinX(), envelope.getMinY(), envelope.getMaxX(), envelope.getMaxY());
            return;
        }
        if(geometry instanceof Polygon){
            Polygon polygon = (Polygon)geometry;
			Coordinate[] coordinates = polygon.getExteriorRing().getCoordinates();
			DoubleBuilder doubleBuilder = BuilderCreator.createDouble(dimension * coordinates.length);
			for(Coordinate coordinate : coordinates){
				doubleBuilder.append(coordinate.getX());
				doubleBuilder.append(coordinate.getY());
				if(dimension == 3){
					doubleBuilder.append(coordinate.getZ());
				}
			}
			Envelope envelope = polygon.getEnvelopeInternal();
			geoBuffer.addPolygonPart(doubleBuilder, envelope.getMinX(), envelope.getMinY(), envelope.getMaxX(), envelope.getMaxY(), RING_TYPE.outside);

			int partNum = polygon.getNumInteriorRing();
            for(int i = 0 ; i < partNum; i++){
                Geometry geometryPart = polygon.getInteriorRingN(i);
                coordinates = geometryPart.getCoordinates();
                doubleBuilder = BuilderCreator.createDouble(dimension * coordinates.length);
                for(Coordinate coordinate : coordinates){
                    doubleBuilder.append(coordinate.getX());
                    doubleBuilder.append(coordinate.getY());
                    if(dimension == 3){
                        doubleBuilder.append(coordinate.getZ());
                    }
                }
                envelope = geometryPart.getEnvelopeInternal();
				geoBuffer.addPolygonPart(doubleBuilder, envelope.getMinX(), envelope.getMinY(), envelope.getMaxX(), envelope.getMaxY(), RING_TYPE.inside);
            }
        }
	}

    /**
     * @param geometry
     * @return
     */
    public GeoBuffer fromGeometry(Geometry geometry) {
      
        int srid = geometry.getSRID();
        int dimension = Double.isNaN(geometry.getCoordinate().getZ()) ? 2: 3;
        if(geometry instanceof Point){
            GeoBuffer geoBuffer = new GeoBuffer(GEO_TYPE.POINT,srid,dimension);
            this.putIntoGeoBuffer(geoBuffer, geometry, dimension);
            return geoBuffer;
        }
        if(geometry instanceof LineString){
            GeoBuffer geoBuffer = new GeoBuffer(GEO_TYPE.LINESTRING,srid,dimension);
            this.putIntoGeoBuffer(geoBuffer, geometry, dimension);
            return geoBuffer;
        }
        if(geometry instanceof Polygon){
            GeoBuffer geoBuffer = new GeoBuffer(GEO_TYPE.POLYGON,srid,dimension);
            this.putIntoGeoBuffer(geoBuffer, geometry, dimension);
            return geoBuffer;
        }
        
        if(geometry instanceof MultiLineString){
            MultiLineString multiLineString = (MultiLineString)geometry;
            GeoBuffer geoBuffer = new GeoBuffer(GEO_TYPE.MULTILINESTRING,srid,dimension);
            int partNum = multiLineString.getNumGeometries();
            for(int i = 0 ; i < partNum; i++){
                this.putIntoGeoBuffer(geoBuffer, geometry.getGeometryN(i), dimension);
            }
            return geoBuffer;
        }
        
        if(geometry instanceof MultiPolygon){
            MultiPolygon multiPolygon = (MultiPolygon)geometry;
            GeoBuffer geoBuffer = new GeoBuffer(GEO_TYPE.MULTIPOLYGON,srid,dimension);
            int partNum = multiPolygon.getNumGeometries();
            for(int i = 0 ; i < partNum; i++){
                this.putIntoGeoBuffer(geoBuffer, geometry.getGeometryN(i), dimension);
            }
            return geoBuffer;
        }
        
        throw new RuntimeException ("不支持数据类型 " + geometry.getGeometryType() + " wkt:" + geometry.toText());
    }
}
