/**
　 * <p>Title: ShapeEncoder.java</p>
　 * <p>Description: </p>
　 * <p>Copyright: Copyright (c) 2019</p>
　 * 
　 * 
　 * @date 2021年6月17日
　 * @version 1.0
*/
package com.northpool.spatial.shape;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import com.northpool.commons.util.DoubleBuilder;
import com.northpool.spatial.Constants.GEO_TYPE;
import com.northpool.spatial.AGeomEncoder;
import com.northpool.spatial.GeomEncoder;
import com.northpool.spatial.geofeature.GeoBuffer;
import com.northpool.spatial.geofeature.GeoPart;
import com.northpool.spatial.geofeature.GeoPart.RING_TYPE;
import com.northpool.spatial.Constants.SPATIAL_TYPE;

/**
 * 
 * 
 *
 */
@AGeomEncoder(type = SPATIAL_TYPE.shape)
public class ShapeEncoder implements GeomEncoder<byte[]> {
    
    public static final ShapeEncoder ENCODER = new ShapeEncoder();
    
    
    public ByteBuffer byteBufferFromGeoBuffer(GeoBuffer geoBuffer,ByteOrder byteOrder) {
        // TODO Auto-generated method stub
        if(geoBuffer == null){
            return this.getEmptyShape();
        }
        
        GEO_TYPE geotype = geoBuffer.getGeoType();
        ByteBuffer shapeBuffer;
        switch (geotype) {
            case POINT:
                shapeBuffer = this.processPoint(geoBuffer,byteOrder);
                break;
            case LINESTRING:
                shapeBuffer = this.processMultiPath(geoBuffer,byteOrder);
                break;
            case POLYGON:
                shapeBuffer = this.processMultiPath(geoBuffer,byteOrder);
                break;
           /* case MULTIPOINT:
               // this.processMultiPoint(buffer, geo,hasZ);
                break;*/
            case MULTILINESTRING:
                shapeBuffer = this.processMultiPath(geoBuffer,byteOrder);
                break;
            case MULTIPOLYGON:
                shapeBuffer = this.processMultiPath(geoBuffer,byteOrder);
                break;
                
            case COLLECTION:
            default:
                throw new RuntimeException("不支持的几何类型：" + geotype.name());
                
            }
        return shapeBuffer;
    }
    
    
    private ByteBuffer allocate(int size,ByteOrder byteOrder){
        ByteBuffer buffer = ByteBuffer.allocate(size);
        //shape为小端
        buffer.order(byteOrder);
        return buffer;
    }
    
    
    
    private ByteBuffer processMultiPath(GeoBuffer geoBuffer,ByteOrder byteOrder){
        int dimension = geoBuffer.getDimension();
        boolean bExportZ = geoBuffer.getDimension() == 2 ? false : true;
        boolean bExportM = false;
        boolean bExportID = false;
        boolean bPolygon = false;
        boolean bHasCurves = false;
        if(geoBuffer.getGeoType() == GEO_TYPE.LINESTRING || geoBuffer.getGeoType() == GEO_TYPE.MULTILINESTRING){
            bPolygon = false;
        } else if(geoBuffer.getGeoType() == GEO_TYPE.POLYGON || geoBuffer.getGeoType() == GEO_TYPE.MULTIPOLYGON){
            bPolygon = true;
        } else {
            throw new RuntimeException("不是LINESTRING或者是POLYGON类型");
        }
        
        
        int partCount = geoBuffer.getPartSize();
        int pointCount = geoBuffer.getCoordinateCount();
    
        if (!bPolygon) {
           /* for (int ipart = 0; ipart < partCount; ipart++)
                if (multipath.isClosedPath(ipart))
                    pointCount++;*/
        } else {
            
          //  pointCount += partCount;
        }
    
        int size = (4 /* type */) + (4 * 8 /* envelope */) + (4 /* part count */)
                + (4 /* point count */) + (partCount * 4 /* start indices */)
                + pointCount * 2 * 8 /* xy coordinates */;
    
        if (bExportZ)
            size += (2 * 8 /* min max */) + (pointCount * 8 /* zs */);
        if (bExportM)
            size += (2 * 8 /* min max */) + (pointCount * 8 /* ms */);
        if (bExportID)
            size += pointCount * 4 /* ids */;
        if (bHasCurves) {
            // to-do: curves
        }
        
        ByteBuffer shapeBuffer = this.allocate(size,byteOrder);
        
        ShapeType shapeType;
        if(bExportZ){
            if(bPolygon){
                shapeType = ShapeType.POLYGONZ;
            }else{
                shapeType = ShapeType.POLYLINEZ;
            }
        }else{
            if(bPolygon){
                shapeType = ShapeType.POLYGON;
            }else{
                shapeType = ShapeType.POLYLINE;
            }
        }
        int offset = 0;
        
     // write type
        shapeBuffer.putInt(offset, shapeType.id);
        offset += 4;

        // write Envelope
        double[] bbox = geoBuffer.getBBOX();
        // calls _VerifyAllStreams
        shapeBuffer.putDouble(offset,bbox[0]);
        offset += 8;
        shapeBuffer.putDouble(offset, bbox[1]);
        offset += 8;
        shapeBuffer.putDouble(offset, bbox[2]);
        offset += 8;
        shapeBuffer.putDouble(offset, bbox[3]);
        offset += 8;

        // write part count
        shapeBuffer.putInt(offset, partCount);
        offset += 4; // to-do: return error if larger than 2^32 - 1

        // write pointCount
        shapeBuffer.putInt(offset, pointCount);
        offset += 4;
        
        // write start indices for each part
        int pointIndexDelta = 0;
        int partOffset = 0;
        for (int ipart = 0; ipart < partCount; ipart++) {
            int istart = partOffset + pointIndexDelta;
            shapeBuffer.putInt(offset, istart);
            offset += 4;
           /* if (bPolygon){
                pointIndexDelta ++;
            }*/
            partOffset += (geoBuffer.getPart(ipart).getCoordinateCount());
        }
        
        if (pointCount > 0) {
            // write xy coordinates
            for (int ipart = 0; ipart < partCount; ipart++) {
                GeoPart geoPart = geoBuffer.getPart(ipart);
                DoubleBuilder doubleBuilder = geoPart.getDoubleBuilder();
                
                if (bPolygon) {
                    //如果是多边形,需要判断下
                    RING_TYPE ringType = geoPart.getRingType();
                    boolean isCCW = GeoBuffer.isCCW(doubleBuilder, dimension);
                    if(isCCW){
                        if(ringType == RING_TYPE.outside){
                            doubleBuilder = GeoBuffer.reverseCoordinates(doubleBuilder, dimension);
                        }
                    }else{
                        if(ringType == RING_TYPE.inside){
                            doubleBuilder = GeoBuffer.reverseCoordinates(doubleBuilder, dimension);
                        }
                    }
                }
                
               // int partStart = multipathImpl.getPathStart(ipart);
               // int partEnd = multipathImpl.getPathEnd(ipart);
                for (int i = 0; i < doubleBuilder.size() / dimension; i++) {
                    double x = doubleBuilder.get(i * dimension);
                    double y = doubleBuilder.get(i * dimension + 1);

                    shapeBuffer.putDouble(offset, x);
                    offset += 8;
                    shapeBuffer.putDouble(offset, y);
                    offset += 8;
                }

                // If the part is closed, then we need to duplicate the start
                // point
                if (bPolygon) {
                    double xBegin = doubleBuilder.get(0);
                    double yBegin = doubleBuilder.get(1);
                    double xEnd = doubleBuilder.get(doubleBuilder.size() - geoBuffer.getDimension());
                    double yEnd = doubleBuilder.get(doubleBuilder.size() - geoBuffer.getDimension() + 1);
                    if(xBegin == xEnd && yBegin == yEnd){
                        
                    }else{
                        shapeBuffer.putDouble(offset, xBegin);
                        offset += 8;
                        shapeBuffer.putDouble(offset, yBegin);
                        offset += 8;
                    }
                }
            }
        }
        
     // write Zs
        if (bExportZ) {
           // Envelope1D zInterval = multipathImpl.queryInterval(Semantics.Z, 0);
            double[] zInterval = geoBuffer.zInterval();
            
            shapeBuffer.putDouble(offset,zInterval[0]);
            offset += 8;
            shapeBuffer.putDouble(offset,zInterval[1]);
            offset += 8;
            if (pointCount > 0) {
                for (int ipart = 0; ipart < partCount; ipart++) {
                    DoubleBuilder doubleBuilder = geoBuffer.getPart(ipart).getDoubleBuilder();
                    for (int i = 0; i < doubleBuilder.size() / dimension; i++) {
                        double z = doubleBuilder.get(i * dimension + 2);
                        shapeBuffer.putDouble(offset,z);
                        offset += 8;
                    }
                    // If the part is closed, then we need to duplicate the
                    // start z
                    if (bPolygon) {
                        double xBegin = doubleBuilder.get(0);
                        double yBegin = doubleBuilder.get(1);
                        double xEnd = doubleBuilder.get(doubleBuilder.size() - dimension);
                        double yEnd = doubleBuilder.get(doubleBuilder.size() - dimension + 1);
                        if(xBegin == xEnd && yBegin == yEnd){
                            
                        }else{
                            double z = doubleBuilder.get(2);
                            shapeBuffer.putDouble(offset, z);
                            offset += 8;
                        }
                    }
                }
            }
        }
        return shapeBuffer;
    }
    
    
    
    private ByteBuffer processPoint(GeoBuffer geoBuffer,ByteOrder byteOrder){
        boolean bExportZ = geoBuffer.getDimension() == 2 ? false : true;
        boolean bExportM = false;
        boolean bExportID = false;
        int size = (4 /* type */) + (2 * 8 /* xy coordinate */);
        if (bExportZ){
            size += 8;
        }
        if (bExportM){
            size += 8;
        }
        if (bExportID){
            size += 4;
        }
        
        ByteBuffer shapeBuffer = this.allocate(size,byteOrder);
        ShapeType shapeType;
        if(bExportZ){
            shapeType = ShapeType.POINTZ;
        }else{
            shapeType = ShapeType.POINT;
        }
        
        int offset = 0;

        shapeBuffer.putInt(offset, shapeType.id);
        offset += 4;

   
        shapeBuffer.putDouble(offset,geoBuffer.getX());
        offset += 8;
        shapeBuffer.putDouble(offset,geoBuffer.getY());
        offset += 8;

        // write Z
        if (bExportZ) {
            shapeBuffer.putDouble(offset,geoBuffer.getZ());
            offset += 8;
        }
        return shapeBuffer;
    }
    
    
    
    
    
    
    
    
    

    ByteBuffer getEmptyShape(){
        ByteBuffer shapeBuffer = ByteBuffer.allocate(4);
        shapeBuffer.putInt(0, ShapeType.NULL.id);
        return shapeBuffer;
    }
    
    static public ShapeType is(GEO_TYPE geoType) {
        
        switch (geoType) {
            case POINT:
                return ShapeType.POINT;  
            case LINESTRING:
                return ShapeType.POLYLINE;
            case POLYGON:
                return ShapeType.POLYGON;       
            case MULTIPOINT:
                return ShapeType.MULTIPOINT;
            case MULTILINESTRING:
                return ShapeType.POLYLINE;
            case MULTIPOLYGON:
                return ShapeType.POLYGON;   
            default:
                throw new RuntimeException("不支持的几何类型：" + geoType.name());
                
        }
    }




    
    @Override
    public byte[] fromGeoBuffer(GeoBuffer geoBuffer) {
        ByteBuffer byteBuffer = this.byteBufferFromGeoBuffer(geoBuffer,ByteOrder.LITTLE_ENDIAN);
        int len = byteBuffer.limit() - byteBuffer.position();
        byte[] bytes = new byte[len];
        byteBuffer.get(bytes);
        return bytes;
    }




    
    @Override
    public byte[] fromGeoBuffer(GeoBuffer geoBuffer, boolean includeSrid, ByteOrder byteOrder) {
        ByteBuffer byteBuffer = this.byteBufferFromGeoBuffer(geoBuffer,byteOrder);
        int len = byteBuffer.limit() - byteBuffer.position();
        byte[] bytes = new byte[len];
        byteBuffer.get(bytes);
        return bytes;
    }
    
    
    
    
    
    
    
    /*public <T> T fromGeoBuffer(GeoBuffer geoBuffer,) {
        // TODO Auto-generated method stub
        return null;
    }*/
    
    
}
