package com.northpool.spatial.wkb;

import java.io.ByteArrayInputStream;
import java.io.IOException;

import org.locationtech.jts.io.ByteOrderValues;
import org.locationtech.jts.io.WKBConstants;

import com.northpool.commons.util.BuilderCreator;
import com.northpool.commons.util.DoubleBuilder;
import com.northpool.spatial.ByteOrderDataInStream;
import com.northpool.spatial.GeometryInfo;
import com.northpool.spatial.Constants.GEO_TYPE;
import com.northpool.spatial.geofeature.GeoBuffer;
import com.northpool.spatial.geofeature.GeoPart.RING_TYPE;


public abstract class AbstractWkbDecoder {


	public static final int UNKNOWN = 0;
	public static final int POINT = 1;
	public static final int LINESTRING = 2;
	public static final int POLYGON = 3;
	public static final int MULTIPOINT = 4;
	public static final int MULTILINESTRING = 5;
	public static final int MULTIPOLYGON = 6;
	public static final int COLLECTION = 7;
		
   // public static final WkbDecoder DECODER = new WkbDecoder();

	
	
	
	
	int getSRID(ByteOrderDataInStream dis) throws IOException{
		byte[] sridByte = dis.readByte4();
		int srid = 0;
	    srid = sridByte[3] << 24 | (sridByte[2] & 0xFF) << 16 | (sridByte[1] & 0xFF) << 8 | sridByte[0] & 0xFF;
	    return srid;
	}
	
	ByteOrderDataInStream getStruct(ByteArrayInputStream instream){
		ByteOrderDataInStream dis = new ByteOrderDataInStream(instream);
		return dis;
	}
	
    void setByteOrder(ByteOrderDataInStream dis) throws IOException{
		byte byteOrderWKB = dis.readByte();
	    // always set byte order, since it may change from geometry to geometry
		int byteOrder = byteOrderWKB == WKBConstants.wkbNDR ? ByteOrderValues.LITTLE_ENDIAN : ByteOrderValues.BIG_ENDIAN;
		dis.setOrder(byteOrder);
	}
	
	
    GeometryInfo buildGeometryTypeAndZM(ByteOrderDataInStream dis) throws IOException{
		int typeInt = dis.readInt();
		GeometryInfo info = new GeometryInfo();
		info.geometryType = typeInt & 0xff;
	    // determine if Z values are present
		info.hasZ = (typeInt & 0x80000000) != 0;
		info.hasM = (typeInt & 0x40000000) != 0;
		return info;
	   
	}
	
    void check(ByteArrayInputStream instream) throws Exception{
		if (null == instream || instream.available() <= 0){
			throw new Exception("流不能为空");
		}
	}
	
    int getSize(ByteArrayInputStream instream){
		return instream.available();
	}
	
	protected GeoBuffer _toGeoBuffer(byte[] bytes) throws Exception{
		
	    ByteArrayInputStream instream = new ByteArrayInputStream(bytes);
	    
		ByteOrderDataInStream dis = new ByteOrderDataInStream(instream);
				
	    byte byteOrderWKB = dis.readByte();
	    // always set byte order, since it may change from geometry to geometry
		int byteOrder = byteOrderWKB == WKBConstants.wkbNDR ? ByteOrderValues.LITTLE_ENDIAN : ByteOrderValues.BIG_ENDIAN;
		
		dis.setOrder(byteOrder);
		
		
		int typeInt = dis.readInt();
		
		GeometryInfo info = new GeometryInfo();
		
		info.geometryType =  (typeInt & 0xffff)%1000;
		
	    // determine if Z values are present
		info.hasZ = ((typeInt & 0x80000000) != 0 || (typeInt & 0xffff)/1000 == 1 || (typeInt & 0xffff)/1000 == 3);
		
		info.hasM =  ((typeInt & 0x40000000) != 0 || (typeInt & 0xffff)/1000 == 2 || (typeInt & 0xffff)/1000 == 3);
		
		boolean haveSrid = (typeInt & 0x20000000) != 0;
		
        int srid = 0;
        
        if (haveSrid) {
            srid = dis.readInt();
        }
	//	int size = instream.available();
		
		GEO_TYPE geoType = is(info.geometryType);
		
		int inputDimension =  info.hasZ ? 3 : 2;

		GeoBuffer geo = new GeoBuffer(geoType,srid,inputDimension);
		
		return this.process(geoType, dis, geo, info);

	}
	
	
	protected GeoBuffer process(GEO_TYPE geoType,ByteOrderDataInStream dis,GeoBuffer geo,GeometryInfo info) throws IOException{
		switch (geoType) {
		case POINT:
			this.processPoint(dis, geo,info);
			
			break;
		case LINESTRING:

			this.processLineString(dis, geo,info);
			break;
		case POLYGON:
			this.processPolygon(dis, geo,info);
			break;
			
		case MULTIPOINT:
			this.processMultiPoint(dis, geo,info);
			break;
		case MULTILINESTRING:
			this.processMultiLine(dis, geo,info);
			break;
		case MULTIPOLYGON:
			this.processMultiPolygon(dis, geo,info);
			break;
			
		case COLLECTION:
		default:
			throw new RuntimeException("不支持的几何类型：" + geoType.name());
			
		}
		return geo;
	}
	
	public GEO_TYPE is(int type) {
		switch (type) {
			case POINT:
				return GEO_TYPE.POINT;
			case LINESTRING:
				return GEO_TYPE.LINESTRING;
			case POLYGON:
				return GEO_TYPE.POLYGON;
			case COLLECTION:
				return GEO_TYPE.COLLECTION;
			case MULTIPOINT:
				return GEO_TYPE.MULTIPOINT;
			case MULTILINESTRING:
				return GEO_TYPE.MULTILINESTRING;
			case MULTIPOLYGON:
				return GEO_TYPE.MULTIPOLYGON;
			}
		throw new RuntimeException("未知的数据类型" + type);
	}
	
	
	private void processMultiPolygon(ByteOrderDataInStream dis,GeoBuffer geo,GeometryInfo info) throws IOException {
		// 解析几何结构
		//int nParts = this.readIntFromHex(ins, 4,_data,_bLittleEndian);
		int nParts = dis.readInt();
		for (int i = 0; i < nParts; i++) {
		    byte byteOrderWKB = dis.readByte();
            
            int byteOrder = byteOrderWKB == WKBConstants.wkbNDR ? ByteOrderValues.LITTLE_ENDIAN : ByteOrderValues.BIG_ENDIAN;

            dis.setOrder(byteOrder);
            
            int typeInt = dis.readInt();
            boolean haveSrid = (typeInt & 0x20000000) != 0;
            if (haveSrid) {
                dis.readInt();
            }
			this.processPolygon(dis, geo,info);
		}// for part
	}
	
	
	// 多part-线处理
	private void processMultiLine(ByteOrderDataInStream dis,GeoBuffer geo,GeometryInfo info) throws IOException {
		// 解析几何结构
		int nParts = dis.readInt();
		for (int i = 0; i < nParts; i++) {
		    
		    byte byteOrderWKB = dis.readByte();
            
            int byteOrder = byteOrderWKB == WKBConstants.wkbNDR ? ByteOrderValues.LITTLE_ENDIAN : ByteOrderValues.BIG_ENDIAN;

            dis.setOrder(byteOrder);
            
            int typeInt = dis.readInt();
            
            boolean haveSrid = (typeInt & 0x20000000) != 0;
            if (haveSrid) {
                dis.readInt();
            }
            
			DoubleBuilder doubleBuilder = null;
	        int count = dis.readInt();
	        int doubleBufferSize;
	        if(info.hasZ){
	            doubleBufferSize  = count * 3;
	        }else{
	            doubleBufferSize  = count * 2;
	        }
	        
	        doubleBuilder = BuilderCreator.createDouble(doubleBufferSize);
	        double[] bbox = this.readPointArray(dis,doubleBuilder,info,count);
			geo.addLinePart(doubleBuilder, bbox[0], bbox[1], bbox[2], bbox[3]);

		}// for part
	}
	
	
	// 多part-点处理
	private void processMultiPoint(ByteOrderDataInStream dis,GeoBuffer geo,GeometryInfo info) throws IOException {
		// 解析几何结构
		int nParts = dis.readInt();

		for (int i = 0; i < nParts; i++) {
		    byte byteOrderWKB = dis.readByte();
		    
		    int byteOrder = byteOrderWKB == WKBConstants.wkbNDR ? ByteOrderValues.LITTLE_ENDIAN : ByteOrderValues.BIG_ENDIAN;

		    dis.setOrder(byteOrder);
		    
            int typeInt = dis.readInt();
            boolean haveSrid = (typeInt & 0x20000000) != 0;
            if (haveSrid) {
                dis.readInt();
            }

			this.processPoint(dis, geo,info);
			
			//this.processMunltiObjectNextPart(grids);
				
		}// for part
	}
	
	
	// 多边形处理
	private void processPolygon(ByteOrderDataInStream dis,GeoBuffer geo,GeometryInfo info) throws IOException {
		// 解析几何结构
		
		int nParts = dis.readInt();
		
		for (int i = 0; i < nParts; i++) {
		    DoubleBuilder doubleBuilder = null;
	        int count = dis.readInt();
	        int doubleBufferSize;
	        if(info.hasZ){
	            doubleBufferSize  = count * 3;
	        }else{
	            doubleBufferSize  = count * 2;
	        }
	        
	        doubleBuilder = BuilderCreator.createDouble(doubleBufferSize);
	        double[] bbox = this.readPointArray(dis,doubleBuilder,info,count);
			if(i == 0){
				geo.addPolygonPart(doubleBuilder, bbox[0], bbox[1], bbox[2], bbox[3],RING_TYPE.outside);
			}else{
				geo.addPolygonPart(doubleBuilder, bbox[0], bbox[1], bbox[2], bbox[3],RING_TYPE.inside);
			}
		}
	}
	
	
	

	
	private void processLineString(ByteOrderDataInStream dis,GeoBuffer geo,GeometryInfo info) throws IOException {
		// 解析几何结构
		DoubleBuilder doubleBuilder = null;
		int count = dis.readInt();
        int doubleBufferSize;
        if(info.hasZ){
            doubleBufferSize  = count * 3;
        }else{
            doubleBufferSize  = count * 2;
        }
        
        doubleBuilder = BuilderCreator.createDouble(doubleBufferSize);
		double[] bbox = this.readPointArray(dis,doubleBuilder,info,count);
		geo.addLinePart(doubleBuilder, bbox[0], bbox[1], bbox[2], bbox[3]);
		
	}
	
	
	// 点处理
	private void processPoint(ByteOrderDataInStream dis,GeoBuffer geo,GeometryInfo info) throws IOException {

		// 解析几何结构
		double x = dis.readDouble();
		double y = dis.readDouble();
		double z = Double.NaN;
		
		
		if(info.hasZ){
			z = dis.readDouble();
		}
		if(info.hasM){
			dis.readDouble();
		}
		geo.addPoint(x, y, z);
	}
	
	
	 
	private double[] readPointArray(ByteOrderDataInStream dis,DoubleBuilder doubleBuilder,GeometryInfo info,int length) throws IOException {
		
		//double[] result = new double[count * 2];
		double minX = 0d;
		double minY = 0d;
		double maxX = 0d;
		double maxY = 0d;
		
		for (int i = 0; i < length; i++) {
			double x = dis.readDouble();
			double y = dis.readDouble();
			if(i == 0){
			    minX = maxX = x;
			    minY = maxY = y;
			}
			if(x > maxX){
				maxX = x;
			}
			if(x < minX){
				minX = x;
			}
			if(y > maxY){
				maxY = y;
			}
			if(y < minY){
				minY = y;
			}
			doubleBuilder.append(x);
			doubleBuilder.append(y);
			if(info.hasZ){
				double z = dis.readDouble();
				doubleBuilder.append(z);
			}
			if(info.hasM){
				@SuppressWarnings("unused")
				double m = dis.readDouble();
			}
		}
		//doubleBuilder.trimToSize();
		return new double[]{minX,minY,maxX,maxY};
	}

	
}
