package com.northpool.resources.datasource.db;

import com.northpool.AppUtil;
import com.northpool.commons.reflect.Bean;
import com.northpool.commons.reflect.Function;
import com.northpool.commons.util.MD5;
import com.northpool.exception.CommonException;
import com.northpool.resources.Constants.DATA_SOURCE_TYPE;
import com.northpool.resources.dialect.db.SQLDialect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.DriverManager;
import java.util.HashMap;
import java.util.Map;


public class JDBCPoolManager {

	protected Logger log = LoggerFactory.getLogger(this.getClass());

	protected static JDBCPoolManager INSTANCE = new JDBCPoolManager();
	
	protected static HashMap<String,DataSource> POOL_CACHE = new HashMap<String,DataSource>();
	
	protected static HashMap<String,DBPoolParameter> POOL_PARAMETER_MAP = new HashMap<String,DBPoolParameter>();
	static{
	    
	    POOL_PARAMETER_MAP.put(DBPoolParameter.DEFAULT_POOL_NAME, DBPoolParameter.DEFAULT);
	    
	    
	    
	}
	
	
	public static JDBCPoolManager getInstance() {
		return INSTANCE;
	}

	
	
	

	
	public DataSource getPool(DbDataSource dbDataSource) {
	    
	    String poolName = dbDataSource.poolName;
	    
	    DBPoolParameter parameter = POOL_PARAMETER_MAP.get(poolName);
	    
	    if(parameter == null){
	        throw new RuntimeException("没有名为 " + poolName + " 的数据库池参数");
	    }
	    
        return this.getDataSource(dbDataSource.getDataSourceType(), dbDataSource.getUrl(), dbDataSource.getUser(), dbDataSource.getPassword(), parameter);
    }
	
	
	
	private DataSource getDataSource(DATA_SOURCE_TYPE type,String url,String user,String password,DBPoolParameter parameter) {
		String key = this.getPoolAliasName(type, url, user,password,parameter);
		DataSource pool = POOL_CACHE.get(key);
		if(pool != null){
			return pool;
		}
		synchronized(this){
			pool = POOL_CACHE.get(key);
			if(pool != null){
				return pool;
			}
			pool = this.createDataSource(type, url, user, password, parameter);
			POOL_CACHE.put(key, pool);
			return pool;
		}
	}
	
	public synchronized void remove(DATA_SOURCE_TYPE type,String url,String user,String password){
		String key = this.getPoolAliasName(type, url, user,password,null);
		DataSource pool = POOL_CACHE.get(key);
		if(pool != null){
			this.destoryPool(pool);
			POOL_CACHE.remove(key);
		}
	}
	
	private void destoryPool(DataSource pool){
		try {
			Function.invokeMethod(pool, "close", new Object[]{});
		} catch (CommonException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	
	private synchronized DataSource createDataSource(DATA_SOURCE_TYPE type,String url,String user,String password,DBPoolParameter parameter)  {
		
		//DataSourceTemplate dataSourceTemplate = (DataSourceTemplate) AppUtil.getBean("dataSourceTemplate");

		
		Map<String,Object> dataSourceConfig = Bean.getObjectValue(parameter);
		
		SQLDialect dialect = SQLDialect.getSQLDialect(type);
        url = dialect.createConnectUrl(url);
        String driver = dialect.getJDBCDriver();
		
		dataSourceConfig.put("driverClassName", driver);
		dataSourceConfig.put("url", url);
		dataSourceConfig.put("username", user);
		dataSourceConfig.put("password", password);
		
		
		//String dataSourceType = dataSourceConfig.get("type");
		String className = findPoolClass();

		log.debug("数据库类型：{}", type.name());
		log.debug("使用连接池：{}", className);

		return initDataSource(className, dataSourceConfig);
	}
	
	
	
	
	
	
	
	

	private String findPoolClass() {
		String className = null;
		try {
			className = JDBCPoolEnum.valueOf("druid").getClassName();
			return className;
		} catch (Exception e) {
			log.info("druid连接池未找到");
		}

		try {
			className = JDBCPoolEnum.valueOf("proxool").getClassName();
			return className;
		} catch (Exception e) {
			log.info("proxool连接池未找到");
		}

		try {
			className = JDBCPoolEnum.valueOf("c3p0").getClassName();
			return className;
		} catch (Exception e) {
			log.info("c3p0连接池未找到");
		}

		log.info("没有找到任何jdbc连接池，需要导入druid，proxool，c3p0任意一种jar包");

		return className;
	}
	
	
	public void inValid(DATA_SOURCE_TYPE type,String url,String user,String password) throws Exception{
		SQLDialect dialect = SQLDialect.getSQLDialect(type);
		String fullClassName = dialect.getJDBCDriver();
		Connection c = null;
		try{
			Class.forName(fullClassName);
			url = dialect.createConnectUrl(url);
			c = DriverManager.getConnection(url, user, password);
		}catch (Exception e) {
			throw e;
		}finally{
		    if(c != null){
		        c.close();
		    }
		}
	}

	private DataSource initDataSource(String fullClassName, Map<String, Object> dataSourceConfig) {
		try {
			Class<?> c3p0 = Class.forName(fullClassName);
		//	com.alibaba.druid.pool.DruidDataSource
			DataSource pool = (DataSource) c3p0.newInstance();
			
			setPoolProperties(pool, dataSourceConfig);
			return pool;
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		} catch (InstantiationException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		}
		return null;
	}

	private void setPoolProperties(Object pool, Map<String, Object> jdbcCfg) {
	    
	    Bean.setObjectValueByFieldName(pool, jdbcCfg);
	    
		/*String val;
		Object oval;
		for (String key : jdbcCfg.keySet()) {
			val = jdbcCfg.get(key);
			if (val == null){
				val = "";
			}
			// 根据属性名称构造set函数
			key = key.trim();
			key = "set" + key.substring(0, 1).toUpperCase() + key.substring(1);
			// 获得类所有的method
			Method[] method = pool.getClass().getMethods();
			for (int i = 0; i < method.length; i++) {
				String methodName = method[i].getName();
				// 如果set函数和类里的函数名称能对应上则调用函数
				if (methodName.equals(key)) {
					
					Class<?>[] t = method[i].getParameterTypes();
					if (t.length > 1){
						continue;
					}
					try {
						oval = val;
						if (t[0].getName().equals("int")) {
							oval = Integer.parseInt(val);
						}
						if (t[0].getName().equals("long")) {
							oval = Long.valueOf(val);
						}
						if (t[0].getName().equals("boolean")) {
							oval = Boolean.valueOf(val);
						}
						method[i].invoke(pool, new Object[] { oval });
					} catch (Exception e1) {
						e1.printStackTrace();
					}
				}
			}
		}*/
	}
	
	
	private String getPoolAliasName(DATA_SOURCE_TYPE type,String url,String user,String password,DBPoolParameter parameter) {
	    if(parameter == null){
	        parameter = DBPoolParameter.DEFAULT;
	    }
		String alias = url.replace("\\", "").replace("/", "").replace("_", "");
		StringBuilder beanName = new StringBuilder();
		beanName.append(type).append("||");
		// 有别名时加上别名前缀
		if (alias != null && !alias.isEmpty()) {
			beanName.append(alias).append("||");
		}
		beanName.append(user).append("||").append(MD5.getMD5String(password));
		beanName.append("||").append(parameter.getName());
		beanName.append("_pool");
		return beanName.toString();
	}
	
	

}