package com.northpool.resources.datatablebuilder.db;

import com.northpool.resources.Constants.DATA_BASE_TYPE;
import com.northpool.resources.datasource.IDataSource;
import com.northpool.resources.datasource.db.DbDataSource;
import com.northpool.resources.datatable.ITable;
import com.northpool.resources.datatable.db.DBTable;
import com.northpool.resources.datatablebuilder.AbstractTableBuilder;
import com.northpool.resources.datatablebuilder.IColumn;
import com.northpool.resources.datatablebuilder.IIndex;
import com.northpool.resources.datatablebuilder.TableSchemaBean;
import com.northpool.resources.datatablebuilder.db.index.Index;
import com.northpool.resources.dialect.db.SQLDialect;
import com.northpool.resources.sql.SQLGenericDao;
import com.northpool.resources.sql.jdbc.SQLGenericDaoImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.CompletableFuture;

public abstract class AbstractDBTableBuilder extends AbstractTableBuilder implements IDBTableBuilder {
	
	public static final String COLUMN_NAME = "COLUMN_NAME";
	public static final String DATA_TYPE = "DATA_TYPE";
	public static final String TYPE_NAME = "TYPE_NAME";
	public static final String COLUMN_SIZE = "COLUMN_SIZE";
	public static final String DECIMAL_DIGITS = "DECIMAL_DIGITS";
	public static final String NULLABLE = "NULLABLE";
	public static final String COLUMN_DEF = "COLUMN_DEF";
	public static final String KEY_SEQ = "KEY_SEQ";
	public static final String REMARKS = "REMARKS";
	public static final String VIEW_PK_COLUMN_NAME = "vid";
	
	
	
	/**
	 * DB类型
	 */
	protected DATA_BASE_TYPE dbType; 
	/**
	 * 列MAP
	 */
	//protected HashMap<String,IColumn> columnMap = new HashMap<String,IColumn>();
	/**
	 * 列array主要用于列排序
	 */
	//protected String[] columnArray;

	protected Logger logger = LoggerFactory.getLogger(AbstractDBTableBuilder.class);
	/**
	 * 表名
	 */
	//protected String tableName;
	protected String schema;
	protected Boolean isCaseSensitive;
	protected Connection dbconn;
	
	protected String cataLog;
				
	protected Boolean isView = false;
	/**
	 * 索引列表
	 */
	//protected HashMap<String,IIndex> indexMap = new HashMap<String,IIndex>();	
	
	//protected DbDataSource dbDataSource;

	//protected SQLDialect sqlDialect;
	/**
	 * 是否有序列
	 */
	protected Boolean hasSequence;
	
	protected SQLGenericDao genericDao;
	
	protected String tableRemarks;
	
	
	 public String getSchema(){
	     return this.schema;
	 }
	
	
	protected Connection getConnection(){
		return this.dbconn;
	}
	@Override
	public void init(String tableName, IDataSource dataSource,Map<String,?> config) throws Exception {
	    this.tableName = tableName;
	    if(config != null){
	        this.isCaseSensitive = config.get("isCaseSensitive") == null ? true : (Boolean)config.get("isCaseSensitive");
	    }else{
	        this.isCaseSensitive = true;
	    }
	    
        this.dataSource = dataSource;
        DbDataSource dbDataSource = (DbDataSource)this.dataSource;
        this.tryConnection(dbDataSource);
        this.dbconn = dbDataSource.connection().getConnection();
        try{
			this.dbType = this.processDataBaseType();
			this.processTableNameAndSchema(tableName, schema, isCaseSensitive);
			this.dialect = SQLDialect.getSQLDialect(dbDataSource.getDataSourceType());
			this.genericDao = new SQLGenericDaoImpl(dbDataSource);
			this.dbconn = this.getConnection();
			this.cataLog = this.getCataLog();
			DatabaseMetaData dbmd = dbconn.getMetaData();
			this.buildTableInfo(dbmd, this.tableName, this.schema);
			if(this.hasSpatialColumn()){
				this.buildSpatialColumnInfo();
			}
		}finally {//防止连接池泄露
        	this.dbconn.close();
		}

	}
	
	protected void tryConnection(DbDataSource dbDataSource) throws Exception{
	    try{
	        dbDataSource.testConnection();
	    }catch(Exception e){
	        throw new Exception(dbDataSource.toString() + " 测试连接失败,请检查数据库信息是否正确:" + e.getMessage());
	    }
	}
	
	protected String getCataLog() throws SQLException{
		ResultSet resultSet = this.dbconn.getMetaData().getCatalogs();
		if(resultSet.wasNull()){
			throw new RuntimeException(this.tableName + "获取信息出现问题,获得不到CataLog信息");
		}
		resultSet.next();
		String catalog = resultSet.getString("TABLE_CAT");
		resultSet.close();
		return catalog;
	}
	
	
	
	protected abstract DATA_BASE_TYPE processDataBaseType();
	
	protected abstract void processTableNameAndSchema(String tableName, String schema,Boolean isCaseSensitive);
	
	protected abstract String getRemarks(ResultSet columns)  throws SQLException;
	
	protected abstract ResultSet getColumns(DatabaseMetaData dbmd,String tableName,String schema) throws SQLException;
	
	protected abstract IColumn createColumn(String columnName,Integer columnSize,Integer digits,String columnTypeName,Boolean nullable,Boolean withDefault,String colRemarks);
	
	protected abstract CompletableFuture<Void> createSpatialIndex(final String colname);
	
	protected String sequenceName(){
		if(this.getPk() == null){
			throw new RuntimeException(this.mark() + "没有主键设置,不能创建序列");
		}
		return new StringBuilder().append(this.tableName).append("_").append(this.getPk().getColname()).append("_seq").toString();
	}
	
	@Override
	public String mark(){
		return new StringBuilder().append(this.cataLog).append('_').append(this.schema).append('.').append(this.tableName).append("@").append(this.dataSource.mark()).toString();
	}
	
	
	public String toString(){
        return new StringBuilder().append(this.cataLog).append('_').append(this.schema).append('.').append(this.tableName).append("@").append(this.dataSource.toString()).toString();
    }
	
	
	
	protected String createSpatialIndexName(String colname){
		String indexName = "sindex_" + colname + "_" + this.tableName;
		return indexName;
	}
	
	@Override
	public CompletableFuture<Void> dropIndex(final String indexName){
		if(this.isView){
			throw new RuntimeException(mark() + "为视图,不支持此操作");
		}
		CompletableFuture<Void> promise = CompletableFuture.runAsync(()->{
			logger.info(mark() + ";删除索引" + indexName);
			String sql = "drop index " + indexName;
			logger.info("execute sql" + sql);
			this.genericDao.doExecuteSql(sql, null ,null);
			this.dropIndexInIndexMap(indexName);
			/*Statement statement = null;
			try {
				statement = connection.createStatement();
				statement.execute(sql);
				statement.close();
				connection.close();
				this.dropIndexInIndexMap(indexName);
			}catch(SQLException e){
				try {
					connection.close();
				} catch (SQLException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				}
				throw new RuntimeException(e);
			}*/
			
		});
		return promise;
		
	}
	
	@Override
	public CompletableFuture<Void> dropPK(){
		if(this.isView){
			throw new RuntimeException(mark() + "为视图,不支持此操作");
		}
		if(this.getPk() == null){
			CompletableFuture<Void> promise = CompletableFuture.runAsync(() ->{
				logger.info(mark() + ";没有主键索引");
			});
			return promise; 
		}else{
			CompletableFuture<Void> promise = CompletableFuture.runAsync(() ->{
				logger.info(mark() + ";删除索引" + this.getPk().getName());
				String sql = "ALTER TABLE " + this.tableName +" DROP CONSTRAINT " + this.getPk().getName() ;
				logger.info("execute sql" + sql);
				this.genericDao.doExecuteSql(sql, null,null);
				this.dropIndexInIndexMap(this.getPk().getName());
				/*Statement statement = null;
				try {
					statement = connection.createStatement();
					statement.execute(sql);
					statement.close();
					connection.close();
					this.dropIndexInIndexMap(this.pkIndexName);
				}catch(SQLException e){
					try {
						connection.close();
					} catch (SQLException e1) {
						// TODO Auto-generated catch block
						e1.printStackTrace();
					}
					throw new RuntimeException(e);
				}*/
			});
			return promise; 
		}
	}
	
	
	@Override
	public CompletableFuture<Void> createPK(final String colname){
		if(this.isView){
			throw new RuntimeException(mark() + "为视图,不支持此操作");
		}
		if(this.getPk() != null){
			throw new RuntimeException(this.mark() + "已有主键字段:" + this.getPk().getColname());
		}
		CompletableFuture<Void> promise = CompletableFuture.runAsync(()->{
			logger.info(mark() + "_" + colname + "创建主键");
			String pkName = this.createPKName();
			String sql = "ALTER TABLE " + this.tableName + " ADD CONSTRAINT " + pkName + " PRIMARY KEY(" + colname + ")";
			logger.info("execute sql" + sql);
			this.genericDao.doExecuteSql(sql, null,null);
			Index index = new Index();
			index.setName(pkName);
			index.setUnique(true);
			index.addColname(colname);
			index.setIsPK(true);
			this.addToIndexMap(colname, index);
		});
		return promise;
	}
	
	
	
	
	@Override
	public CompletableFuture<Void> createIndex(Boolean unique,final String... colnames){
		if(this.isView){
			throw new RuntimeException(mark() + "为视图,不支持此操作");
		}
		Boolean isSpatialIndex = false;
		
		for(String colname: colnames){
			IColumn column = this.columnMap.get(colname);
			if(column == null){
				throw new RuntimeException(mark() + "中 " + colname + " 字段不存在");
			}
			if(column.isSpatial()){
				if(colnames.length > 1){
					throw new RuntimeException(mark() + "中 " + colname + " 空间字段不能为联合索引");
				}else{
					isSpatialIndex = true;
				}
			}
			
		}
		if(isSpatialIndex){
			return this.createSpatialIndex(colnames[0]);
		}else{
			CompletableFuture<Void> promise = CompletableFuture.runAsync(()->{
				logger.info(mark() + "_" + String.join(",", colnames) + "创建索引");
				String indexName = this.createIndexName(colnames);
				String sql = "CREATE " + (unique == true ? "UNIQUE" : "") +  " INDEX " + indexName +  " ON " + this.tableName + " (" + String.join(",", colnames) + ");";
				logger.info("execute sql" + sql);
				this.genericDao.doExecuteSql(sql, null ,null);
				Index index = new Index();
				index.setName(indexName);
				index.setUnique(unique);
				for(String colname: colnames){
					index.addColname(colname);
					this.addToIndexMap(colname, index);
				}
			});
			return promise;
		}
	}
	
	
	
	
	protected void buildIndexInfo(DatabaseMetaData dbmd) throws SQLException{
		this.indexMap = this.getIndexInfo(dbmd);
	}
	
	protected HashMap<String,IIndex> getIndexInfo(DatabaseMetaData dbmd) throws SQLException{
		HashMap<String,IIndex> map = new HashMap<String,IIndex>();
		if(this.isView){
			
		}else{
			/*ResultSet indexes = dbmd.getIndexInfo(this.cataLog, this.schema, this.tableName, false, false);
			while(indexes.next()){
				String indname = indexes.getString("INDEX_NAME");
				String asc = indexes.getString("ASC_OR_DESC");
				boolean nonunique = indexes.getBoolean("NON_UNIQUE");
				String colname = indexes.getString("COLUMN_NAME");
				Index index = new Index();
				index.setAsc(asc);
				index.setName(indname);
				index.setUnique(!nonunique);
				index.addColname(colname);
				IColumn column = this.columnMap.get(colname);
				if(column != null){
					if(column.isSpatial()){
						index.setSpatial(true);
					}
				}
				map.put(indname, index);
			}
			indexes.close();*/
		}
		return map;
	}
	
	
	
	
	public IIndex getPrimaryInfo(DatabaseMetaData dbmd) throws SQLException{
		IIndex pk = null;
		if(this.isView){
			//如果是试图
			for(Entry<String, IColumn> set : this.columnMap.entrySet()){
				IColumn column = set.getValue();
				String columnName = column.getColumnName();
				if(VIEW_PK_COLUMN_NAME.equalsIgnoreCase(columnName)){
					pk = new Index();
					pk.setName(this.createPKName());
					pk.setUnique(false);
					pk.setIsPK(true);
					pk.setColname(columnName);
				}
			}
		}else{
			ResultSet pkeys = dbmd.getPrimaryKeys(this.cataLog, this.schema, this.tableName.replace("\"", ""));
			String pkName = null ;
			if(pkeys.next() == true) {
				String pkeystr = pkeys.getString("COLUMN_NAME");
			//	String seq = pkeys.getString("KEY_SEQ");
				IColumn colunm = this.columnMap.get(pkeystr);
				pkName = pkeys.getString("PK_NAME");
				if(colunm != null){
					pk = new Index();
					pk.setName(pkName);
					pk.setUnique(false);
					pk.setIsPK(true);
					pk.setColname(colunm.getColumnName());
				}
			}
			if(pkeys.next() == true){
				throw new RuntimeException("表/视图" + mark() + "为联合主键,系统不支持联合主键");
			}
			pkeys.close();
		}
		return pk;
	}
	
	
	
	protected void buildPrimaryInfo(DatabaseMetaData dbmd) throws SQLException{
		IIndex pk = this.getPrimaryInfo(dbmd);
		if(pk != null){
			this.addToIndexMap(pk.getName(), pk);
			if(pk.isComposite()){
	            throw new RuntimeException("不支持混合主键");
	        }
			this.columnMap.get(pk.getColname()).setPK();
		}
		
		
	}
	
	
    protected void buildTableInfo(DatabaseMetaData dbmd, String tableName, String schema) throws Exception {
        // TODO Auto-generated method stub
        this.getTableInfo(dbmd, tableName);
        this.buildColumns(dbmd, tableName, schema);
        this.buildIndexInfo(dbmd);
        this.buildPrimaryInfo(dbmd);
    }
    
	/*protected void buildTableInfo(DatabaseMetaData dbmd,String tableName,String schema) throws Exception{
		this.getTableInfo(dbmd, tableName);
		this.buildColumns(dbmd, tableName, schema);
		this.buildIndexInfo(dbmd);
	    this.buildPrimaryInfo(dbmd);
	}*/
	
	
	protected void getTableInfo(DatabaseMetaData dbmd,String tableName) throws SQLException{
		ResultSet resultSet = dbmd.getTables(this.cataLog, this.schema, this.tableName, new String[]{"TABLE","VIEW"});
		if(resultSet.wasNull()){
			throw new RuntimeException("表/视图" + mark() + "不存在");
		}
		if(resultSet.next()){
			String type = resultSet.getString("TABLE_TYPE");
			this.tableRemarks = resultSet.getString("REMARKS");
			resultSet.close();
			if(!"TABLE".equalsIgnoreCase(type)){
				this.isView = true;
			}
		}else{
			throw new RuntimeException("表/视图" + mark() + "不存在");
		}
	}
	
	
	
	
	protected void buildColumns(DatabaseMetaData dbmd,String tableName,String schema) throws SQLException{
	
		ResultSet columns = this.getColumns(dbmd, tableName, schema);
		if(columns.wasNull()){
			throw new RuntimeException(schema + "." + tableName + "获取列失败");
		}
//		String[] types = new String[1];
		Set<String> colnameSet = new HashSet<String>();
//		List<String> spatialColumnNameList = new ArrayList<String>();
		
		List<String> tmp = new ArrayList<String>();
		
		while (columns.next()) {
			
			String columName = columns.getString(COLUMN_NAME);
	
			colnameSet.add(columName);
			
			Short columType = columns.getShort(DATA_TYPE);
			
			Integer columSize = columns.getInt(COLUMN_SIZE);
			
			Integer digits = 0;
			
			String columTypeName = columns.getString(TYPE_NAME);
			
			Boolean nullable = false;
			
			Boolean withDefault = false;

			if ((columType == 3) || (columType == 2)) {
				digits = columns.getInt(DECIMAL_DIGITS);
			}

			if (columns.getInt(NULLABLE) == 1)
				nullable = true;
			else {
				nullable = false;
			}
			withDefault = columns.getString(COLUMN_DEF) != null;
			String colRemarks = this.getRemarks(columns);
			IColumn column = this.createColumn(columName, columSize, digits, columTypeName, nullable, withDefault, colRemarks);
			this.columnMap.put(column.getColumnName(), column);
			tmp.add(column.getColumnName());
		}
		
		this.columnArray = tmp.toArray(new String[tmp.size()]);
		columns.close();
	}
	
	
	
	

	
	
	@Override
	protected ITable createTable(){
		return new DBTable();
	}
	
	@Override
	public TableSchemaBean toSchemaBean(){
	    TableSchemaBean bean = new TableSchemaBean();
		bean.setCataLog(cataLog);
		bean.setColumnMap(columnMap);
		bean.setHasSequence(hasSequence);
		bean.setIndexMap(indexMap);
		bean.setSchema(schema);
		bean.setTableName(tableName);
		bean.setTableRemarks(tableRemarks);
		return bean;
	}
	
	
	
	
	
}
