package com.northpool.resources.dialect.db.oracle;

import com.northpool.resources.Constants.DATA_BASE_TYPE;
import com.northpool.resources.dialect.db.AbstractSQLDialect;
import com.northpool.resources.dialect.db.SQLDialect;
import com.northpool.resources.dialect.function.sql.StandardSQLFunction;
import com.northpool.resources.type.*;
import org.locationtech.jts.geom.Geometry;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

public class OracleSQLDialect extends AbstractSQLDialect implements SQLDialect {

	public static OracleSQLDialect INSTANCE = new OracleSQLDialect();

	public static final int INTERSECTS = 0;
	public static final int CONTAINS = 1;
	public static final int CROSSES = 2;
	public static final int DISJOINT = 3;
	public static final int EQUALS = 4;
	public static final int OVERLAPS = 5;
	public static final int TOUCHES = 6;
	public static final int WITHIN = 7;
	public static final int FILTER = 8;
	public static final String PAGE_ROW_NUM_MARK = "rownum__";
	
	@Override
	public boolean isSystemMarkFiled(String filedName){
	    if(PAGE_ROW_NUM_MARK.equalsIgnoreCase(filedName)){
	        return true;
	    }else{
	        return false;
	    }
	}
	
	public OracleSQLDialect() {
		super();
		this.name = DATA_BASE_TYPE.oracle.name();
		this.registerTypesAndFunctions();
	}

	@Override
	public DATA_BASE_TYPE getDataSourceType() {
		return DATA_BASE_TYPE.oracle;
	}

	@Override
	public String getLimitString(String sql, boolean hasOffset, List<Object> parameterList,List<Type> inputTypes, int firstResult, int maxResults) {
		sql = sql.trim();
		boolean isForUpdate = false;
		if (sql.toLowerCase().endsWith(" for update")) {
			sql = sql.substring(0, sql.length() - 11);
			isForUpdate = true;
		}

		StringBuilder pagingSelect = new StringBuilder(sql.length() + 100);
		if (hasOffset) {
			pagingSelect.append("select * from ( select row_.*, rownum ").append(PAGE_ROW_NUM_MARK).append(" from ( ");
		} else {
			pagingSelect.append("select * from ( ");
		}
		pagingSelect.append(sql);
		if (hasOffset) {
			pagingSelect.append(" ) row_ ) where ").append(PAGE_ROW_NUM_MARK).append(" <= ? and ").append(PAGE_ROW_NUM_MARK).append(" > ?");
		} else {
			pagingSelect.append(" ) where rownum <= ?");
		}

		if (isForUpdate) {
			pagingSelect.append(" for update");
		}
		parameterList.add(maxResults);
		inputTypes.add(TypeInteger.INSTANCE);
		return pagingSelect.toString();
	}

	@Override
	public String getSelfDesc() {
		return "Oracle方言";
	}
	@Override
	public String getJDBCDriver() {
		return "oracle.jdbc.driver.OracleDriver";
	}
	@Override
	public String createConnectUrl(String url) {
		return "jdbc:oracle:thin:@" + url;
	}

	@Override
	protected void registerTypes() {

		this.registerType(Types.BIGDECIMAL, "NUMBER", "DECIMAL");
		this.registerClassType(Types.BIGDECIMAL, BigDecimal.class);

		this.registerClassType(Types.INTEGER, Integer.class);

		this.registerType(Types.LONG, "LONG");
		this.registerClassType(Types.LONG, Long.class);

		this.registerType(Types.DOUBLE, "FLOAT", "DOUBLE","DOUBLE PRECISION");
		this.registerClassType(Types.DOUBLE, Double.class,Float.class);
		
		
		this.registerType(TypeOracleGeometry.INSTANCE, "SDO_GEOMETRY");
		this.registerClassType(TypeOracleGeometry.INSTANCE, Geometry.class);

		this.registerType(Types.STRING, "VARCHAR2", "CHAR", "NCHAR", "NVARCHAR2");
		this.registerClassType(Types.STRING, String.class);

		this.registerType(Types.TIME_STAMP, "DATE", "TIMESTAMP");
		this.registerClassType(Types.TIME_STAMP, Date.class);

		this.registerType(Types.BOOLEAN, "boolean");
		this.registerClassType(Types.BOOLEAN, Boolean.class);

		this.registerType(TypeBLOB.INSTANCE, "BLOB");
		this.registerClassType(TypeBLOB.INSTANCE, java.sql.Blob.class);
		
		this.registerType(TypeCLOB.INSTANCE, "CLOB");
        this.registerClassType(TypeCLOB.INSTANCE, java.sql.Clob.class);
	}

	protected void registerTypesAndFunctions() {
		this.registerTypes();
		this.registerFunctions();
	}
	@Override
	protected void registerFunctions() {
		Type[] typeIn = new Type[] { TypeOracleGeometry.INSTANCE, TypeOracleGeometry.INSTANCE };
		// 注册空间关系函数
		this.registerFunction("intersects",
				new SpatialRelateFunction("SDO_RELATE", Types.VOID, typeIn, OracleSQLDialect.INTERSECTS));
		this.registerFunction("equals",
				new SpatialRelateFunction("SDO_RELATE", Types.VOID, typeIn, OracleSQLDialect.EQUALS));
		this.registerFunction("contains",
				new SpatialRelateFunction("SDO_RELATE", Types.VOID, typeIn, OracleSQLDialect.CONTAINS));
		this.registerFunction("crosses",
				new SpatialRelateFunction("SDO_RELATE", Types.VOID, typeIn, OracleSQLDialect.CROSSES));
		this.registerFunction("disjoint",
				new SpatialRelateFunction("SDO_RELATE", Types.VOID, typeIn, OracleSQLDialect.DISJOINT));
		this.registerFunction("touches",
				new SpatialRelateFunction("SDO_RELATE", Types.VOID, typeIn, OracleSQLDialect.TOUCHES));
		this.registerFunction("within",
				new SpatialRelateFunction("SDO_RELATE", Types.VOID, typeIn, OracleSQLDialect.WITHIN));
		this.registerFunction("mbr_intersects",
				new SpatialRelateFunction("SDO_FILTER", Types.VOID, typeIn, OracleSQLDialect.FILTER));
	}

	public class SpatialRelateFunction extends StandardSQLFunction {

		private int spatialRelation;

		public SpatialRelateFunction(String name, Type returnType, Type[] argumentsType, int spatialRelation) {
			super(name, argumentsType, returnType);
			this.spatialRelation = spatialRelation;
		}
		@Override
		public String render(List<?> arguments) {
			String mask = "";
			//boolean negate = false;
			switch (spatialRelation) {
			case OracleSQLDialect.INTERSECTS:
				mask = "ANYINTERACT";
				break;
			case OracleSQLDialect.CONTAINS:
				mask = "CONTAINS+COVERS";
				break;
			case OracleSQLDialect.CROSSES:
				throw new UnsupportedOperationException("Oracle Spatial does't have equivalent CROSSES relationship");
			case OracleSQLDialect.DISJOINT:
				mask = "ANYINTERACT";
				//negate = true;
				break;
			case OracleSQLDialect.EQUALS:
				mask = "EQUAL";
				break;
			case OracleSQLDialect.OVERLAPS:
				mask = "OVERLAPBDYDISJOINT+OVERLAPBDYINTERSECT";
				break;
			case OracleSQLDialect.TOUCHES:
				mask = "TOUCH";
				break;
			case OracleSQLDialect.WITHIN:
				mask = "INSIDE+COVEREDBY";
				break;
			case OracleSQLDialect.FILTER:
				return getSpatialFilterExpression(arguments);
			default:
				throw new IllegalArgumentException("undefined SpatialRelation passed (" + spatialRelation + ")");
			}
			StringBuffer buffer;
			buffer = new StringBuffer("SDO_RELATE(");
			/*if (negate) {
				buffer = new StringBuffer("CASE WHEN SDO_RELATE(");
			} else {
				buffer = new StringBuffer("SDO_RELATE(");
			}*/

			for (int i = 0; i < arguments.size(); i++) {
				buffer.append(arguments.get(i));
				if (i < arguments.size() - 1) {
					buffer.append(", ");
				}
			}

			buffer.append(",'mask=" + mask + "') = 'TRUE'");
			/*if (negate) {
				buffer.append(" = 'TRUE' THEN 'FALSE' ELSE 'TRUE' END");
			}*/

			return buffer.toString();
		}

		public String getSpatialFilterExpression(List<?> arguments) {
			StringBuffer buffer = new StringBuffer("SDO_FILTER(");
			for (int i = 0; i < arguments.size(); i++) {
				buffer.append(arguments.get(i));
				if (i < arguments.size() - 1) {
					buffer.append(", ");
				}
			}
			buffer.append(") = 'TRUE'");
			return buffer.toString();
		}

		@Override
		public Boolean isSpatial() {
			return true;
		}

	}

	@Override
	public Boolean markForTableNameAndColumnName() {
		// TODO Auto-generated method stub
		return true;
	}

    /* (non-Javadoc)
     * @see com.northpool.resources.dialect.db.SQLDialect#getDefaultSchema()
     */
    @Override
    public String getDefaultSchema() {
        // TODO Auto-generated method stub
        return null;
    }

    
   

}