package com.northpool.resources.sql.jdbc;

import com.northpool.resources.datasource.db.DbDataSource;
import com.northpool.resources.datasource.db.JDBCPoolManager;
import com.northpool.resources.datatable.Scroll;
import com.northpool.resources.dialect.db.SQLDialect;
import com.northpool.resources.type.Type;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.sql.*;
import java.util.*;

public class AbstractsSQLCell<T> {
	
	protected SQLDialect dialect;
	//protected Connection connection;
	protected SQLTransformer<T> transformer;
	protected String sql;
	protected Map<String,Type> returnTypeMap = new HashMap<String,Type>();
	protected List<Type> inputTypes = new ArrayList<>();
	protected DbDataSource dbDataSource;
	
	protected boolean ignoreWarnings = false;
	
	protected static final Logger logger = LoggerFactory.getLogger(AbstractsSQLCell.class);
	
	
	
	public AbstractsSQLCell(DbDataSource dbDataSource,SQLDialect dialect,String sql) {
		this.dialect = dialect;
		this.dbDataSource = dbDataSource;
		
		//this.connection = connection;
		//this.jdbcTemplate = jdbcTemplate;
		this.sql = sql;
	}
	
	public void setInputTypes(Type[] inputTypes){
	    if(inputTypes != null){
	        this.inputTypes = new ArrayList<Type>(Arrays.asList(inputTypes));
	    }
	}
	
	
	
	
	/*protected void applyStatementSettings(Statement stmt) throws SQLException {
        int fetchSize = getFetchSize();
        if (fetchSize != -1) {
            stmt.setFetchSize(fetchSize);
        }
        int maxRows = getMaxRows();
        if (maxRows != -1) {
            stmt.setMaxRows(maxRows);
        }
        DataSourceUtils.applyTimeout(stmt, getDataSource(), getQueryTimeout());
    }*/
	
	protected <RETURN> RETURN execute(PreparedStatementCreator psc,PreparedStatementCallback<RETURN> action) throws SQLException{
	    assert psc != null;
	    assert action != null;
	    if (logger.isTraceEnabled()) {
	        logger.trace("Executing prepared SQL statement" + (sql != null ? " [" + sql + "]" : ""));
	    }
	    Connection con;
        try {
            con = getConnection(dbDataSource);
        } catch (CannotGetJdbcConnectionException e) {
            // TODO Auto-generated catch block
            throw new SQLException(e);
        }
        PreparedStatement ps = null;
        try {
            ps = psc.createPreparedStatement(con);
            RETURN result = action.doInPreparedStatement(ps);
            handleWarnings(ps);
            return result;
        }
        catch (SQLException ex) {
         
            psc = null;
            closeStatement(ps);
            ps = null;
            releaseConnection(con, this.dbDataSource);
            con = null;
            throw new SQLException(sql, ex);
        }
        finally {
            
            closeStatement(ps);
            releaseConnection(con, this.dbDataSource);
        }

	}
	
	protected void setValues(PreparedStatement ps,Object[] args,Type[] types) throws PreparedStatementSetValueException{
        int index = 1;
        for(int i = 0 ; i < args.length ; i ++){
            Type t = types[i];
            try {
                t.preparedStatementSetValue(ps, args[i], index);
            } catch (Exception e) {
                // TODO Auto-generated catch block
                throw new PreparedStatementSetValueException(e);
            }
            index++;
        }
      
    }
	
	
	
	private static Connection getConnection(DbDataSource dbDataSource) throws  CannotGetJdbcConnectionException{
	    DataSource dataSource = Transactions.getThreadDataSource(dbDataSource); 
        if(dataSource == null){
            dataSource = JDBCPoolManager.getInstance().getPool(dbDataSource);
        }
        try{
            return dataSource.getConnection();
        }
        catch (SQLException ex) {
            throw new CannotGetJdbcConnectionException("Failed to obtain JDBC Connection", ex);
        }
        catch (IllegalStateException ex) {
            throw new CannotGetJdbcConnectionException("Failed to obtain JDBC Connection: ", ex);
        }
	}
	
	public static void closeStatement(Statement stmt) {
        if (stmt != null) {
            try {
                stmt.close();
            }
            catch (SQLException ex) {
                logger.trace("Could not close JDBC Statement", ex);
            }
            catch (Throwable ex) {
                logger.trace("Unexpected exception on closing JDBC Statement", ex);
            }
        }
    }
	
	public static void releaseConnection(Connection con, DbDataSource dbDataSource){
	    if (con == null) {
            return;
        }
	    try {
	        DataSource dataSource = Transactions.getThreadDataSource(dbDataSource);
	        //没有线程占用的连接,自动释放
	        if(dataSource == null){
	            con.close();
	        }
	       
        }
        catch (SQLException ex) {
            logger.debug("Could not close JDBC Connection", ex);
        }
        catch (Throwable ex) {
            logger.debug("Unexpected exception on closing JDBC Connection", ex);
        }
	}
	
	
	
	protected Scroll<T> executeScroll(PreparedStatementCreator psc,Object[] args,Type[] types) throws SQLException {

        assert psc != null; 
        Connection con;
        try {
            con = getConnection(dbDataSource);
        } catch (CannotGetJdbcConnectionException e) {
            // TODO Auto-generated catch block
            throw new SQLException(e);
        }
        PreparedStatement ps = null;
        if (logger.isTraceEnabled()) {
            logger.trace("Executing prepared SQL statement" + (sql != null ? " [" + sql + "]" : ""));
        }
        try {
            ps = psc.createPreparedStatement(con);
        //    DataSourceUtils.applyTimeout(ps, jdbcTemplate.getDataSource(), jdbcTemplate.getQueryTimeout());
            ResultSet rs = null;
            Scroll<T> _scroll;
            
            this.setValues(ps, args,types);
            rs = ps.executeQuery();
            _scroll = this.transformer.extractDataScroll(dialect, rs, returnTypeMap, psc, this.dbDataSource, con, ps);
           
            handleWarnings(ps);
            return _scroll;
        }
        catch (SQLException | PreparedStatementSetValueException ex) {

            psc = null;
            closeStatement(ps);
            ps = null;
            releaseConnection(con,  this.dbDataSource);
            con = null;
            if(ex instanceof SQLException){
                throw (SQLException)ex;
            }
            throw new SQLException(sql, ex);
        }
        finally {
           
            /*JdbcUtils.closeStatement(ps);
            DataSourceUtils.releaseConnection(con, getDataSource());*/
        }
    }

    
   
    
    protected void handleWarnings(Statement stmt) throws SQLException {
        if (ignoreWarnings) {
            if (logger.isDebugEnabled()) {
                SQLWarning warningToLog = stmt.getWarnings();
                while (warningToLog != null) {
                    logger.debug("SQLWarning ignored: SQL state '" + warningToLog.getSQLState() + "', error code '" +
                            warningToLog.getErrorCode() + "', message [" + warningToLog.getMessage() + "]");
                    warningToLog = warningToLog.getNextWarning();
                }
            }
        }
        
    }

  
	
}
