package com.northpool.resources.sql.jdbc;

import com.northpool.exception.UException;
import com.northpool.resources.datasource.db.DbDataSource;
import com.northpool.resources.datatable.Scroll;
import com.northpool.resources.datatable.dao.DataAccessException;
import com.northpool.resources.dialect.db.SQLDialect;
import com.northpool.resources.sql.Query;
import com.northpool.resources.type.Type;
import com.northpool.resources.type.TypeInteger;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;


public class SQLQuery<T> extends AbstractsSQLCell<T> implements Query<T,SQLTransformer<T>> {
	
	Logger logger = LoggerFactory.getLogger(SQLQuery.class);

	
	protected int firstResult = -1;
	protected int maxResults = -1;
	protected static final Integer default_Fetch_Size = 50;
	protected List<Object> parameterList = new ArrayList<Object>();
	protected Integer fetchSize;

	
	
	public SQLQuery(DbDataSource dbDataSource,SQLDialect dialect,String sql) {
		super(dbDataSource,dialect,sql);
	}

	
 
	public SQLQuery<T> addScalar(Map<String,Type> returnTypeMap) { 
		this.returnTypeMap = returnTypeMap;
		return this;
	}
	
	
	@Override
	public SQLQuery<T>  addScalar(String key, Type type) {
		this.returnTypeMap.put(key, type);
		return this;
	}

	@Override
	public SQLQuery<T>  setParameter(int index, Object arg) {
		this.parameterList.add(index, arg);
		return this;
	}
	
	public SQLQuery<T>  setParameters(Object[] args) {
		this.parameterList.addAll(Arrays.asList(args));
		return this;
	}
	
	
	

	@Override
	public SQLQuery<T>  setFirstResult(Integer firstResult) {
	    if(firstResult != null){
	        this.firstResult = firstResult;
	    }
		return this;
	}

	@Override
	public SQLQuery<T>  setMaxResults(Integer maxResults) {
	    if(maxResults != null){
	        this.maxResults = maxResults;
	    }
		return this;
	}

	
	
	protected void printSQL(){
		//System.out.println(this.sql);
		//System.out.println(StringUtils.join(this.parameterList, ","));
		this.logger.debug(this.sql);
		this.logger.debug("use parameters:" + StringUtils.join(this.parameterList, ","));
	}

	
	
	 
	
	@Override
	public List<T> list() throws DataAccessException {
		this.createLimit();
		this.printSQL();
	
		final int fetchsize = this.fetchSize == null ? default_Fetch_Size : fetchSize;
		Object[] args = this.parameterList.toArray(new Object[this.parameterList.size()]);
		
		List<T> iListReturn;
        try {
            iListReturn = this.execute((Connection con) ->{
            	return this.prepareQueryStatement(con,fetchsize);
            }, (PreparedStatement ps) ->{
            	ResultSet rs = null;
            	List<T> iList = null;
            	try {
            		this.setValues(ps, args,this.inputTypes.toArray(new Type[this.inputTypes.size()]));
            		rs = ps.executeQuery();
            		iList = this.transformer.extractData(this.dialect, rs, this.returnTypeMap);
            	} catch (Exception e) {
            		UException.printStackTrace(e);
            		throw new SQLException(e);
            	} finally {
            		rs.close();
            	}
            	return iList;
            });
            return iListReturn;
        } catch (SQLException e) {
            throw new DataAccessException(e);
        }
		
		
	}
	
	
	
	
	
   

    protected final PreparedStatement prepareQueryStatement(Connection connection,Integer fetchSize) throws SQLException{
		PreparedStatement preparedStatement = connection.prepareStatement(this.sql);
		if(fetchSize != null){
		    if(fetchSize <= 0){
		        fetchSize = null;
		    }else{
		        dialect.setFetchSize(connection,preparedStatement,fetchSize);
		    }
		}
		return preparedStatement;
	}
	
	protected void createLimit(){
		boolean useLimit = false;
		boolean hasOffset = false;
		if(this.maxResults > 0){
			if(this.firstResult < 0){
				this.firstResult = 0;
			}else{
				hasOffset = true;
			}
			useLimit = true;
		}else{
			if(firstResult > 0){
				hasOffset = true;
			}
		}
		if(useLimit){
			this.sql = this.dialect.getLimitString(sql, hasOffset,this.parameterList,this.inputTypes,this.firstResult,this.maxResults);
			
			if(hasOffset){
				this.parameterList.add(this.firstResult);
				this.inputTypes.add(TypeInteger.INSTANCE);
			}
		}
		return;
	}

   
    @Override
    public SQLQuery<T> setFetchSize(Integer fetchSize) {
        this.fetchSize = fetchSize;
        return this;
    }

  
    @Override
    public Scroll<T> scroll() throws DataAccessException {
        this.createLimit();
        this.printSQL();
        final int fetchsize = this.fetchSize == null ? default_Fetch_Size : this.fetchSize;
        Object[] args = this.parameterList.toArray(new Object[this.parameterList.size()]);
        Scroll<T> scroll;
        try {
            scroll = this.executeScroll((Connection con) ->{
                return this.prepareQueryStatement(con,fetchsize);
            },args,this.inputTypes.toArray(new Type[this.inputTypes.size()]));
        } catch (SQLException e) {
            // TODO Auto-generated catch block
           throw new DataAccessException(e);
        }
        return scroll;
        
    }


    @Override
    public SQLQuery<T> setResultTransformer(SQLTransformer<T> transformer) {
        // TODO Auto-generated method stub
        this.transformer = transformer;
        return this;
    }
    
    
  
}
