package com.northpool.spatial.oracle;



import java.io.IOException;
import java.sql.SQLException;

import com.northpool.commons.util.BuilderCreator;
import com.northpool.commons.util.DoubleBuilder;
import com.northpool.spatial.AGeomDecoder;
import com.northpool.spatial.ByteOrderDataInStream;
import com.northpool.spatial.Constants;
import com.northpool.spatial.GeomDecoder;
import com.northpool.spatial.GeometryInfo;
import com.northpool.spatial.Constants.GEO_TYPE;
import com.northpool.spatial.Constants.SPATIAL_TYPE;
import com.northpool.spatial.geofeature.GeoBuffer;
import com.northpool.spatial.geofeature.GeoPart.RING_TYPE;

import oracle.jdbc.internal.OracleArray;
import oracle.jdbc.internal.OracleStruct;
import oracle.sql.Datum;
@AGeomDecoder(type = SPATIAL_TYPE.sdo)
public class OracleDecoder implements GeomDecoder<Object>{
	
	protected final static int UNKNOWN = 00;
	protected final static int POINT = 01;
	protected final static int LINESTRING = 02;
	protected final static int POLYGON = 03;
	protected final static int COLLECTION = 04;
	protected final static int MULTIPOINT = 05;
	protected final static int MULTILINESTRING = 06;
	protected final static int MULTIPOLYGON = 07;
	
	
	
	
	protected static OracleDecoder _decoder = new OracleDecoder();
	
	public static OracleDecoder decoder(){
		return _decoder;
	}
	
	public GeometryInfo buildGeometryTypeAndZM(Integer sdoGtype) throws IOException{
	    GeometryInfo info = new GeometryInfo();
	    String gtype = sdoGtype.toString();
	    if(gtype.length() != 4){
	        throw new RuntimeException("sdoGtype" + sdoGtype + "长度不为4");
	    }
	    if(gtype.charAt(0) == '3'){
	        info.hasZ = true;
	    }
	    if(gtype.charAt(0) == '4'){
	        info.hasM = true;
	    }
	    info.geometryType = Integer.valueOf(gtype.substring(2, 4));
	    return info;
	}
	
	@Override
	public GeoBuffer toGeoBuffer(Object value) throws Exception{
		if (value instanceof GeoBuffer){
			return (GeoBuffer) value;
		}else {
			OracleStruct struct = (OracleStruct) value;
			Datum[] data = null;

			int type = 2000;

			data = struct.getOracleAttributes();

			type = data[0].intValue();

			GeometryInfo info = buildGeometryTypeAndZM(type);

			Integer srid = null;

			if(data[1] != null){
				srid = data[1].intValue();
			}
			int size = Long.valueOf(struct.getLength()).intValue();

			Constants.GEO_TYPE geoType = is(info.geometryType);

			int inputDimension =  info.hasZ() ? 3 : 2;

			GeoBuffer geo = new GeoBuffer(geoType,srid,inputDimension);

			switch (geoType) {
				case POINT:
					this.dealWithPoint(struct, geo , info);

					break;
				case LINESTRING:
					OracleArray datasarray = (OracleArray) data[4];

					this.dealWithLineString(datasarray.getDoubleArray(), geo , info);
					break;
				case POLYGON:
					this.dealWithPolygon(data, geo , info);
					break;

				case MULTIPOINT:
					this.dealWithMultiPoint(data, geo , info);
					break;
				case MULTILINESTRING:
					this.dealWithMultiLine(data, geo , info);
					break;
				case MULTIPOLYGON:
					this.dealWithMultiPolygon(data, geo , info);
					break;

				case COLLECTION:
				default:
					throw new RuntimeException("不支持的几何类型：" + geoType.name());

			}
			return geo;
		}
		

	}
	
	
	public GEO_TYPE is(int type) {
		switch (type) {
			case POINT:
				return GEO_TYPE.POINT;
			case LINESTRING:
				return GEO_TYPE.LINESTRING;
			case POLYGON:
				return GEO_TYPE.POLYGON;
			case COLLECTION:
				return GEO_TYPE.COLLECTION;
			case MULTIPOINT:
				return GEO_TYPE.MULTIPOINT;
			case MULTILINESTRING:
				return GEO_TYPE.MULTILINESTRING;
			case MULTIPOLYGON:
				return GEO_TYPE.MULTIPOLYGON;
			}
		throw new RuntimeException("未知的数据类型" + type);
	}
	

	
	
	
	
	
	
	
	
	
	
	
	// 点处理
	private void dealWithPoint(OracleStruct struct,GeoBuffer geo,GeometryInfo info) throws SQLException {
		
		
		
		double x = 0;
		double y = 0;
		double z = 0;
		Object[] data = null;
		
		data = struct.getAttributes();
		OracleStruct point = (OracleStruct) data[2];
		
		if(point == null){
			OracleArray datasarray = (OracleArray) data[4];
			double[] datasords = datasarray.getDoubleArray();
			x = datasords[0];
			y = datasords[1];
			if(info.hasZ){
			    z = datasords[2];
			}
		}else{
			Datum[] xyz = point.getOracleAttributes();
			x = xyz[0].doubleValue();
			y = xyz[1].doubleValue();
			if(info.hasZ){
                z = xyz[2].doubleValue();
            }
		}
		geo.addPoint(x, y, z);

	}
	
	
	// 多part-点处理
	private void dealWithMultiPoint(Datum[] data,GeoBuffer geo,GeometryInfo info) throws Exception {
		double x = 0;
		double y = 0;
		double z = 0;
		double[] dataords = null;
		OracleArray datasarray = (OracleArray) data[4];
		dataords = datasarray.getDoubleArray();
		int nums = dataords.length / 2;

		for(int i = 0 ; i < nums; i++){
			x = dataords[2*i];
			y = dataords[2*i + 1];
			geo.addPoint(x, y, z);
		}
	}
	
	
	private double[] readPointArray(double[] data,DoubleBuilder doubleBuilder,GeometryInfo info) throws SQLException {
	    double[] datasords = data;
       
        double minX = 0d;
        double minY = 0d;
        double maxX = 0d;
        double maxY = 0d;
        int count = datasords.length;
        int inputDimension =  info.hasZ() ? 3 : 2;
        for (int i = 0; i < count / inputDimension; i++) {
            double x = datasords[i * inputDimension];
            double y = datasords[i * inputDimension + 1];
            if(x > maxX){
                maxX = x;
            }
            if(x < minX){
                minX = x;
            }
            if(y > maxY){
                maxY = y;
            }
            if(y < minY){
                minY = y;
            }
            doubleBuilder.append(x);
            doubleBuilder.append(y);
            if(info.hasZ){
                double z =  datasords[i * inputDimension + 2];
                doubleBuilder.append(z);
            }
            if(info.hasM){
                @SuppressWarnings("unused")
                double m = datasords[i * inputDimension + 3];
            }
        }
        doubleBuilder.trimToSize();
        return new double[]{minX,minY,maxX,maxY};
    }
	
	
	
	private void dealWithLineString(double[] data,GeoBuffer geo,GeometryInfo info) throws SQLException {
	    DoubleBuilder doubleBuilder = BuilderCreator.createDouble(data.length);
        double[] bbox = this.readPointArray(data,doubleBuilder,info);
        geo.addLinePart(doubleBuilder, bbox[0], bbox[1], bbox[2], bbox[3]);
	}
	
	// 多part-线处理
	private void dealWithMultiLine(Datum[] data,GeoBuffer geo,GeometryInfo info) throws SQLException {
		// 解析几何结构
		double[] dataords = null;
		int[] struct = null;
		OracleArray structarray = (OracleArray) data[3];
		OracleArray datasarray = (OracleArray) data[4];
		struct = structarray.getIntArray();
		dataords = datasarray.getDoubleArray();
		int previousIndex = 0;
		for(int i = 0 ; i < struct.length; i = i+3){
			int beginIndex = struct[i] -1;
			int inout = struct[i+1];
			int endIndex = -1;
			if(i + 3 >= struct.length){
			
			}else{
				endIndex = struct[i+3] -1;
			}
			if(inout == 1003 || inout == 2 || inout == 2003){
				
			    previousIndex = inout;
                
                double[] dataDouble = null;
                int length = 0;
                if(endIndex != -1){
                    length = endIndex - beginIndex;
                    
                }else{
                    length = dataords.length - beginIndex;
                }
                dataDouble = new double[length];
                System.arraycopy(dataords, beginIndex, dataDouble, 0, length);
                DoubleBuilder doubleBuilder = BuilderCreator.createDouble(length);
                double[] bbox = this.readPointArray(dataDouble,doubleBuilder,info);
                if(inout == 2){
                    geo.addLinePart(doubleBuilder, bbox[0], bbox[1], bbox[2], bbox[3]);
                }else if(inout == 1003){
                    geo.addPolygonPart(doubleBuilder, bbox[0], bbox[1], bbox[2], bbox[3],RING_TYPE.outside);
				}else{
				    geo.addPolygonPart(doubleBuilder, bbox[0], bbox[1], bbox[2], bbox[3],RING_TYPE.inside);
				}
				
		       
			//	geo.add(doubleBuilder);
				
			}
		}   
		//geo.polygonEnd();
	}
	
	
	// 多边形处理
	private void dealWithPolygon(Datum[] data,GeoBuffer geo,GeometryInfo info) throws SQLException {
		this.dealWithMultiLine(data, geo,info);
		
	}
	
	private void dealWithMultiPolygon(Datum[] data,GeoBuffer geo,GeometryInfo info) throws SQLException {
		this.dealWithMultiLine(data, geo,info);
	}
	
	

	
	
}