package com.northpool.resources.sql.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.northpool.resources.datasource.IDataSource;
import com.northpool.resources.datasource.db.DbDataSource;
import com.northpool.resources.datasource.db.JDBCPoolManager;
import com.northpool.resources.datatable.dao.IScroll;
import com.northpool.resources.dialect.IResourcesDataInput;
import com.northpool.resources.dialect.sql.ISQLDialect;
import com.northpool.type.Type;

public class AbstractsJDBCCell<T> {
	
	protected ISQLDialect dialect;
	protected IJDBCTransformer<T> transformer;
	protected String sql;
	protected Map<String,Type> returnTypeMap = new HashMap<String,Type>();
	protected List<Type> inputTypes = new ArrayList<>();
	protected DbDataSource dbDataSource;
	
	protected boolean ignoreWarnings = false;
	
	protected static final Logger logger = LoggerFactory.getLogger(AbstractsJDBCCell.class);
	
	protected IResourcesDataInput<PreparedStatement,Integer> dataInput;
	
	public AbstractsJDBCCell(IDataSource dataSource,ISQLDialect dialect,String sql) {
		this.dialect = dialect;
		if(!(dataSource instanceof DbDataSource)){
		    throw new RuntimeException("数据源不能支持JDBC操作,type:" + dataSource.getDataSourceType());
		}
		this.dbDataSource = (DbDataSource)dataSource;
		this.dataInput = dialect.getResourcesDataDataInput();
		
		this.sql = sql;
	}
	
	public void setInputTypes(Type[] inputTypes){
	    if(inputTypes != null){
	        this.inputTypes = new ArrayList<Type>(Arrays.asList(inputTypes));
	    }
	}
	
	
	
	
	
	protected <RETURN> RETURN execute(PreparedStatementCreator psc,PreparedStatementCallback<RETURN> action) throws SQLException{
	    assert psc != null;
	    assert action != null;
	    if (logger.isTraceEnabled()) {
	        logger.trace("Executing prepared SQL statement" + (sql != null ? " [" + sql + "]" : ""));
	    }
	    Connection con;
        try {
            con = getConnection(dbDataSource);
        } catch (CannotGetJdbcConnectionException e) {
            // TODO Auto-generated catch block
            throw new SQLException(e);
        }
        PreparedStatement ps = null;
        try {
            ps = psc.createPreparedStatement(con);
            RETURN result = action.doInPreparedStatement(ps);
            handleWarnings(ps);
            return result;
        }
        catch (SQLException ex) {
         
            psc = null;
            closeStatement(ps);
            ps = null;
            releaseConnection(con, this.dbDataSource);
            con = null;
            throw new SQLException(sql, ex);
        }
        finally {
            
            closeStatement(ps);
            releaseConnection(con, this.dbDataSource);
        }

	}
	
	protected void setValues(PreparedStatement ps,Object[] args,Type[] types) throws PreparedStatementSetValueException{
        int index = 1;
        for(int i = 0 ; i < args.length ; i ++){
            Type t = types[i];
            try {
                Object valueInput = t.toType(args[i]);
                dialect.getResourcesDataDataInput().input(ps, t, valueInput, index);
            } catch (Exception e) {
                // TODO Auto-generated catch block
                throw new PreparedStatementSetValueException(e);
            }
            index++;
        }
      
    }
	
	
	
	private static Connection getConnection(DbDataSource dbDataSource) throws  CannotGetJdbcConnectionException{
	    DataSource dataSource = Transactions.getThreadDataSource(dbDataSource); 
        if(dataSource == null){
            dataSource = JDBCPoolManager.getInstance().getPool(dbDataSource);
        }
        try{
            return dataSource.getConnection();
        }
        catch (SQLException ex) {
            throw new CannotGetJdbcConnectionException("Failed to obtain JDBC Connection", ex);
        }
        catch (IllegalStateException ex) {
            throw new CannotGetJdbcConnectionException("Failed to obtain JDBC Connection: " + ex.getMessage());
        }
	}
	
	public static void closeStatement(Statement stmt) {
        if (stmt != null) {
            try {
                stmt.close();
            }
            catch (SQLException ex) {
                logger.trace("Could not close JDBC Statement", ex);
            }
            catch (Throwable ex) {
                logger.trace("Unexpected exception on closing JDBC Statement", ex);
            }
        }
    }
	
	public static void releaseConnection(Connection con, DbDataSource dbDataSource){
	    if (con == null) {
            return;
        }
	    try {
	        DataSource dataSource = Transactions.getThreadDataSource(dbDataSource);
	        //没有线程占用的连接,自动释放
	        if(dataSource == null){
	            con.close();
	        }
	       
        }
        catch (SQLException ex) {
            logger.debug("Could not close JDBC Connection", ex);
        }
        catch (Throwable ex) {
            logger.debug("Unexpected exception on closing JDBC Connection", ex);
        }
	}
	
	
	
	protected IScroll<T> executeScroll(PreparedStatementCreator psc,Object[] args,Type[] types) throws SQLException {

        assert psc != null; 
        Connection con;
        try {
            con = getConnection(dbDataSource);
        } catch (CannotGetJdbcConnectionException e) {
            // TODO Auto-generated catch block
            throw new SQLException(e);
        }
        PreparedStatement ps = null;
        if (logger.isTraceEnabled()) {
            logger.trace("Executing prepared SQL statement" + (sql != null ? " [" + sql + "]" : ""));
        }
        try {
            ps = psc.createPreparedStatement(con);
        //    DataSourceUtils.applyTimeout(ps, jdbcTemplate.getDataSource(), jdbcTemplate.getQueryTimeout());
            ResultSet rs = null;
            IScroll<T> _scroll;
            
            this.setValues(ps, args,types);
            rs = ps.executeQuery();
            _scroll = this.transformer.extractDataScroll(dialect, rs, returnTypeMap, new JDBCScrollStatement(this.dbDataSource, con, ps));
           
            handleWarnings(ps);
            return _scroll;
        }
        catch (SQLException | PreparedStatementSetValueException ex) {

            psc = null;
            closeStatement(ps);
            ps = null;
            releaseConnection(con,  this.dbDataSource);
            con = null;
            if(ex instanceof SQLException){
                throw (SQLException)ex;
            }
            throw new SQLException(sql, ex);
        }
        finally {
           
            /*JdbcUtils.closeStatement(ps);
            DataSourceUtils.releaseConnection(con, getDataSource());*/
        }
    }

    
   
    
    protected void handleWarnings(Statement stmt) throws SQLException {
        if (ignoreWarnings) {
            if (logger.isDebugEnabled()) {
                SQLWarning warningToLog = stmt.getWarnings();
                while (warningToLog != null) {
                    logger.debug("SQLWarning ignored: SQL state '" + warningToLog.getSQLState() + "', error code '" +
                            warningToLog.getErrorCode() + "', message [" + warningToLog.getMessage() + "]");
                    warningToLog = warningToLog.getNextWarning();
                }
            }
        }
        
    }
    
   
   
  
	
}
