package com.northpool.service.manager.abstractclass;

import com.northpool.bean.Idable;
import com.northpool.bean.Jsonable;
import com.northpool.bean.JsonableBuilder;
import com.northpool.commons.event.EventContainer;
import com.northpool.commons.event.Listener;
import com.northpool.resources.command.QueryFilter;
import com.northpool.resources.datatable.dao.IScroll;
import com.northpool.resources.exception.IdFieldValueEmptyException;
import com.northpool.service.client.Client;
import com.northpool.service.config.IDocumentAble;
import com.northpool.service.config.IVersionAble;
import com.northpool.service.dao.AbstractZkDao;
import com.northpool.service.dao.IMongoDao;
import com.northpool.structure.queryhashtable.QueryHashTableHeap;
import org.apache.curator.framework.recipes.cache.PathChildrenCache;

import java.util.Date;
import java.util.List;

public abstract class AbstractManager<T extends Jsonable & Idable<String> & IDocumentAble, Builder extends JsonableBuilder<T> & DocumentableBuilder<T>> {
	protected QueryHashTableHeap<String,T> table;
	protected String idFieldName;
	protected Client client;
	protected Boolean readOnly = false;
	protected String managerRoot;	
	protected PathChildrenCache HolderCache;
	protected static final String INFO_FIELD = "data";
	protected Builder beanBuilder;
	protected Boolean isReady = false;
	protected EventContainer<Listener> eventContainer = new EventContainer<Listener>();

	protected AbstractZkDao zkDao;

	protected IMongoDao<T> mongoDao;
	
	public AbstractManager(String idFieldName,Builder beanBuilder, String managerRoot){
		this.idFieldName = idFieldName;
		this.managerRoot = managerRoot;
		this.beanBuilder = beanBuilder;

	}
	

	public void init(){
		try {
			if (this.client.getZoo() != null){
				this.table = new QueryHashTableHeap<String,T>(this.idFieldName);
			}else {

			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		this.isReady = true;
	}
	


	protected void saveBeanToHashTable(T bean){
		if (this.table == null){
			return;
		}
		try {
			this.table.insert(bean);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}
	
	protected void removeBeanToHashTable(String pk){
		if (this.table == null){
			return;
		}
		this.table.remove(pk);
	}
	
	protected void updateBeanToHashTable(T bean){
		if (this.table == null){
			return;
		}
		try {
			this.table.update(bean);
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}
	

	
	public Boolean exists(String id) throws ZKException {
		if (this.zkDao != null){
			if (this.table.get(id) != null){//内存里有则认为有，内存没有再请求zk
				return true;
			}
			boolean isExists = this.zkDao.exists(id);
			return isExists;
		}else if (this.mongoDao != null){
			return this.mongoDao.exists(id);

		}
		return true;

	}
	

	protected void isReady() throws NotReadyException{
		if(!this.isReady){
			throw new NotReadyException();
		}
	}
	
	/**
	 * 处理add事件
	 * @param t
	 * @throws ExistsIdException
	 * @throws ZKException
	 * @throws IdFieldValueEmptyException
	 * @throws NotReadyException
	 */
	protected void doAdd(T t) throws ExistsIdException,ZKException, IdFieldValueEmptyException, NotReadyException{
		this.isReady();
		String id = t.getId();
		if(id == null){
			throw new IdFieldValueEmptyException(this.idFieldName, t);
		}
		if (this.exists(id)){
			throw new ExistsIdException(id);
		}

		if (this.mongoDao != null){
			this.mongoDao.insertOne(t);
		}

		if (this.zkDao != null){
			this.saveBeanToHashTable(t);
			try {
				this.zkDao.saveBeanTozk(t);
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				this.removeBeanToHashTable(id);
				throw new ZKException(e);
			}
		}

		eventContainer.syncFireEvent(EventMessage.EVENT_TYPE.register.name(), id,t);
	}


	/**
	 * 响应更新事件
	 * @param t
	 * @throws ZKException
	 * @throws IdFieldValueEmptyException
	 * @throws NotReadyException
	 * @throws NotFoundException
	 */
	protected void doUpdate(T t) throws Exception {
		this.isReady();
		String id = t.getId();
		if(id == null){
			throw new IdFieldValueEmptyException(this.idFieldName, t);
		}
		if (!this.exists(id)){
			throw new NotFoundException(id);
		}

		if (t instanceof IVersionAble){//有版本号的，每次更新修改版本号
			IVersionAble versionAble = (IVersionAble) t;
			versionAble.setVersion(new Date().getTime() + "");
		}

		if (this.zkDao != null){
			this.table.update(t);
			this.zkDao.updateBeanTozk(t);
		}
		if (this.mongoDao != null){
			this.mongoDao.updateOne(id, t);
		}
		eventContainer.syncFireEvent(EventMessage.EVENT_TYPE.update.name(), id,t);
	}

	protected void doRemove(String id) throws ZKException, NotReadyException, NotFoundException{
		this.isReady();
		if (!this.exists(id)){
			throw new NotFoundException(id);
		}
		if (this.zkDao != null){
			try {
				this.zkDao.zkRemoveBean(id);
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				throw new ZKException(e);
			}
			this.removeBeanToHashTable(id);
		}
		if (this.mongoDao != null){
			this.mongoDao.deleteone(id);
		}
		eventContainer.syncFireEvent(EventMessage.EVENT_TYPE.unRegister.name(), id, this.get(id));
	}

	public void on(EventMessage.EVENT_TYPE type,EventMessage<T> e){
		eventContainer.on(type.name(), (Object ...parameter) -> {
			e.fire((String)parameter[0],(T)parameter[1]);
		});
	}
	
	public List<T> list(QueryFilter queryFilter) throws Exception {
		if (this.table != null){
			return this.table.query(queryFilter);
		}else {
			return null;
		}
	}
	
	public IScroll<T> scroll(QueryFilter queryFilter){
		if (this.table != null){
			return this.table.scroll(queryFilter);
		}else {
			return null;
		}
	}
	
	public T get(String id) {
		// TODO Auto-generated method stub
		if (this.table != null){
			return this.table.get(id);
		}else if (this.mongoDao != null){
			T bean = this.mongoDao.findone(id);
            return bean;
        }else {
				return null;
        }
    }

    public String getJSON(String id){
		T bean = this.get(id);
		if (bean != null){
			return bean.toJson();
		}
		return null;
	}

	public void setClient(Client client) {
		this.client = client;
	}

	public void setReadOnly(boolean readOnly){
		this.readOnly = readOnly;
	}

}

