package com.northpool.service.dao;

import com.mongodb.WriteConcern;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Indexes;
import com.northpool.bean.Idable;
import com.northpool.bean.Jsonable;
import com.northpool.bean.JsonableBuilder;
import com.northpool.service.config.IDocumentAble;
import com.northpool.service.manager.abstractclass.DocumentableBuilder;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.bson.conversions.Bson;

import java.util.ArrayList;
import java.util.List;

public class AbstractMongoDao<T extends Idable<String> & IDocumentAble & Jsonable, Builder extends JsonableBuilder<T> & DocumentableBuilder<T>> implements IMongoDao<T>  {

    protected MongoCollection<Document> collection;

    protected MongoDatabase database;

    protected String idFieldName;

    protected Builder beanBuilder;


    public AbstractMongoDao(MongoDatabase mongoDatabase, String collectionName, String idFieldName, Builder builder  ) {
        this.database = mongoDatabase;
        if (StringUtils.isNotEmpty(collectionName)){
            collection = mongoDatabase.getCollection(collectionName);
        }
        this.idFieldName = idFieldName;
        this.beanBuilder = builder;
    }

    @Override
    public void updateOne(String id, T bean) {
        Document doc = bean.toDocument();
        this.updateOne(id, doc);
    }

    @Override
    public void insertOne(T bean) {
        Document doc = bean.toDocument();
        collection.withWriteConcern(WriteConcern.ACKNOWLEDGED).insertOne(doc);

    }

    public void deleteone(String id) {
        // TODO Auto-generated method stub
        collection.deleteOne(new Document(this.idFieldName, id));
    }

    public T findone(String id) {
        FindIterable<Document> it = this.findCollection(new Document(this.idFieldName, id));
        Document doc = it.first();
        if (doc == null){
            return null;
        }
        T bean = null;
        try {
            bean = beanBuilder.fromDocument(doc);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return bean;
    }

    public T findone(Bson filter) {
        FindIterable<Document> it = this.findCollection(filter);
        Document doc = it.first();
        T bean = null;
        if (doc == null){
            return null;
        }
        try {
            bean = beanBuilder.fromDocument(doc);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return bean;
    }

    public List<T> find(Bson filter) {
        FindIterable<Document> it = this.findCollection(filter);
        MongoCursor<Document> cursor = it.iterator();
        List<T> list = new ArrayList<>();
        while (cursor.hasNext()){
            Document doc = cursor.next();
            T bean = null;
            try {
                bean = beanBuilder.fromDocument(doc);
            } catch (Exception e) {
                e.printStackTrace();
            }
            list.add(bean);
        }
        return list;
    }

    public Long count(Bson filter) {
        return collection.countDocuments(filter);
    }

    public Boolean exists(String id){
        return this.findone(new Document(idFieldName, id)) != null;
    }


    public void createIndex(String[] indexArray) {
        collection.createIndex(Indexes.ascending(indexArray));
    }









    public MongoCursor<Document> find(Bson filter, Bson projection) {
        FindIterable<Document> it = this.findCollection(filter).projection(projection);
        MongoCursor<Document> cursor = it.iterator();
        return cursor;
    }



    public Document findone(Bson filter, Bson projection) {
        FindIterable<Document> it = this.findCollection(filter).projection(projection);
        return it.first();
    }



    public void insert(Document doc) {
        collection.withWriteConcern(WriteConcern.ACKNOWLEDGED).insertOne(doc);
    }


    public void insert(List<Document> list) {
        collection.withWriteConcern(WriteConcern.ACKNOWLEDGED).insertMany(list);
    }


    public void update(Bson filter, Document doc) {
        collection.withWriteConcern(WriteConcern.ACKNOWLEDGED).updateOne(filter, doc);
    }


    public void updateOne(Bson filter, Document doc) {
        collection.withWriteConcern(WriteConcern.ACKNOWLEDGED).replaceOne(filter, doc);
    }

    public void updateOne(String id, Document doc) {
        collection.withWriteConcern(WriteConcern.ACKNOWLEDGED).replaceOne(new Document(this.idFieldName, id), doc);
    }

    public void insertSaft(Document doc) {
        collection.withWriteConcern(WriteConcern.JOURNALED).insertOne(doc);
    }

    public FindIterable<Document> findCollection(Bson filter) {
        FindIterable<Document> it = collection.find(filter);
        return it;
    }

    public FindIterable<Document> findCollection(Bson filter, Bson projection) {
        return this.findCollection(filter).projection(projection);
    }

    public void deleteone(Bson filter) {
        // TODO Auto-generated method stub
        collection.deleteOne(filter);
    }
}
